% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions1.R
\name{DetectClustersModel}
\alias{DetectClustersModel}
\title{Detects clusters and computes their significance.}
\usage{
DetectClustersModel(stfdf, thegrid = NULL, radius = Inf, step = NULL,
  fractpop, alpha, typeCluster = "S", minDateUser = NULL,
  maxDateUser = NULL, R = NULL, model0,
  ClusterSizeContribution = "Population")
}
\arguments{
\item{stfdf}{object containing the data.
If data is spatial, stfdf is a \link{SpatialPolygonsDataFrame} object from sp.
If data is spatio-temporal, stfdf is a \link{STFDF} object from spacetime.
The data contain a \link{SpatialPolygons} object with the coordinates,
and if applicable, a time object holding time information,
an endTime vector of class \link{POSIXct} holding end points of time intervals.
It also contain a data.frame with the Observed, Expected and potential covariates
in each location and time (if applicable). Note that the function DetectClustersModel
does not use the endTime vector. We can define endTime, for example,
as the vector of class \link{POSIXct} which contains the same dates as the ones
contained in the time object.}

\item{thegrid}{two-columns matrix containing the points of the grid to be
used. If it is null, a rectangular grid is built.}

\item{radius}{maximum radius of the clusters.}

\item{step}{step of the thegrid built.}

\item{fractpop}{maximum fraction of the total population inside the cluster.}

\item{alpha}{significance level used to determine the existence of clusters.}

\item{typeCluster}{type of clusters to be detected. "ST" for spatio-temporal
or "S" spatial clusters.}

\item{minDateUser}{start date of the clusters.}

\item{maxDateUser}{end date of the clusters.}

\item{R}{If the cluster's significance is calculated based on the chi-square
distribution or DIC, R is NULL. If the cluster's significance is calculated using a
Monte Carlo procedure, R represents the number replicates under the null hypothesis.}

\item{model0}{Initial model (including covariates).}

\item{ClusterSizeContribution}{Indicates the variable to be used as the population at risk in the cluster. This is the variable name to be used by 'fractpop' when checking the fraction of the population inside the cluster. The default column name is 'Population'.
This can be "glm" for generalized linear models (\link{glm} {stats}),
"glmer" for generalized linear mixed model (\link{glmer} {lme4}),
"zeroinfl" for zero-inflated models (\link{zeroinfl} {pscl}), or
"inla" for generalized linear, generalized linear mixed or zero-inflated models fitted with \link[INLA]{inla}.}
}
\value{
data frame with information of the detected clusters ordered by its
log-likelihood ratio value or DIC. Each row represents the information of
one of the clusters. It contains the coordinates of the center, the size,
the start and end dates, the log-likelihood ratio or DIC, the p-value,
the risk of the cluster, and a boolean indicating if it is a
cluster (TRUE in all cases).
It also returns alpha_bonferroni which is the level of significance
adjusted for multiple testing using Bonferroni correction.
Thus, rows that should be considered clusters are the ones with
p-value less than alpha_bonferroni.
}
\description{
Searches all possible clusters with start and end dates within minDateUser
and maxDateUser, so that the maximum fraction of the total population inside
the cluster is less than fractpop, and the maximum distance to the center is
less than radius.
The search can be done for spatial or spatio-temporal clusters.
The significance of the clusters is obtained with a Monte Carlo procedure
or based on the chi-square distribution (\link{glm}, \link{glmer} or \link{zeroinfl} models)
or DIC (\link[INLA]{inla} models).
}
\examples{
library("DClusterm")
data("NY8")

NY8$Observed <- round(NY8$Cases)
NY8$Expected  <- NY8$POP8 * sum(NY8$Observed) / sum(NY8$POP8)

NY8$x <- coordinates(NY8)[, 1]
NY8$y <- coordinates(NY8)[, 2]


#Model to account for covariates
ny.m1 <- glm(Observed ~ offset(log(Expected)) + PCTOWNHOME + PCTAGE65P +
PEXPOSURE, family = "poisson", data = NY8)

#Indices of areas that are possible cluster centres
idxcl <- c(120, 12, 89, 139, 146)

#Cluster detection adjusting for covariates
ny.cl1 <- DetectClustersModel(NY8,
thegrid = as.data.frame(NY8)[idxcl, c("x", "y")],
fractpop = 0.15, alpha = 0.05,
typeCluster = "S", R = NULL, model0 = ny.m1,
ClusterSizeContribution = "POP8")

#Display results
ny.cl1

}
\references{
Bilancia M, Demarinis G (2014) Bayesian scanning of spatial disease rates
with the Integrated Nested Laplace Approximation (INLA). Statistical
Methods & Applications 23(1): 71 - 94. \url{http://dx.doi.org/10.1007/s10260-013-0241-8}

Jung I (2009) A generalized linear models approach to spatial scan 
statistics for covariate adjustment. Statistics in Medicine 28(7): 1131 - 1143.

Fast Bayesian classification for disease mapping and the detection of 
disease clusters (2017) Gomez-Rubio V, Molitor J, Moraga P. Submitted.
}
