\name{CleanCoordinatesDS}
\alias{CleanCoordinatesDS}

\title{
Geographic Coordinate Cleaning based on Dataset Properties
}
\description{
Identifies potentially problematic coordinates based on dataset properties. Includes test to flag potential errors with converting ddmm to dd.dd, and periodicity in the data decimals indicating rounding or a raster basis linked to low coordinate precision.
}
\usage{
CleanCoordinatesDS(x, lon = "decimallongitude", lat = "decimallatitude",
                   ds = "dataset",
                   ddmm = TRUE, periodicity = TRUE,
                   ddmm.pvalue = 0.025, ddmm.diff = 0.2, 
                   periodicity.T1 = 7,
                   periodicity.reg.thresh = 2,
                   periodicity.dist.min = 0.1,
                   periodicity.dist.max = 2,
                   periodicity.min.size = 4, 
                   periodicity.target = "both",
                   periodicity.diagnostics = TRUE,
                   value = "dataset", verbose = TRUE)
}

\arguments{
  \item{x}{
a data.frame. Containing geographical coordinates and species names.
}
  \item{lon}{
a character string. The column with the longitude coordinates. Default = \dQuote{decimallongitude}.
}
  \item{lat}{
a character string. The column with the longitude coordinates. Default = \dQuote{decimallatitude}.
}
  \item{ds}{
a character string. The column with the dataset of each record. In case \code{x} should be treated as a single dataset, identical for all records. Default = \dQuote{dataset}.
}
  \item{ddmm}{
logical. If TRUE, testing for erroneous conversion from a degree minute format (ddmm) to a decimal degree (dd.dd) format. See details.
}
  \item{periodicity}{
logical. If TRUE, testing for periodicity in the data, which can indicate imprecise coordinates, due to rounding or rasterization.
}
  \item{ddmm.pvalue}{
numeric. The p-value for the one-sided t-test to flag the ddmm test as passed or not. Both ddmm.pvalue and ddmm.diff must be met. Default = 0.025.
}
  \item{ddmm.diff}{
numeric. The threshold difference for the ddmm test. Indicates by which fraction the records with decimals below 0.6 must outnumber the records with decimals above 0.025. Default = 0.2
}
  \item{periodicity.T1}{
numeric.  The threshold for outlier detection in a in an interquantile range based test. This is the major parameter to specify the sensitivity of the test: lower values, equal higher detection rate. Values between 7-11 are recommended. Default = 7.
}
  \item{periodicity.reg.thresh}{
numeric. Threshold on the number of equal distances between outlier points.  See details.  Default = 2.
}
  \item{periodicity.dist.min}{
numeric.  The minimum detection distance between outliers in degrees (the minimum resolution of grids that will be flagged).  Default = 0.1.
}
  \item{periodicity.dist.max}{
numeric.  The maximum detection distance between outliers in degrees (the maximum resolution of grids that will be flagged).  Default = 2.
}
  \item{periodicity.min.size}{
numeric.  The minimum number of unique locations (values in the tested column) for datasets to be included in the test.  Default = 4.
}
  \item{periodicity.target}{
character string.  Indicates which column to test. Either \dQuote{lat] for latitude, \dQuote{lon} for longitude, or \dQuote{both} for both.  In the latter case datasets are only flagged if both test are failed.  Default = \dQuote{both}}
}
  \item{periodicity.diagnostics}{
logical. If TRUE, diagnostic plots are produced.  Default = TRUE.
}
  \item{value}{
a character string.  Defining the output value. See value. Default = \dQuote{dataset}.
}
  \item{verbose}{
logical. If TRUE reports the name of the test and the number of records flagged.
}
}
\details{
This function checks the statistical distribution of decimals within datasets of geographic distribution records to identify datasets with potential errors/biases. Three potential error sources can be identified. The ddmm flag tests for the particular pattern that emerges if geographical coordinates in a degree minute annotation are transferred into decimal degrees, simply replacing the degree symbol with the decimal point. This kind of problem has been observed by in older datasets first recorded on paper using typewriters, where e.g. a floating point was used as symbol for degrees. The function uses a binomial test to check if more records then expected have decimals blow 0.6 (which is the maximum that can be obtained in minutes, as one degree has 60 minutes) and if the number of these records is higher than those above 0.59 by a certain proportion. The periodicity test uses rate estimation in a poison process to estimate if there is periodicity in the decimals of a dataset (as would be expected by for example rounding or data that was collected in a raster format) and if there is an over proportional number of records with the decimal 0 (full degrees) which indicates rounding and thus low precision. The default values are empirically optimized by with GBIF data, but should probably be adapted.
}
\value{
Depending on the \sQuote{value} argument, either a summary per dataset \code{dataset}, a data.frame containing the records considered correct by the test (\dQuote{clean}) or a logical vector, with TRUE = test passed and FALSE = test failed/potentially problematic (\dQuote{flags}). Default = \dQuote{clean}. If \dQuote{dataset}: \code{data.frame} with one row for each dataset in \code{x}.
}

\seealso{
\code{\link{CleanCoordinates}}
}
\examples{
#Create test dataset
clean <- data.frame(dataset = rep("clean", 1000),
                    decimallongitude = runif(min = -42, max = -40, n = 1000),
                    decimallatitude = runif(min = -12, max = -10, n = 1000))
                    
bias.long <- c(round(runif(min = -42, max = -40, n = 500), 1),
               round(runif(min = -42, max = -40, n = 300), 0),
               runif(min = -42, max = -40, n = 200))
bias.lat <- c(round(runif(min = -12, max = -10, n = 500), 1),
              round(runif(min = -12, max = -10, n = 300), 0),
              runif(min = -12, max = -10, n = 200))
bias <- data.frame(dataset = rep("biased", 1000),
                   decimallongitude = bias.long,
                   decimallatitude = bias.lat)
test <- rbind(clean, bias)

\dontrun{                  
#run CleanCoordinatesDS
flags <- CleanCoordinatesDS(test)

#check problems
#clean
hist(test[test$dataset == rownames(flags[flags$summary,]), "decimallongitude"])
#biased
hist(test[test$dataset == rownames(flags[!flags$summary,]), "decimallongitude"])

}
}
                  
\keyword{ Coordinate cleaning wrapper}

