% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/getParents.R
\name{getParents}
\alias{getParents}
\title{Estimate the connectivity matrix of a causal graph}
\usage{
getParents(X, environment = NULL, interventions = NULL,
  parentsOf = 1:ncol(X), method = c("ICP", "hiddenICP", "backShift", "pc",
  "LINGAM", "ges", "gies", "CAM", "rfci", "regression", "bivariateANM",
  "bivariateCAM")[1], alpha = 0.1, variableSelMat = NULL,
  excludeTargetInterventions = TRUE, onlyObservationalData = FALSE,
  indexObservationalData = 1, returnAsList = FALSE, pointConf = FALSE,
  setOptions = list(), directed = TRUE, verbose = FALSE)
}
\arguments{
\item{X}{A (nxp)-data matrix with n observations of p variables.}

\item{environment}{An optional vector of length n, where the entry for
observation i is an index for the environment in which observation i took
place (Simplest case: entries \code{1} for observational data and entries
 \code{2} for interventional data of unspecified type. Encoding for observational
 data can we changed with \code{indexObservationalData}). Is required for
 methods \code{ICP}, \code{hiddenICP} and \code{backShift}.}

\item{interventions}{A optional list of length n. The entry for observation
i is a numeric vector that specifies the variables on which interventions
happened for observation i (a scalar if an intervention happened on just
one variable and \code{numeric(0)} if no intervention occured for this
observation). Is used for methods \code{gies} and \code{CAM} and will
generate the vector \code{environment} if the latter is set to \code{NULL}.
(However, this might generate too many different environments for some data
sets, so a hand-picked vector \code{environment} is preferable). Is also used
for \code{ICP} and \code{hiddenICP} to exclude interventions on the target
variable of interest.}

\item{parentsOf}{The variables for which we would like to estimate the
parents. Default are all variables.}

\item{method}{A string that specfies the method to use. The methods
\code{pc} (PC-algorithm), \code{LINGAM} (LINGAM), \code{ges}
(Greedy equivalence search), \code{gies} (Greedy interventional equivalence
search) and \code{rfci} (Really fast causal inference) are imported from the
package "pcalg" and are documented there in more detail, including the
additional options that can be supplied via \code{setOptions}. The method
\code{CAM} (Causal additive models) is documented in the package "CAM" and
the methods \code{ICP} (Invariant causal prediction), \code{hiddenICP}
(Invariant causal prediction with hidden variables) are from the package
"InvariantCausalPrediction". The method \code{backShift} comes from the
package "backShift". Finally, the methods \code{bivariateANM} and
\code{bivariateCAM} are for now implemented internally but will hopefully
be part of another package at some point in the near future.}

\item{alpha}{The level at which tests are done. This leads to confidence
intervals for \code{ICP} and \code{hiddenICP} and is used internally for
\code{pc} and \code{rfci}.}

\item{variableSelMat}{An optional logical matrix of dimension (pxp). An
entry \code{TRUE} for entry (i,j) says that variable i should be considered
as a potential parent for variable j and vice versa for \code{FALSE}. If the
default value of \code{NULL} is used, all variables will be considered, but
this can be very slow, especially for methods \code{pc}, \code{ges},
\code{gies}, \code{rfci} and \code{CAM}.}

\item{excludeTargetInterventions}{When looking for parents of variable k
in 1,...,p, set to \code{TRUE} if observations where an intervention on
variable k occured should be excluded. Default is \code{TRUE}.}

\item{onlyObservationalData}{If set to \code{TRUE}, only observational data
is used. It will take the index in \code{environment} specified by
\code{indexObservationalData}. If \code{environment} is \code{NULL}, all
observations are used. Default is \code{FALSE}.}

\item{indexObservationalData}{Index in \code{environment} that encodes
observational data. Default is \code{1}.}

\item{returnAsList}{If set to \code{TRUE}, will return a list, where entry
k is a list containing the estimated parents of variable k. The option
\code{directed} will be ignored if set to \code{TRUE}. Default is
\code{FALSE}.}

\item{pointConf}{If \code{TRUE}, numerical estimates will be returned if
possible. For methods \code{ICP} and \code{hiddenICP}, these are the values
in the individual confidence intervals (at chosen level \code{alpha}) that
are closest to 0; for other methods these are point estimates. Some methods
do not return numerical point estimates; for these the output will remain
binary 0/1 (no-edge/edge). Default is \code{FALSE}.}

\item{setOptions}{A list that can take method-specific options; see the
individual documentations of the methods for more options and their
possible values.}

\item{directed}{If \code{TRUE}, an edge will be returned if and only if an
edge has been detected to be directed (ie entry will be set to 0 for entry
(j,k) if both j->k and k-> j are estimated). Ignored if not the whole graph
is estimated or if \code{returnAsList} is \code{TRUE}.}

\item{verbose}{If \code{TRUE}, detailed output is provided.}
}
\value{
If option \code{returnAsList} is \code{FALSE}, a sparse matrix,
where a 0 entry in position (j,k) corresponds to an estimate of "no edge"
\code{j} -> \code{k}, while an entry 1 corresponds to an
estimated egde. If option \code{pointConf} is \code{TRUE}, the 1 entries
will be replaced by numerical values that are either point estimates of the
causal coefficients or confidence bounds (see above).
If option \code{returnAsList} is \code{TRUE}, a list will be returned.
The k-th entry in the list is the numeric vector with the indices of the
estimated parents of node \code{k}.
}
\description{
Estimates the connectivity matrix of a directed causal graph,
using various possible methods. Supported methods at the moment are
backShift, bivariateANM, bivariateCAM, CAM, hiddenICP, ICP, GES, GIES, LINGAM,
PC, regression and RFCI.
}
\examples{
## load the backShift package for data generation and plotting functionality
if(!requireNamespace("backShift", quietly = TRUE))
 stop("The package 'backShift' is needed for the examples to
 work. Please install it.", call. = FALSE)

require(backShift)

# Simulate data with connectivity matrix A with assumptions
# 1) hidden variables present
# 2) precise location of interventions is assumed unknown
# 3) different environments can be distinguished

## simulate data
myseed <- 1

# sample size n
n <- 10000

# p=3 predictor variables and connectivity matrix A
p <- 3
labels <- c("1", "2", "3")
A <- diag(p)*0
A[1,2] <- 0.8
A[2,3] <- 0.8
A[3,1] <- -0.4

# divide data in 10 different environments
G <- 10

# simulate
simResult <- simulateInterventions(n, p, A, G, intervMultiplier = 3,
             noiseMult = 1, nonGauss = TRUE, hiddenVars = TRUE,
             knownInterventions = FALSE, fracVarInt = NULL, simulateObs = TRUE,
             seed = myseed)
X <- simResult$X
environment <- simResult$environment

## apply all  methods given in vector 'methods'
## (using all data pooled for pc/LINGAM/rfci/ges -- can be changed with option
## 'onlyObservationalData=TRUE')

methods <- c("backShift", "LINGAM") #c("pc", "rfci", "ges")

# select whether you want to run stability selection
stability <- FALSE

# arrange graphical output into a rectangular grid
sq <- ceiling(sqrt(length(methods)+1))
par(mfrow=c(ceiling((length(methods)+1)/sq),sq))

## plot and print true graph
cat("\\n true graph is  ------  \\n" )
print(A)
plotGraphEdgeAttr(A, plotStabSelec = FALSE, labels = labels, thres.point = 0,
 main = "TRUE GRAPH")

## loop over all methods and compute and print/plot estimate
for (method in methods){
  cat("\\n result for method", method,"  ------  \\n" )

  if(!stability){
    # Option 1): use this estimator as a point estimate
    Ahat <- getParents(X, environment, method=method, alpha=0.1, pointConf = TRUE)
  }else{
    # Option 2): use a stability selection based estimator
    # with expected number of false positives bounded by EV=2
    Ahat <- getParentsStable(X, environment, EV=2, method=method, alpha=0.1)
  }

  # print and plot estimate (point estimate thresholded if numerical estimates
  # are returned)
  print(Ahat)
  if(!stability)
    plotGraphEdgeAttr(Ahat, plotStabSelec = FALSE, labels = labels,
     thres.point = 0.05,
     main=paste("POINT ESTIMATE FOR METHOD\\n", toupper(method)))
  else
    plotGraphEdgeAttr(Ahat, plotStabSelec = TRUE, labels = labels,
     thres.point = 0, main = paste("STABILITY SELECTION
     ESTIMATE\\n FOR METHOD", toupper(method)))
 }
}
\author{
Christina Heinze \email{heinze@stat.math.ethz.ch},
 Nicolai Meinshausen \email{meinshausen@stat.math.ethz.ch}
}
\seealso{
\code{\link{getParentsStable}} for stability selection-based
estimation of the causal graph.
}
\keyword{Causality,}
\keyword{Graph}
\keyword{estimations}

