# Copyright 2022 DARWIN EU (C)
#
# This file is part of CohortCharacteristics
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

#' Create a visual table from the output of summariseCohortAttrition.
#' `r lifecycle::badge("experimental")`
#'
#' @param result A summarised_result object generated by summariseCohortAttrition().
#' @param header Columns to use as headers.
#' @param groupColumn Columns to use to group.
#' @param type Whether a 'gt', 'flextable' or a 'tibble' should be created.
#'
#' @return A visual table.
#'
#' @export
#'
#' @examples
#' \donttest{
#' library(CohortCharacteristics)
#'
#' cdm <- mockCohortCharacteristics()
#'
#' cdm$cohort2 |>
#'   summariseCohortAttrition() |>
#'   tableCohortAttrition()
#' }
#'
tableCohortAttrition <- function(result,
                                 header = "cdm_name",
                                 groupColumn = "cohort_name",
                                 type = "gt") {
  # initial checks
  assertClass(result, "summarised_result")
  assertChoice(type, c("gt", "flextable", "tibble"), length = 1)
  assertChoice(header, c("cdm_name", "cohort_name"))
  assertChoice(groupColumn, c("cdm_name", "cohort_name"))

  if(nrow(result) == 0){
   cli::cli_warn("Empty result object")
   return(emptyResultTable(type = type))
  }

  header <- correct(header)
  groupColumn <- correct(groupColumn)

  # showMinCellCount
  settings <- omopgenerics::settings(result) |>
    dplyr::filter(.data$result_type == "cohort_attrition")

  if(nrow(settings) == 0){
    cli::cli_warn("No cohort_attrition result found")
    return(emptyResultTable(type = type))
  }


  if ("min_cell_count" %in% colnames(settings)) {
    result <- result |>
      dplyr::left_join(
        settings |>
          dplyr::select("result_id", "min_cell_count"),
        by = "result_id"
      ) |>
      dplyr::mutate(estimate_value = dplyr::if_else(
        is.na(.data$estimate_value), paste0("<", .data$min_cell_count), .data$estimate_value
      )) |>
      dplyr::select(!"min_cell_count")
  } else {
    cli::cli_inform(c("!" = "Results have not been suppressed."))
  }

  # create table
  result <- result |>
    visOmopResults::filterSettings(.data$result_type == "cohort_attrition") |>
    visOmopResults::splitAll() |>
    dplyr::arrange(.data$cohort_name, .data$reason_id) |>
    dplyr::select(-c(
      "result_id", "variable_level", "estimate_name", "estimate_type", "reason_id"
    )) |>
    dplyr::mutate("variable_name" = stringr::str_to_sentence(gsub("_", " ", .data$variable_name))) |>
    dplyr::rename(
      "Reason" = "reason",
      "Variable" = "variable_name",
      "CDM name" = "cdm_name",
      "Cohort name" = "cohort_name"
    ) |>
    visOmopResults::formatHeader(header = c(header, "Variable"))

  if (type == "gt") {
    result <- result |>
      visOmopResults::gtTable(groupColumn = groupColumn)
  } else if (type == "flextable") {
    result <- result |>
      visOmopResults::fxTable(groupColumn = groupColumn)
  }

  return(result)
}

correct <- function(x) {
  x[x == "cdm_name"] <- "CDM name"
  x[x == "cohort_name"] <- "Cohort name"
  return(x)
}


emptyResultTable <- function(type){
  if(type == "gt"){
   result <- gt::gt(dplyr::tibble())
  } else if(type == "flextable"){
    result <- flextable::flextable(dplyr::tibble("Table has no data" = "Empty result provided"))
  } else{
    result <- dplyr::tibble()
  }

  result
}
