% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cste_bin.R
\name{cste_bin}
\alias{cste_bin}
\title{Estimate the CSTE curve for binary outcome.}
\usage{
cste_bin(
  x,
  y,
  z,
  beta_ini = NULL,
  lam = 0,
  nknots = 1,
  max.iter = 200,
  eps = 0.001
)
}
\arguments{
\item{x}{samples of covariates which is a \eqn{n*p} matrix.}

\item{y}{samples of binary outcome which is a \eqn{n*1} vector.}

\item{z}{samples of treatment indicator which is a \eqn{n*1} vector.}

\item{beta_ini}{initial values for \eqn{(\beta_1', \beta_2')'}, default value is NULL.}

\item{lam}{value of the lasso penalty parameter \eqn{\lambda} for \eqn{\beta_1} and
\eqn{\beta_2}, default value is 0.}

\item{nknots}{number of knots for the B-spline for estimating \eqn{g_1} and \eqn{g_2}.}

\item{max.iter}{maximum iteration for the algorithm.}

\item{eps}{numeric scalar \eqn{\geq} 0, the tolerance for the estimation 
of \eqn{\beta_1} and \eqn{\beta_2}.}
}
\value{
A S3 class of cste, which includes: 
\itemize{
   \item \code{beta1}: estimate of \eqn{\beta_1}.
   \item \code{beta2}: estimate of \eqn{\beta_2}.
   \item \code{B1}: the B-spline basis for estimating \eqn{g_1}.
   \item \code{B2}: the B-spline basis for estimating \eqn{g_2}.
   \item \code{delta1}: the coefficient of B-spline for estimating \eqn{g_1}.
   \item \code{delta2}: the coefficient for B-spline for estimating \eqn{g_2}.
   \item \code{iter}: number of iteration.
   \item \code{g1}: the estimate of \eqn{g_1(X\beta_1)}.
   \item \code{g2}: the estimate of \eqn{g_2(X\beta_2)}. 
}
}
\description{
Estimate covariate-specific treatment effect (CSTE) curve. Input data
contains covariates \eqn{X}, treatment assignment \eqn{Z} and binary outcome
\eqn{Y}. The working model is \deqn{logit(\mu(X, Z)) = g_1(X\beta_1)Z + g_2(X\beta_2),} 
where \eqn{\mu(X, Z) = E(Y|X, Z)}. The model implies that \eqn{CSTE(x) = g_1(x\beta_1)}.
}
\examples{
## Quick example for the cste

library(mvtnorm)
library(sigmoid)

# --------  Example 1: p = 20 ---------  #  
## generate data 
n <- 2000
p <- 20
set.seed(100)

# generate X
sigma <- outer(1:p, 1:p, function(i, j){ 2^(-abs(i-j)) } )
X <- rmvnorm(n, mean = rep(0,p), sigma = sigma)
X <- relu(X + 2) - 2
X <- 2 - relu(2 - X)

# generate Z
Z <- rbinom(n, 1, 0.5)

# generate Y
beta1 <- rep(0, p)
beta1[1:3] <- rep(1/sqrt(3), 3)
beta2 <- rep(0, p)
beta2[1:2] <- c(1, -2)/sqrt(5)
mu1 <- X \%*\% beta1
mu2 <- X \%*\% beta2
g1 <- mu1*(1 - mu1)
g2 <- exp(mu2)      
prob <- sigmoid(g1*Z + g2)
Y <- rbinom(n, 1, prob)

## estimate the CSTE curve
fit <- cste_bin(X, Y, Z)

## plot 
plot(mu1, g1, cex = 0.5, xlim = c(-2,2), ylim = c(-8, 3), 
     xlab = expression(X*beta), ylab = expression(g1(X*beta)))
     ord <- order(mu1)
     points(mu1[ord], fit$g1[ord], col = 'blue', cex = 0.5)
     
## compute 95\% simultaneous confidence band (SCB)
res <- cste_bin_SCB(X, fit, alpha = 0.05)

## plot 
plot(res$or_x, res$fit_x, col = 'red', 
     type="l", lwd=2, lty = 3, ylim = c(-10,8),
     ylab=expression(g1(X*beta)), xlab = expression(X*beta), 
     main="Confidence Band")
lines(res$or_x, res$lower_bound, lwd=2.5, col = 'purple', lty=2)
lines(res$or_x, res$upper_bound, lwd=2.5, col = 'purple', lty=2)
abline(h=0, cex = 0.2, lty = 2)
legend("topleft", legend=c("Estimates", "SCB"), 
        lwd=c(2, 2.5), lty=c(3,2), col=c('red', 'purple'))
        
        
# --------  Example 2: p = 1 ---------  #  

## generate data 
set.seed(15)
p <- 1
n <- 2000
X <- runif(n)
Z <- rbinom(n, 1, 0.5)
g1 <- 2 * sin(5*X) 
g2 <- exp(X-3) * 2
prob <- sigmoid( Z*g1 + g2)
Y <- rbinom(n, 1, prob)

## estimate the CSTE curve
fit <- cste_bin(X, Y, Z)  

## simultaneous confidence band (SCB)
X <- as.matrix(X)
res <- cste_bin_SCB(X, fit)  

## plot 
plot(res$or_x, res$fit_x, col = 'red', type="l", lwd=2, 
     lty = 3, xlim = c(0, 1), ylim = c(-4, 4), 
     ylab=expression(g1(X)), xlab = expression(X), 
     main="Confidence Band")
lines(res$or_x, res$lower_bound, lwd=2.5, col = 'purple', lty=2)
lines(res$or_x, res$upper_bound, lwd=2.5, col = 'purple', lty=2)
abline(h=0, cex = 0.2)
lines(X[order(X)], g1[order(X)], col = 'blue', lwd = 1.5)
legend("topright", legend=c("Estimates", "SCB",'True CSTE Curve'), 
lwd=c(2, 2.5, 1.5), lty=c(3,2,1), col=c('red', 'purple','blue'))

}
\references{
Guo W., Zhou X. and Ma S. (2021).
Estimation of Optimal Individualized Treatment Rules
Using a Covariate-Specific Treatment Effect Curve with 
High-dimensional Covariates,
\emph{Journal of the American Statistical Association}, 116(533), 309-321
}
\seealso{
\code{\link{cste_bin_SCB}, \link{predict_cste_bin}, \link{select_cste_bin}}
}
