% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bess.one.R
\name{bess.one}
\alias{bess.one}
\title{Best subset selection/Best subset ridge regression with a
specified model size and a shrinkage parameter}
\usage{
bess.one(
  x,
  y,
  family = c("gaussian", "binomial", "poisson", "cox"),
  type = c("bss", "bsrr"),
  s,
  lambda = 0,
  always.include = NULL,
  screening.num = NULL,
  normalize = NULL,
  weight = NULL,
  max.iter = 20,
  group.index = NULL
)
}
\arguments{
\item{x}{Input matrix, of dimension \eqn{n \times p}; each row is an observation
vector and each column is a predictor/feature/variable.}

\item{y}{The response variable, of \code{n} observations. For \code{family = "binomial"} should be
a factor with two levels. For \code{family="poisson"}, \code{y} should be a vector with positive integer.
 For \code{family = "cox"}, \code{y} should be a two-column matrix
with columns named \code{time} and \code{status}.}

\item{family}{One of the following models: \code{"gaussian"}, \code{"binomial"},
\code{"poisson"}, or \code{"cox"}. Depending on the response.}

\item{type}{One of the two types of problems.
\code{type = "bss"} for the best subset selection,
and \code{type = "bsrr"} for the best subset ridge regression.}

\item{s}{A specified model size}

\item{lambda}{A shrinkage parameter for \code{"bsrr"}.}

\item{always.include}{An integer vector containing the indexes of variables that should always be included in the model.}

\item{screening.num}{Users can pre-exclude some irrelevant variables according to maximum marginal likelihood estimators before fitting a
model by passing an integer to \code{screening.num} and the sure independence screening will choose a set of variables of this size.
Then the active set updates are restricted on this subset.}

\item{normalize}{Options for normalization. \code{normalize = 0} for
no normalization. Setting \code{normalize = 1} will
only subtract the mean of columns of \code{x}.
\code{normalize = 2} for scaling the columns of \code{x} to have \eqn{\sqrt n} norm.
\code{normalize = 3} for subtracting the means of the columns of \code{x} and \code{y}, and also
normalizing the columns of \code{x} to have \eqn{\sqrt n} norm.
If \code{normalize = NULL}, by default, \code{normalize} will be set \code{1} for \code{"gaussian"},
\code{2} for \code{"binomial"} and \code{"poisson"}, \code{3} for \code{"cox"}.}

\item{weight}{Observation weights. Default is \code{1} for each observation.}

\item{max.iter}{The maximum number of iterations in the bess function.
In most of the case, only a few steps can guarantee the convergence. Default
is \code{20}.}

\item{group.index}{A vector of integers indicating the which group each variable is in.
For variables in the same group, they should be located in adjacent columns of \code{x}
and their corresponding index in \code{group.index} should be the same.
Denote the first group as \code{1}, the second \code{2}, etc.
If you do not fit a model with a group structure,
please set \code{group.index = NULL}. Default is \code{NULL}.}
}
\value{
A list with class attribute 'bess' and named components:
\item{beta}{The best fitting coefficients.}
 \item{coef0}{The best fitting
intercept.}
\item{bestmodel}{The best fitted model for \code{type = "bss"}, the class of which is \code{"lm"}, \code{"glm"} or \code{"coxph"}.}
\item{loss}{The training loss of the fitting model.}
\item{s}{The model size.}
\item{lambda}{The shrinkage parameter.}
\item{family}{Type of the model.}
\item{nsample}{The sample size.}
\item{type}{Either \code{"bss"} or \code{"bsrr"}.}
}
\description{
Best subset selection with a specified model size for generalized
linear models and Cox's proportional hazard model.
}
\details{
Given a model size \eqn{s}, we consider the following best subset selection problem:
\deqn{\min_\beta -2 \log L(\beta) ;{ s.t.} \|\beta\|_0 = s.}
And given a model size \eqn{s} and a shrinkage parameter \eqn{\lambda}
, consider the following best subset ridge regression problem:
\deqn{\min_\beta -2 \log L(\beta) + \lambda \Vert\beta \Vert_2^2; { s.t.} \|\beta\|_0 = s.}

In the GLM case, \eqn{\log L(\beta)} is the log likelihood function;
In the Cox model, \eqn{\log L(\beta)} is the log partial likelihood function.

The best subset selection problem is solved by the primal dual active set algorithm,
see Wen et al. (2017) for details. This algorithm utilizes an active set updating strategy
via primal and dual variables and fits the sub-model by exploiting the fact that their
support set are non-overlap and complementary.
}
\examples{

#-------------------linear model----------------------#
# Generate simulated data
n <- 200
p <- 20
k <- 5
rho <- 0.4
seed <- 10
Tbeta <- rep(0, p)
Tbeta[1:k*floor(p/k):floor(p/k)] <- rep(1, k)
Data <- gen.data(n, p, k, rho, family = "gaussian", beta = Tbeta, seed = seed)
x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
lm.bss <- bess.one(x, y, s = 5)
lm.bsrr <- bess.one(x, y, type = "bsrr", s = 5, lambda = 0.01)
coef(lm.bss)
coef(lm.bsrr)
print(lm.bss)
print(lm.bsrr)
summary(lm.bss)
summary(lm.bsrr)
pred.bss <- predict(lm.bss, newx = x_new)
pred.bsrr <- predict(lm.bsrr, newx = x_new)

#-------------------logistic model----------------------#
#Generate simulated data
Data <- gen.data(n, p, k, rho, family = "binomial", beta = Tbeta, seed = seed)

x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
logi.bss <- bess.one(x, y, family = "binomial", s = 5)
logi.bsrr <- bess.one(x, y, type = "bsrr", family = "binomial", s = 5, lambda = 0.01)
coef(logi.bss)
coef(logi.bsrr)
print(logi.bss)
print(logi.bsrr)
summary(logi.bss)
summary(logi.bsrr)
pred.bss <- predict(logi.bss, newx = x_new)
pred.bsrr <- predict(logi.bsrr, newx = x_new)

#-------------------poisson model----------------------#
Data <- gen.data(n, p, k, rho=0.3, family = "poisson", beta = Tbeta, seed = seed)

x <- Data$x[1:140, ]
y <- Data$y[1:140]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200]
poi.bss <- bess.one(x, y, family = "poisson", s=5)
lambda.list <- exp(seq(log(5), log(0.1), length.out = 10))
poi.bsrr <- bess.one(x, y, type = "bsrr", family = "poisson", s = 5, lambda = 0.01)
coef(poi.bss)
coef(poi.bsrr)
print(poi.bss)
print(poi.bsrr)
summary(poi.bss)
summary(poi.bsrr)
pred.bss <- predict(poi.bss, newx = x_new)
pred.bsrr <- predict(poi.bsrr, newx = x_new)

#-------------------coxph model----------------------#
#Generate simulated data
Data <- gen.data(n, p, k, rho, family = "cox", beta = Tbeta, scal = 10)

x <- Data$x[1:140, ]
y <- Data$y[1:140, ]
x_new <- Data$x[141:200, ]
y_new <- Data$y[141:200, ]
cox.bss <- bess.one(x, y, family = "cox", s = 5)
cox.bsrr <- bess.one(x, y, type = "bsrr", family = "cox", s = 5, lambda = 0.01)
coef(cox.bss)
coef(cox.bsrr)
print(cox.bss)
print(cox.bsrr)
summary(cox.bss)
summary(cox.bsrr)
pred.bss <- predict(cox.bss, newx = x_new)
pred.bsrr <- predict(cox.bsrr, newx = x_new)
#----------------------High dimensional linear models--------------------#
\dontrun{
data <- gen.data(n, p = 1000, k, family = "gaussian", seed = seed)

# Best subset selection with SIS screening
fit <- bess.one(data$x, data$y, screening.num = 100, s = 5)
}

#-------------------group selection----------------------#
beta <- rep(c(rep(1,2),rep(0,3)), 4)
Data <- gen.data(200, 20, 5, rho=0.4, beta = beta, seed =10)
x <- Data$x
y <- Data$y

group.index <- c(rep(1, 2), rep(2, 3), rep(3, 2), rep(4, 3),
                rep(5, 2), rep(6, 3), rep(7, 2), rep(8, 3))
lm.group <- bess.one(x, y, s = 5, type = "bss", group.index = group.index)
lm.groupbsrr <- bess.one(x, y, type = "bsrr", s = 5, lambda = 0.01, group.index = group.index)
coef(lm.group)
coef(lm.groupbsrr)
print(lm.group)
print(lm.groupbsrr)
summary(lm.group)
summary(lm.groupbsrr)
pred.group <- predict(lm.group, newx = x_new)
pred.groupl0l2 <- predict(lm.groupbsrr, newx = x_new)
#-------------------include specified variables----------------------#
Data <- gen.data(n, p, k, rho, family = "gaussian", beta = Tbeta, seed = seed)
lm.bss <- bess.one(Data$x, Data$y, s = 5, always.include = 2)

\dontrun{
#-------------------code demonstration in doi: 10.18637/jss.v094.i04----------------------#
Tbeta <- rep(0, 20)
Tbeta[c(1, 2, 5, 9)] <- c(3, 1.5, -2, -1)

data <- gen.data(n = 200, p = 20, family = "gaussian", beta = Tbeta,
rho = 0.2, seed = 123)
fit.one <- bess.one(data$x, data$y, s = 4, family = "gaussian")
print(fit.one)
summary(fit.one)
coef(fit.one, sparse = FALSE)
pred.one <- predict(fit.one, newdata = data$x)
bm.one <- fit.one$bestmodel
summary(bm.one)
}
}
\references{
Wen, C., Zhang, A., Quan, S. and Wang, X. (2020). BeSS: An R
Package for Best Subset Selection in Linear, Logistic and Cox Proportional
Hazards Models, \emph{Journal of Statistical Software}, Vol. 94(4).
doi:10.18637/jss.v094.i04.
}
\seealso{
\code{\link{bess}}, \code{\link{summary.bess}}
\code{\link{coef.bess}}, \code{\link{predict.bess}}.
}
\author{
Canhong Wen, Aijun Zhang, Shijie Quan, Liyuan Hu, Kangkang Jiang, Yanhang Zhang, Jin Zhu and Xueqin Wang.
}
