#'R function for summarizing MCMC output in a regression-style table
#'@title Summarize Bayesian MCMC Output
#'@description R function for summarizing MCMC output in a regression-style table.
#'@param sims Bayesian model object generated by R2jags, rjags, R2WinBUGS, R2OpenBUGS, 
#'MCMCpack, rstan, and rstanarm.
#'@param ci desired level for credible intervals; defaults to c(0.025, 0.975).
#'@param pars character vector of parameters to be printed; defaults to \code{NULL} 
#'(all parameters are printed). If not \code{NULL}, the user can either specify the exact names of 
#'parameters to be printed (e.g. \code{c("alpha", "beta1", "beta2")}) or part of a name 
#'so that all parameters containing that name will be printed (e.g. \code{"beta"} will print \code{beta1}, \code{beta2}, etc.).
#'@param Pr print percent of posterior draws with same sign as median; defaults to \code{FALSE}.
#'@param ROPE defaults to \code{NULL}. If not \code{NULL}, a vector of two values defining the region of 
#'practical equivalence ("ROPE"); returns \% of posterior draws to the left/right of ROPE. For this quantity 
#'to be meaningful, all parameters must be on the same scale (e.g. standardized coefficients 
#'or first differences). See Kruschke (2013, Journal of Experimental 
#'Psychology 143(2): 573-603) for more on the ROPE.
#'@references Kruschke, John K. 2013. “Bayesian Estimation Supersedes the T-Test.” Journal of 
#'Experimental Psychology: General 142 (2): 573–603. https://doi.org/10.1037/a0029146.
#'@return a data frame containing MCMC summary statistics.
#'@examples
#' \donttest{
#' ## simulating data
#' set.seed(123456)
#' b0 <- 0.2 # true value for the intercept
#' b1 <- 0.5 # true value for first beta
#' b2 <- 0.7 # true value for second beta
#' n <- 500 # sample size
#' X1 <- runif(n, -1, 1)
#' X2 <- runif(n, -1, 1)
#' Z <- b0 + b1 * X1 + b2 * X2
#' pr <- 1 / (1 + exp(-Z)) # inv logit function
#' Y <- rbinom(n, 1, pr)
#' data <- data.frame(cbind(X1, X2, Y))
#' 
#' ## formatting the data for jags
#' datjags <- as.list(data)
#' datjags$N <- length(datjags$Y)
#' 
#' ## creating jags model
#' model <- function()  {
#' 
#'   for(i in 1:N){
#'     Y[i] ~ dbern(p[i])  ## Bernoulli distribution of y_i
#'     logit(p[i]) <- mu[i]    ## Logit link function
#'     mu[i] <- b[1] +
#'       b[2] * X1[i] +
#'       b[3] * X2[i]
#'   }
#' 
#'   for(j in 1:3){
#'     b[j] ~ dnorm(0, 0.001) ## Use a coefficient vector for simplicity
#'   }
#' 
#' }
#' 
#' params <- c("b")
#' inits1 <- list("b" = rep(0, 3))
#' inits2 <- list("b" = rep(0, 3))
#' inits <- list(inits1, inits2)
#' 
#' ## fitting the model with R2jags
#' set.seed(123)
#' fit <- R2jags::jags(data = datjags, inits = inits,
#'                     parameters.to.save = params, n.chains = 2, n.iter = 2000,
#'                     n.burnin = 1000, model.file = model)
#' 
#' ## printing out table
#' object <- mcmcTab(fit, 
#'           ci = c(0.025, 0.975), 
#'           pars = NULL, 
#'           Pr = FALSE,
#'           ROPE = NULL)
#' object
#' }
#'@export

mcmcTab <- function(sims, 
                    ci = c(0.025, 0.975), 
                    pars = NULL, 
                    Pr = FALSE,
                    ROPE = NULL) {
  
    if(class(sims)[1] == "jags" || class(sims)[1] == "rjags"){
      sims <- as.matrix(coda::as.mcmc(sims))
    }
    if(class(sims)[1] == "bugs"){
      sims <- sims$sims.matrix
    }  
    if(class(sims)[1] == "mcmc"){
      sims <- as.matrix(sims)
    }    
    if(class(sims)[1] == "mcmc.list"){
      sims <- as.matrix(sims)
    }      
    if(class(sims)[1] == "stanreg"){
      sims <- as.matrix(sims)
    } 
    if(class(sims)[1] == "stanfit"){
      sims <- as.matrix(sims)
    }     
    
    if(is.null(pars) == TRUE){
      dat <- sims
    }
    
    if(is.null(pars) == FALSE & length(pars) == 1){
      dat <- sims[, grepl(x = colnames(sims), pattern = pars)]
    }
    
    if(is.null(pars) == FALSE & length(pars) > 1){
      dat <- sims[, pars]
    }
    
    dat_wide <- t(dat)
    
    mcmctab <- apply(dat_wide, 1, 
                     function(x) c(Median = round(median(x), digits = 3), # Posterior median
                                   SD = round(sd(x), digits = 3), # Posterior SD
                                   Lower = as.numeric(round(quantile(x, probs = ci[1]), digits = 3)), # Lower CI of posterior
                                   Upper = as.numeric(round(quantile(x, probs = ci[2]), digits = 3)), # Upper CI of posterior
                                   Pr = round(ifelse(median(x) > 0, length(x[x > 0]) / length(x), length(x[x < 0]) / length(x)), digits = 3) # % of posterior draws with same sign as median
                     ))
    
    if(Pr == FALSE){
      mcmctab <- apply(dat_wide, 1, 
                       function(x) c(Median = round(median(x), digits = 3), # Posterior median
                                     SD = round(sd(x), digits = 3), # Posterior SD
                                     Lower = as.numeric(round(quantile(x, probs = ci[1]), digits = 3)), # Lower CI of posterior
                                     Upper = as.numeric(round(quantile(x, probs = ci[2]), digits = 3))))
    }
    
    if(!is.null(ROPE)){
      message("This table contains an estimate for parameter values outside of the region of 
          practical equivalence (ROPE). For this quantity to be meaningful, all parameters 
          must be on the same scale (e.g. standardized coefficients or first differences).")
      
      mcmctab <- apply(dat_wide, 1, 
                       function(x) c(Median = round(median(x), digits = 3), # Posterior median
                                     SD = round(sd(x), digits = 3), # Posterior SD
                                     Lower = as.numeric(round(quantile(x, probs = ci[1]), digits = 3)), # Lower CI of posterior
                                     Upper = as.numeric(round(quantile(x, probs = ci[2]), digits = 3)),
                                     PrOutROPE = round(ifelse(median(x) > 0, length(x[x > ROPE[2]]) / length(x), length(x[x < ROPE[1]]) / length(x)), digits = 3)))
    }
    
    # return(t(mcmctab))
    out_dat <- data.frame("Variable" = colnames(mcmctab), 
                          t(mcmctab),
                          row.names = NULL)
    
    return(out_dat)
  
} 