% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensor_calendarPlot.R
\name{sensor_calendarPlot}
\alias{sensor_calendarPlot}
\title{Plot time series values in conventional calendar format}
\usage{
sensor_calendarPlot(
  sensor = NULL,
  colors = NULL,
  breaks = NULL,
  labels = NULL,
  limits = c(0, 100),
  title = NULL,
  data.thresh = 50
)
}
\arguments{
\item{sensor}{An 'airsensor' object}

\item{colors}{Colours to be used for plotting. Options include "aqi", "scaqmd", “default”, 
“increment”, “heat”, “jet” and \pkg{RColorBrewer} colours — see the \pkg{openair}
\code{openColours} function for more details. For user defined the user can 
supply a list of colour names recognised by R (type \code{colours()} to see 
the full list). An example would be cols = \code{c("yellow", "green", "blue")}}

\item{breaks}{If a categorical scale is required then these breaks will be 
used. For example, \code{breaks = c(0, 50, 100, 1000)}. In this case “good” 
corresponds to values berween 0 and 50 and so on. Users should set the 
maximum value of breaks to exceed the maximum data value to ensure it is 
within the maximum final range e.g. 100–1000 in this case.}

\item{labels}{If a categorical scale is required then these labels will be 
used. Note there is one less label than break. For example, 
\code{labels = c("good", "bad", "very bad")}. breaks must also be supplied if 
labels are given.}

\item{limits}{Use this option to manually set the colour scale limits. This 
is useful in the case when there is a need for two or more plots and a 
consistent scale is needed on each. Set the limits to cover the maximimum 
range of the data for all plots of interest. For example, if one plot had 
data covering 0–60 and another 0–100, then set \code{limits = c(0, 100)}. 
Note that data will be ignored if outside the limits range.}

\item{title}{Optional title. If \code{NULL}, a default title will be constructed.}

\item{data.thresh}{Data capture threshold passed to \code{openair::timeAverage()}. 
For example, \code{data.thresh = 75} means that at least 75% of the data must 
be available in a day for the value to be calculate, else the data is removed.}
}
\value{
A plot and an object of class "openair".
}
\description{
Function for plotting PM2.5 concentration in a calendar format. 
This function wraps the \pkg{openair} \code{calendarPlot()} function.
}
\details{
Data are trimmed to the local-time year or month boundaries as appropriate.

Two special options are provided to specify a set of \code{colors}, 
\code{breaks} and \code{labels}.

Using \code{colors = "aqi"} will use US EPA Air Quality Index colors and breaks
defined by \code{breaks <- c(-Inf, 12, 35.5, 55.5, 150.5, 250.5, Inf)}.

Using \code{colors = "scaqmd"} will use a custom set of colors and breaks
defined by \code{breaks <- c(-Inf, 12, 35, 55, 75, Inf)}.
}
\note{
Daily averages are calculated using LST (Local Standard Time) day 
boundaries as specified by the US EPA. LST assumes that standard time applies
all year round and guarantees that every day has 24 hours -- no "spring forward"
or "fall back". Because of this, LST daily averages calculated during months
where daylight savings time is in effect will differ very slightly from daily 
averages calculated using local "clock time".
}
\examples{
\donttest{
library(AirSensor)

setArchiveBaseUrl("http://data.mazamascience.com/PurpleAir/v1")

# Monthly plot
sensor <- 
  sensor_loadMonth("scaqmd", 202007) \%>\%
  sensor_filterMeta(label == "SCSC_33")

sensor_calendarPlot(sensor)

# Annual plot
sensor <- 
  sensor_loadYear("scaqmd", 2020) \%>\%
  sensor_filterMeta(label == "SCSC_33")

sensor_calendarPlot(sensor)

# SCAQMD colors
sensor_calendarPlot(sensor, "scaqmd")

# Custom continuous color palette from RColorBrewer
sensor_calendarPlot(
  sensor,
  colors = "BuPu",
  title = "2020 Purple Scale",
  limits = range(sensor$data[,-1], na.rm = TRUE) # don't use data$datetime
)


# Custom categorical colors
sensor_calendarPlot(
  sensor,
  colors = c("springgreen2", "gold", "tomato3"),
  breaks = c(-Inf, 25, 50, Inf),
  labels = c("Good", "Fair", "Poor"),
  title = "2020 -- Air Quality Stoplight"
)

}
}
\references{
\href{https://aqs.epa.gov/aqsweb/documents/AQS_Data_Dictionary.html}{EPA AQS Data Dictionary}
}
\seealso{
\url{https://davidcarslaw.github.io/openair/reference/calendarPlot.html}
}
