% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_APCheatmap.R
\name{plot_APCheatmap}
\alias{plot_APCheatmap}
\title{Heatmap of an APC surface}
\usage{
plot_APCheatmap(
  dat,
  y_var = NULL,
  model = NULL,
  dimensions = c("period", "age"),
  apc_range = NULL,
  bin_heatmap = TRUE,
  bin_heatmapGrid_list = NULL,
  markLines_list = NULL,
  markLines_displayLabels = c("age", "period", "cohort"),
  y_var_logScale = FALSE,
  plot_CI = TRUE,
  legend_limits = NULL
)
}
\arguments{
\item{dat}{Dataset with columns \code{period} and \code{age}. If \code{y_var}
is specified, the dataset must contain the respective column. If \code{model}
is specified, the dataset must have been used for model estimation with
\code{gam} or \code{bam}.}

\item{y_var}{Optional character name of a metric variable to be plotted.}

\item{model}{Optional regression model estimated with \code{\link[mgcv]{gam}}
or \code{\link[mgcv]{bam}} to estimate a smoothed APC surface. Only used if
\code{y_var} is not
specified.}

\item{dimensions}{Character vector specifying the two APC dimensions that
should be visualized along the x-axis and y-axis. Defaults to
\code{c("period","age")}.}

\item{apc_range}{Optional list with one or multiple elements with names
\code{"age","period","cohort"} to filter the data. Each element should
contain a numeric vector of values for the respective variable that should
be kept in the data. All other values are deleted.}

\item{bin_heatmap, bin_heatmapGrid_list}{\code{bin_heatmap} indicates if the
heatmap surface should be binned. Defaults to TRUE. If TRUE, the binning
grid borders are defined by \code{bin_heatmapGrid_list}. This is a list with
each element a numeric vector and a name out of
\code{c("age","period","cohort")}. Can maximally have three elements. Defaults
to NULL, where the heatmap is binned in 5 year steps along the x-axis and the
y-axis.}

\item{markLines_list}{Optional list that can be used to highlight the borders
of specific age groups, time intervals or cohorts. Each element must be a
numeric vector of values where horizontal, vertical or diagonal lines should
be drawn (depends on which APC dimension is displayed on which axis).
The list can maximally have three elements and must have names out of
\code{c("age","period","cohort")}.}

\item{markLines_displayLabels}{Optional character vector defining for which
dimensions the lines defined through \code{markLines_list} should be marked
by a respective label. The vector should be a subset of
\code{c("age","period","cohort")}, or NULL to suppress all labels.
Defaults to \code{c("age","period","cohort")}.}

\item{y_var_logScale}{Indicator if \code{y_var} should be log10 transformed.
Only used if \code{y_var} is specified. Defaults to FALSE.}

\item{plot_CI}{Indicator if the confidence intervals should be plotted.
Only used if \code{y_var} is not specified. Defaults to TRUE.}

\item{legend_limits}{Optional numeric vector passed as argument \code{limits}
to \code{\link[ggplot2]{scale_fill_gradient2}}.}
}
\value{
Plot grid created with \code{\link[ggpubr]{ggarrange}} (if
\code{plot_CI} is TRUE) or a \code{ggplot2} object (if \code{plot_CI} is
FALSE).
}
\description{
Plot the heatmap of an APC structure. The function can be used in two ways:
Either to plot the observed mean structure of a metric variable, by
specifying \code{dat} and the variable \code{y_var}, or by specifying
\code{dat} and the \code{model} object, to plot some mean structure
represented by an estimated two-dimensional tensor product surface. The model
must be estimated with \code{\link[mgcv]{gam}} or \code{\link[mgcv]{bam}}.
}
\details{
See also \code{\link{plot_APChexamap}} to plot a hexagonal heatmap with
adapted axes.

If the plot is created based on the \code{model} object and the model was
estimated with a log or logit link, the function automatically performs an
exponential transformation of the effect.
}
\examples{
library(APCtools)
library(mgcv)

data(travel)

# variant A: plot observed mean structures
# observed heatmap
plot_APCheatmap(dat = travel, y_var = "mainTrip_distance",
                bin_heatmap = FALSE, y_var_logScale = TRUE)

# with binning
plot_APCheatmap(dat = travel, y_var = "mainTrip_distance",
                bin_heatmap = TRUE, y_var_logScale = TRUE)

# variant B: plot some smoothed, estimated mean structure
model <- gam(mainTrip_distance ~ te(age, period) + residence_region +
             household_size + s(household_income), data = travel)

# plot the smooth tensor product surface
plot_APCheatmap(dat = travel, model = model, bin_heatmap = FALSE, plot_CI = FALSE)

# ... same plot including the confidence intervals
plot_APCheatmap(dat = travel, model = model, bin_heatmap = FALSE)

# the APC dimensions can be flexibly assigned to the x-axis and y-axis
plot_APCheatmap(dat = travel, model = model, dimensions = c("age","cohort"),
                bin_heatmap = FALSE, plot_CI = FALSE)

# add some reference lines
plot_APCheatmap(dat = travel, model = model, bin_heatmap = FALSE, plot_CI = FALSE,
                markLines_list = list(cohort = c(1910,1939,1955,1980)))

# default binning of the tensor product surface in 5-year-blocks
plot_APCheatmap(dat = travel, model = model, plot_CI = FALSE)

# manual binning
manual_binning <- list(period = seq(min(travel$period, na.rm = TRUE) - 1,
                                    max(travel$period, na.rm = TRUE), by = 5),
                       cohort = seq(min(travel$period - travel$age, na.rm = TRUE) - 1,
                                    max(travel$period - travel$age, na.rm = TRUE), by = 10))
plot_APCheatmap(dat = travel, model = model, plot_CI = FALSE,
                bin_heatmapGrid_list = manual_binning)

}
\references{
Weigert, M., Bauer, A., Gernert, J., Karl, M., Nalmpatian, A.,
Küchenhoff, H., and Schmude, J. (2021). Semiparametric APC analysis of
destination choice patterns: Using generalized additive models to quantify
the impact of age, period, and cohort on travel distances.
\emph{Tourism Economics}. doi:10.1177/1354816620987198.
}
\seealso{
plot_APChexamap
}
\author{
Alexander Bauer \email{alexander.bauer@stat.uni-muenchen.de},
Maximilian Weigert \email{maximilian.weigert@stat.uni-muenchen.de}
}
