% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apci.R
\name{apci}
\alias{apci}
\title{Run APC-I model}
\usage{
apci(
  outcome = "inlfc",
  age = "acc",
  period = "pcc",
  cohort = NULL,
  weight = NULL,
  covariate = NULL,
  data,
  family = "quasibinomial",
  dev.test = TRUE,
  print = TRUE,
  gee = FALSE,
  id = NULL,
  corstr = "exchangeable",
  unequal_interval = FALSE,
  age_range = NULL,
  period_range = NULL,
  age_interval = NULL,
  period_interval = NULL,
  age_group = NULL,
  period_group = NULL,
  ...
)
}
\arguments{
\item{outcome}{An object of class character containing the name of the
outcome variable. The outcome variable can be continuous, categorical,
or count.}

\item{age}{An object of class character representing the age group index
taking on a small number of distinct values in the data. Usually, the vector
should be converted to a factor (or the terms of "category" and "enumerated
type").}

\item{period}{An object of class character, similar to the argument of age,
representing the time period index in the data.}

\item{cohort}{An optional object of class character representing cohort
membership index in the data. Usually, the cohort index can be generated
from the age group index and time period index in the data because of the
intrinsic relationship among these three time-related indices.}

\item{weight}{An optional vector of sample weights to be used in the model
fitting process. If non-NULL, the weights will be used in the first step to
estimate the model. Observations with negative weights will be automatically
dropped in modeling.}

\item{covariate}{An optional vector of characters, representing the name(s)
of the user-specified covariate(s) to be used in the model. If the
variable(s) are not found in data, there will be an error message reminding
the users to check the data again.}

\item{data}{A data frame containing the outcome variable, age group
indicator, period group indicator, and covariates to be used in the model.
If the variable(s) are not found in data, there will be an error message
reminding the users to check the input data again.}

\item{family}{Used to specify the statistical distribution of the error
term and link function to be used in the model. Usually, it is a character
string naming a family function. For example, family can be "binomial",
"multinomial"", or "gaussian". Users could also check R package glm for
more details of family functions.}

\item{dev.test}{Logical, specifying if the global F test should be
implemented before fitting the APC-I model. If \code{TRUE}, apci will first run the
global F test and report the test results; otherwise, apci will skip this
step and return NULL. The default setting is \code{TRUE}. However, users should be
aware that the algorithm will not automatically stop even if there is no
significant age-by-period interactions based on the global F test.}

\item{print}{Logical, specifying if the intermediate results should be
displayed in the console when fitting the model. The default setting is
\code{TRUE} to display the results of each procedure.}

\item{gee}{Logical, indicating if the data is cross-sectional data or
longitudinal/panel data. If \code{TRUE}, the generalized estimating equation
will be used to correct the standard error estimates. The default is
\code{FALSE}, indicating that the data are cross-sectional.}

\item{id}{A vector of character, specifying the cluster index in longitudinal
data. It is required when \code{gee} is \code{TRUE}. The length of the vector
should be the same as the number of observations.}

\item{corstr}{A character string, specifying a possible correlation
structure in the error terms when \code{gee} is \code{TRUE}. The following
are allowed: \code{independence}, \code{fixed}, \code{stat\_M\_dep},
\code{non\_stat\_M\_dep}, \code{exchangeable}, \code{AR-M} and
\code{unstructured}. The default value is \code{exchangeable}.}

\item{unequal_interval}{Logical, indicating if age and period groups are
of the same interval width. The default is set as \code{TRUE}.}

\item{age_range, period_range}{Numeric vector indicating the actual
age and period range (e.g., 10 to 59 years old from 2000 to 2019).}

\item{age_interval, period_interval, age_group, period_group}{Numeric
values or character vectors indicating how age and period are
grouped. \code{age_interval} and \code{period_interval} are numbers
indicating the width of age and period groups respectively.
\code{age_group} and \code{period_group} are character vectors
explicitly listing all potential age and period groups. Either
\code{age_interval}(\code{period_interval}) or \code{age_group}
(\code{period_group}) have to be defined when \code{unequal_interval}
is \code{TRUE}.}

\item{...}{Additional arguments to be passed to the function.}
}
\value{
A list containing:
\item{model}{The fitted generalized linear model.}
\item{intercept}{The overall intercept.}
\item{age_effect}{The estimated age main effect.}
\item{period_effect}{The estimated period main effect.}
\item{cohort_average}{The estimated inter-cohort average deviations from age
and period main effects.}
\item{cohort_slope}{The estimated intra-cohort life-course linear slopes.}
\item{int_matrix}{A matrix containing the estimated coefficients for
age-by-period interactions.}
\item{cohort_index}{Indices indicating different cohorts.}
\item{data}{Data used for fitting APC-I model.}
}
\description{
Run APC-I model
}
\examples{
# load package
library("APCI")
# load data
test_data <- APCI::women9017
test_data$acc <- as.factor(test_data$acc)
test_data$pcc <- as.factor(test_data$pcc)
test_data$educc <- as.factor(test_data$educc)
test_data$educr <- as.factor(test_data$educr)

# fit APC-I model
APC_I <- APCI::apci(outcome = "inlfc",
                    age = "acc",
                    period = "pcc",
                    cohort = "ccc",
                    weight = "wt",
                    data = test_data,dev.test=FALSE,
                    print = TRUE,
                    family = "gaussian")
summary(APC_I)

# explore the raw data pattern
apci.plot.raw(data = test_data, outcome_var = "inlfc",age = "acc",
              period = "pcc")
## alternatively,
apci.plot(data = test_data, outcome_var = "inlfc", age = "acc",model=APC_I,
          period = "pcc", type = "explore")

# visaulze estimated cohort effects with bar plot
apci.bar(model = APC_I, age = "acc",
         period = "pcc", outcome_var = "inlfc")

# visaulze estimated cohort effects with heatmap plot
apci.plot.heatmap(model = APC_I, age = "acc",period = "pcc")
## alternatively,
apci.plot(data = test_data, outcome_var = "inlfc", age = "acc",model=APC_I,
          period = "pcc")

}
