% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mo.R
\name{as.mo}
\alias{as.mo}
\alias{mo}
\alias{is.mo}
\alias{guess_mo}
\title{Transform to microorganism ID}
\usage{
as.mo(x, Becker = FALSE, Lancefield = FALSE, allow_uncertain = FALSE,
  reference_df = NULL)

is.mo(x)

guess_mo(x, Becker = FALSE, Lancefield = FALSE,
  allow_uncertain = FALSE, reference_df = NULL)
}
\arguments{
\item{x}{a character vector or a \code{data.frame} with one or two columns}

\item{Becker}{a logical to indicate whether \emph{Staphylococci} should be categorised into Coagulase Negative \emph{Staphylococci} ("CoNS") and Coagulase Positive \emph{Staphylococci} ("CoPS") instead of their own species, according to Karsten Becker \emph{et al.} [1].

  This excludes \emph{Staphylococcus aureus} at default, use \code{Becker = "all"} to also categorise \emph{S. aureus} as "CoPS".}

\item{Lancefield}{a logical to indicate whether beta-haemolytic \emph{Streptococci} should be categorised into Lancefield groups instead of their own species, according to Rebecca C. Lancefield [2]. These \emph{Streptococci} will be categorised in their first group, e.g. \emph{Streptococcus dysgalactiae} will be group C, although officially it was also categorised into groups G and L.

  This excludes \emph{Enterococci} at default (who are in group D), use \code{Lancefield = "all"} to also categorise all \emph{Enterococci} as group D.}

\item{allow_uncertain}{a logical to indicate whether empty results should be checked for only a part of the input string. When results are found, a warning will be given about the uncertainty and the result.}

\item{reference_df}{a \code{data.frame} to use for extra reference when translating \code{x} to a valid \code{mo}. The first column can be any microbial name, code or ID (used in your analysis or organisation), the second column must be a valid \code{mo} as found in the \code{\link{microorganisms}} data set.}
}
\value{
Character (vector) with class \code{"mo"}. Unknown values will return \code{NA}.
}
\description{
Use this function to determine a valid microorganism ID (\code{mo}). Determination is done using Artificial Intelligence (AI) and the complete taxonomic kingdoms \emph{Bacteria}, \emph{Fungi} and \emph{Protozoa} (see Source), so the input can be almost anything: a full name (like \code{"Staphylococcus aureus"}), an abbreviated name (like \code{"S. aureus"}), an abbreviation known in the field (like \code{"MRSA"}), or just a genus. You could also \code{\link{select}} a genus and species column, zie Examples.
}
\details{
A microbial ID from this package (class: \code{mo}) typically looks like these examples:\cr
\preformatted{
  Code                Full name
  ---------------     --------------------------------------
  B_KLBSL             Klebsiella
  B_KLBSL_PNE         Klebsiella pneumoniae
  B_KLBSL_PNE_RHI     Klebsiella pneumoniae rhinoscleromatis
  |   |    |   |
  |   |    |   |
  |   |    |    ----> subspecies, a 3-4 letter acronym
  |   |     ----> species, a 3-4 letter acronym
  |    ----> genus, a 5-7 letter acronym, mostly without vowels
   ----> taxonomic kingdom, either B (Bacteria), F (Fungi) or P (Protozoa)
}

Use the \code{\link{mo_property}} functions to get properties based on the returned code, see Examples.

This function uses Artificial Intelligence (AI) to help getting fast and logical results. It tries to find matches in this order:
\itemize{
  \item{Taxonomic kingdom: it first searches in bacteria, then fungi, then protozoa}
  \item{Human pathogenic prevalence: it first searches in more prevalent microorganisms, then less prevalent ones}
  \item{Valid MO codes and full names: it first searches in already valid MO code and known genus/species combinations}
  \item{Breakdown of input values: from here it starts to breakdown input values to find possible matches}
}

A couple of effects because of these rules
\itemize{
  \item{\code{"E. coli"} will return the ID of \emph{Escherichia coli} and not \emph{Entamoeba coli}, although the latter would alphabetically come first}
  \item{\code{"H. influenzae"} will return the ID of \emph{Haemophilus influenzae} and not \emph{Haematobacter influenzae} for the same reason}
  \item{Something like \code{"p aer"} will return the ID of \emph{Pseudomonas aeruginosa} and not \emph{Pasteurella aerogenes}}
  \item{Something like \code{"stau"} or \code{"S aur"} will return the ID of \emph{Staphylococcus aureus} and not \emph{Staphylococcus auricularis}}
}
This means that looking up human pathogenic microorganisms takes less time than looking up human \strong{non}-pathogenic microorganisms.

\code{guess_mo} is an alias of \code{as.mo}.
}
\section{ITIS}{

\if{html}{\figure{itis_logo.jpg}{options: height=60px style=margin-bottom:5px} \cr}
This package contains the \strong{complete microbial taxonomic data} (with all eight taxonomic ranks - from kingdom to subspecies) from the publicly available Integrated Taxonomic Information System (ITIS, \url{https://www.itis.gov}).

All (sub)species from \strong{the taxonomic kingdoms Bacteria, Fungi and Protozoa are included in this package}, as well as all previously accepted names known to ITIS. Furthermore, the responsible authors and year of publication are available. This allows users to use authoritative taxonomic information for their data analysis on any microorganism, not only human pathogens. It also helps to quickly determine the Gram stain of bacteria, since all bacteria are classified into subkingdom Negibacteria or Posibacteria.

ITIS is a partnership of U.S., Canadian, and Mexican agencies and taxonomic specialists [3].
}

\section{Source}{

[1] Becker K \emph{et al.} \strong{Coagulase-Negative Staphylococci}. 2014. Clin Microbiol Rev. 27(4): 870–926. \url{https://dx.doi.org/10.1128/CMR.00109-13}

[2] Lancefield RC \strong{A serological differentiation of human and other groups of hemolytic streptococci}. 1933. J Exp Med. 57(4): 571–95. \url{https://dx.doi.org/10.1084/jem.57.4.571}

[3] Integrated Taxonomic Information System (ITIS). Retrieved September 2018. \url{http://www.itis.gov}
}

\examples{
# These examples all return "B_STPHY_AUR", the ID of S. aureus:
as.mo("stau")
as.mo("STAU")
as.mo("staaur")
as.mo("S. aureus")
as.mo("S aureus")
as.mo("Staphylococcus aureus")
as.mo("MRSA") # Methicillin Resistant S. aureus
as.mo("VISA") # Vancomycin Intermediate S. aureus
as.mo("VRSA") # Vancomycin Resistant S. aureus
as.mo(369)    # Search on TSN (Taxonomic Serial Number), a unique identifier
              # for the Integrated Taxonomic Information System (ITIS)

as.mo("Streptococcus group A")
as.mo("GAS") # Group A Streptococci
as.mo("GBS") # Group B Streptococci

# guess_mo is an alias of as.mo and works the same
guess_mo("S. epidermidis")                 # will remain species: B_STPHY_EPI
guess_mo("S. epidermidis", Becker = TRUE)  # will not remain species: B_STPHY_CNS

guess_mo("S. pyogenes")                    # will remain species: B_STRPTC_PYO
guess_mo("S. pyogenes", Lancefield = TRUE) # will not remain species: B_STRPTC_GRA

# Use mo_* functions to get a specific property based on `mo`
Ecoli <- as.mo("E. coli")     # returns `B_ESCHR_COL`
mo_genus(Ecoli)               # returns "Escherichia"
mo_gramstain(Ecoli)           # returns "Gram negative"
# but it uses as.mo internally too, so you could also just use:
mo_genus("E. coli")           # returns "Escherichia"


\dontrun{
df$mo <- as.mo(df$microorganism_name)

# the select function of tidyverse is also supported:
library(dplyr)
df$mo <- df \%>\%
  select(microorganism_name) \%>\%
  as.mo()

# and can even contain 2 columns, which is convenient for genus/species combinations:
df$mo <- df \%>\%
  select(genus, species) \%>\%
  as.mo()
# although this works easier and does the same:
df <- df \%>\%
  mutate(mo = as.mo(paste(genus, species)))
}
}
\seealso{
\code{\link{microorganisms}} for the \code{data.frame} with ITIS content that is being used to determine ID's. \cr
The \code{\link{mo_property}} functions (like \code{\link{mo_genus}}, \code{\link{mo_gramstain}}) to get properties based on the returned code.
}
\keyword{Becker}
\keyword{Lancefield}
\keyword{becker}
\keyword{guess}
\keyword{lancefield}
\keyword{mo}
