% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seqic_9.R
\name{seqic_indicator_9}
\alias{seqic_indicator_9}
\title{SEQIC Indicator 9 - Emergency Department Transfer Timeliness}
\usage{
seqic_indicator_9(
  data,
  level,
  included_levels = c("I", "II", "III", "IV"),
  transfer_out_indicator,
  transport_method,
  unique_incident_id,
  trauma_team_activated,
  risk_group,
  ed_LOS,
  ed_decision_LOS,
  ed_decision_discharge_LOS,
  groups = NULL,
  calculate_ci = NULL,
  ...
)
}
\arguments{
\item{data}{A data frame containing trauma incident records.}

\item{level}{Column indicating the trauma center designation level (e.g., I,
II, III, IV).}

\item{included_levels}{Character vector indicating what facility levels to
include in the analysis.  Defaults to \code{c("I", "II", "III", "IV")}.}

\item{transfer_out_indicator}{Column name indicating whether the patient was
transferred out of the initial trauma center to definitive care. Logical,
character, or factor type. Values representing "No" (e.g., FALSE, "No")
indicate no transfer out.}

\item{transport_method}{Column identifying the EMS transport method (e.g.,
ambulance, private vehicle). Used to exclude non-qualified modes of
arrival.}

\item{unique_incident_id}{Unique identifier for each record.}

\item{trauma_team_activated}{Column indicating whether the trauma team was
activated (character, factor, or logical).}

\item{risk_group}{A character or factor column indicating the patient's risk
group (e.g., "High", "Moderate", "Low"). See risk definitions below.}

\item{ed_LOS}{Column for the calculated ED length of stay, measured in
minutes.}

\item{ed_decision_LOS}{Numeric column representing minutes from ED arrival to
decision to transfer.}

\item{ed_decision_discharge_LOS}{Numeric column representing minutes from ED
decision to discharge to physical discharge.}

\item{groups}{Additional columns passed as a vector of strings to
\code{dplyr::summarize()} via the \code{.by} argument for grouped summaries. Defaults
to \code{NULL}.}

\item{calculate_ci}{If \code{NULL}, 95\% confidence intervals will not be
calculated for the performance estimates.  Otherwise, options of "wilson"
or "clopper-pearson" can be supplied to utilize the corresponding methods
to calculate the confidence intervals for the proportions. Defaults to
\code{NULL}.}

\item{...}{
  Arguments passed on to \code{\link[nemsqar:nemsqa_binomial_confint]{nemsqar::nemsqa_binomial_confint}}
  \describe{
    \item{\code{conf.level}}{Numeric value between 0 and 1 indicating the confidence
level. Defaults to 0.95 (95\% confidence interval).}
    \item{\code{correct}}{Logical, indicating whether to apply continuity correction for
Wilson intervals. Defaults to \code{TRUE}.}
  }}
}
\value{
A list of four tibbles, with optional 95\% confidence intervals:
\itemize{
\item{\code{seqic_9_all}}: Proportion of transferred trauma patients with ED
discharge or decision delays >2 or >3 hours, grouped by optional
variables.
\item{\code{seqic_9_activations}}: Same proportions as above, further stratified
by trauma team activation status.
\item{\code{seqic_9_risk}}: Same proportions as above, further stratified by
risk groups.
\item{\code{seqic_9_activations_risk}}: Same proportions as above, further
stratified by risk groups and trauma team activation status.
}

Each tibble includes numerators, denominators, proportions, and (optionally)
confidence intervals for:
\itemize{
\item{9a}: Delayed discharge >2 hours
\item{9b}: Delayed discharge >3 hours
\item{9c}: Delayed decision >1 hours
\item{9d}: Delayed decision >2 hours
\item{9e}: Delayed decision to discharge >1 hour
\item{9f}: Delayed decision to discharge >2 hours
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Calculates the proportion of EMS-transferred trauma patients who experienced
delayed transfer from the emergency department (ED) based on disposition and
decision-to-transfer time frames. This includes both overall rates and
stratified results by trauma team activation status, with optional confidence
intervals.
}
\details{
This function:
\itemize{
\item Filters the dataset to include only transfers out from trauma centers
designated Level I through IV.
\item Deduplicates records using \code{unique_incident_id}.
\item Flags records where emergency department decision to discharge
occurred more than 60 or 120 minutes after ED arrival.
\item Flags records where physical departure from the ED occurred more than
120 or 180 minutes after ED arrival.
\item Flags records where physical discharge occurred more than 60 or 120
minutes after ED decision to discharge.
\item Stratifies results by trauma team activation status and
one or more grouping variables.
\item Stratifies results by risk groups and one or more grouping variables.
\item Returns a summarized tibble with the number of delayed cases
(numerator), eligible records (denominator), and the proportion for each
delay threshold.
\item Optionally includes 95\% confidence intervals if \code{calculate_ci = TRUE}.
}
}
\note{
This function calculates discharge timeliness outcomes for patients
transported to trauma centers, stratified by risk of mortality. Risk
groups—low, moderate, and high— are defined by the Iowa System Evaluation and
Quality Improvement Committee (SEQIC) as described below. Users may also
apply alternative risk stratification methods if preferred.

\itemize{
\item Abnormal Physiology Criteria: GCS 3–5; Respirations <5 or >30 per
minute; Systolic BP <60 mm Hg
\item High Risk: Probability of Survival < 0.2; ISS > 41; ISS > 24 with
abnormal physiology
\item Moderate Risk: Probability of Survival 0.2–0.5; ISS 16–41
\item Low Risk: Probability of Survival > 0.5; ISS < 16; Normal physiology
}

Users must ensure appropriate column names are passed and data is
pre-processed to include the necessary fields without missing critical
identifiers or timestamps.
}
\examples{
# Packages
library(dplyr)
library(traumar)

# Simulated dataset for SEQIC Indicator 9
test_data <- tibble::tibble(
  id = as.character(1:10),
  trauma_level = c("I", "II", "III", "IV", "V", "II", "III", "IV", "I",
  "II"),
  transport = c("Ambulance", "Ambulance", "Private Vehicle", "Ambulance",
  "Helicopter", "Ambulance", "Ambulance", "Ambulance", "Ambulance",
  "Ambulance"),
  activated = c(TRUE, FALSE, TRUE, FALSE, TRUE, FALSE, TRUE, TRUE, FALSE,
  FALSE),
  ed_LOS = c(120, 180, 90, 60, 200, 130, 110, 160, 95, 220),
  ed_decision = c(55, 125, 65, 30, 190, 80, 70, 45, 61, 130),
  ed_discharge = c(130, 185, 110, 65, 150, 160, 95, 180, 70, 210),
  transfer_out = c(TRUE, TRUE, FALSE, TRUE, TRUE, TRUE, TRUE, FALSE, TRUE,
  TRUE),
  risk = c("High", "High", "Moderate", "Low", "Moderate", "Low",
           "High", "Low", "Moderate", "High")
)

# Run the function, and store as a list object
seqic_9_result <- traumar::seqic_indicator_9(
  data = test_data,
  level = trauma_level,
  included_levels = c("I", "II", "III", "IV"),
  unique_incident_id = id,
  transport_method = transport,
  transfer_out_indicator = transfer_out,
  ed_LOS = ed_LOS,
  ed_decision_LOS = ed_decision,
  ed_decision_discharge_LOS = ed_discharge,
  trauma_team_activated = activated,
  risk_group = risk
)

# Take a look at the overall output of the function
seqic_9_result$overall |>
tidyr::pivot_longer(cols = -1,
                    names_to = "Indicator",
                    values_to = "Values"
                    )

}
\author{
Nicolas Foss, Ed.D., MS
}
