% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Pn.R
\name{Pn}
\alias{Pn}
\alias{psi_Pn}
\alias{Gegen_coefs_Pn}
\alias{akx}
\alias{f_locdev_Pn}
\title{Utilities for projected-ecdf statistics of spherical uniformity}
\usage{
psi_Pn(theta, q, type, Rothman_t = 1/3, tilde = FALSE, psi_Gauss = TRUE,
  psi_N = 320, tol = 1e-06)

Gegen_coefs_Pn(k, p, type, Rothman_t = 1/3, Gauss = TRUE, N = 320,
  tol = 1e-06, verbose = FALSE)

akx(x, p, k, sqr = FALSE)

f_locdev_Pn(p, type, K = 1000, N = 320, K_max = 10000, thre = 0.001,
  Rothman_t = 1/3, verbose = FALSE)
}
\arguments{
\item{theta}{vector with values in \eqn{[0, \pi]}.}

\item{q}{integer giving the dimension of the sphere \eqn{S^q}.}

\item{type}{type of projected-ecdf test statistic. Must be either
\code{"PCvM"} (Cramér--von Mises), \code{"PAD"} (Anderson--Darling), or
\code{"PRt"} (Rothman).}

\item{Rothman_t}{\eqn{t} parameter for the Rothman test, a real in
\eqn{(0, 1)}. Defaults to \code{1 / 3}.}

\item{tilde}{include the constant and bias term? Defaults to \code{FALSE}.}

\item{psi_Gauss}{use a \link[=Gauss_Legen_nodes]{Gauss--Legendre quadrature}
rule with \code{psi_N} nodes in the computation of the kernel function?
Defaults to \code{TRUE}.}

\item{psi_N}{number of points used in the Gauss--Legendre quadrature for
computing the kernel function. Defaults to \code{320}.}

\item{tol}{tolerance passed to \code{\link{integrate}}'s \code{rel.tol} and
\code{abs.tol} if \code{Gauss = FALSE}. Defaults to \code{1e-6}.}

\item{k}{vector with the index of coefficients.}

\item{p}{integer giving the dimension of the ambient space \eqn{R^p} that
contains \eqn{S^{p-1}}.}

\item{Gauss}{use a Gauss--Legendre quadrature rule of \code{N} nodes
in the computation of the Gegenbauer coefficients? Otherwise, call
\code{\link{integrate}}. Defaults to \code{TRUE}.}

\item{N}{number of points used in the \link[=Gauss_Legen_nodes]{
Gauss--Legendre quadrature} for computing the Gegenbauer coefficients.
Defaults to \code{320}.}

\item{verbose}{flag to print informative messages. Defaults to \code{FALSE}.}

\item{x}{evaluation points for \eqn{a_{k, p}^x}, a vector with values in
\eqn{[-1, 1]}.}

\item{sqr}{return the \emph{signed} square root of \eqn{a_{k, p}^x}?
Defaults to \code{FALSE}.}

\item{K}{number of equispaced points on \eqn{[-1, 1]} used for evaluating
\eqn{f} and then interpolating. Defaults to \code{1e3}.}

\item{K_max}{integer giving the truncation of the series. Defaults to
\code{1e4}.}

\item{thre}{proportion of norm \emph{not} explained by the first terms of the
truncated series. Defaults to \code{1e-3}.}
}
\value{
\itemize{
  \item \code{psi_Pn}: a vector of size \code{length(theta)} with the
  evaluation of \eqn{\psi}.
  \item \code{Gegen_coefs_Pn}: a vector of size \code{length(k)} containing
  the coefficients \eqn{b_{k, p}^W}.
  \item \code{akx}: a matrix of size \code{c(length(x), length(k))}
  containing the coefficients \eqn{a_{k, p}^x}.
  \item \code{f_locdev_Pn}: the projected alternative \eqn{f} as a function
  ready to be evaluated.
}
}
\description{
Computation of the kernels
\deqn{\psi_p^W(\theta) := \int_{-1}^1 A_x(\theta)\,\mathrm{d}W(F_p(x)),}{
\psi_p^W(\theta) := \int_{-1}^1 A_x(\theta) dW(F_p(x)),
}
where \eqn{A_x(\theta)} is the proportion of area surface of
\eqn{S^{p - 1}} covered by the
\link[=A_theta_x]{intersection of two hyperspherical caps} with common solid
angle \eqn{\pi - \cos^{-1}(x)} and centers separated by
an angle \eqn{\theta \in [0, \pi]}, \eqn{F_p} is the distribution function
of the \link[=p_proj_unif]{projected spherical uniform distribution},
and \eqn{W} is a measure on \eqn{[0, 1]}.

Also, computation of the \link[=Gegen_coefs]{Gegenbauer coefficients} of
\eqn{\psi_p^W}:
\deqn{b_{k, p}^W := \frac{1}{c_{k, p}}\int_0^\pi \psi_p^W(\theta)
C_k^{p / 2 - 1}(\cos\theta)\,\mathrm{d}\theta.}{
b_{k, p}^W := \frac{1}{c_{k, p}} \int_0^\pi \psi_p^W(\theta)
C_k^(p / 2 - 1)(\cos\theta) d\theta.}
These coefficients can also be computed via
\deqn{b_{k, p}^W = \int_{-1}^1 a_{k, p}^x\,\mathrm{d}W(F_p(x))}{
b_{k, p}^W = \int_{-1}^1 a_{k, p}^x dW(F_p(x))}
for a certain function \eqn{x\rightarrow a_{k, p}^x}. They serve to define
\link[=locdev]{projected alternatives to uniformity}.
}
\details{
The evaluation of \eqn{\psi_p^W} and \eqn{b_{k, p}^W} depends on the type of
projected-ecdf statistic:
\itemize{
\item PCvM: closed-form expressions for \eqn{\psi_p^W} and \eqn{b_{k, p}^W}
with \eqn{p = 2, 3, 4}, numerical integration required for \eqn{p \ge 5}.
\item PAD: closed-form expressions for \eqn{\psi_2^W} and \eqn{b_{k, 3}^W},
numerical integration required for \eqn{\psi_p^W} with \eqn{p \ge 3} and
\eqn{b_{k, p}^W} with \eqn{p = 2} and \eqn{p \ge 4}.
\item PRt: closed-form expressions for \eqn{\psi_p^W} and \eqn{b_{k, p}^W}
for any \eqn{p \ge 2}.
}
See García-Portugués et al. (2023) for more details.
}
\examples{
# Kernels in the projected-ecdf test statistics
k <- 0:10
coefs <- list()
(coefs$PCvM <- t(sapply(2:5, function(p)
  Gegen_coefs_Pn(k = k, p = p, type = "PCvM"))))
(coefs$PAD <- t(sapply(2:5, function(p)
  Gegen_coefs_Pn(k = k, p = p, type = "PAD"))))
(coefs$PRt <- t(sapply(2:5, function(p)
  Gegen_coefs_Pn(k = k, p = p, type = "PRt"))))

# Gegenbauer expansion
th <- seq(0, pi, length.out = 501)[-501]
old_par <- par(mfrow = c(3, 4))
for (type in c("PCvM", "PAD", "PRt")) {

  for (p in 2:5) {

    plot(th, psi_Pn(theta = th, q = p - 1, type = type), type = "l",
         main = paste0(type, ", p = ", p), xlab = expression(theta),
         ylab = expression(psi(theta)), axes = FALSE, ylim = c(-1.5, 1))
    axis(1, at = c(0, pi / 4, pi / 2, 3 * pi / 4, pi),
         labels = expression(0, pi / 4, pi / 2, 3 * pi / 4, pi))
    axis(2); box()
    lines(th, Gegen_series(theta = th, coefs = coefs[[type]][p - 1, ],
                           k = k, p = p), col = 2)

  }

}
par(old_par)

# Analytical coefficients vs. numerical integration
test_coef <- function(type, p, k = 0:20) {

  plot(k, log1p(abs(Gegen_coefs_Pn(k = k, p = p, type = type))),
       ylab = "Coefficients", main = paste0(type, ", p = ", p))
  points(k, log1p(abs(Gegen_coefs(k = k, p = p, psi = psi_Pn, type = type,
                                  q = p - 1))), col = 2)
  legend("topright", legend = c("log(1 + Gegen_coefs_Pn))",
                                "log(1 + Gegen_coefs(psi_Pn))"),
         lwd = 2, col = 1:2)

}

# PCvM statistic
old_par <- par(mfrow = c(2, 2))
for (p in 2:5) test_coef(type = "PCvM", p = p)
par(old_par)

# PAD statistic
old_par <- par(mfrow = c(2, 2))
for (p in 2:5) test_coef(type = "PAD", p = p)
par(old_par)

# PRt statistic
old_par <- par(mfrow = c(2, 2))
for (p in 2:5) test_coef(type = "PRt", p = p)
par(old_par)

# akx
akx(x = seq(-1, 1, l = 5), k = 1:4, p = 2)
akx(x = 0, k = 1:4, p = 3)

# PRt alternative to uniformity
z <- seq(-1, 1, l = 1e3)
p <- c(2:5, 10, 15, 17)
col <- viridisLite::viridis(length(p))
plot(z, f_locdev_Pn(p = p[1], type = "PRt")(z), type = "s",
     col = col[1], ylim = c(0, 0.6), ylab = expression(f[Rt](z)))
for (k in 2:length(p)) {
  lines(z, f_locdev_Pn(p = p[k], type = "PRt")(z), type = "s", col = col[k])
}
legend("topleft", legend = paste("p =", p), col = col, lwd = 2)
}
\references{
García-Portugués, E., Navarro-Esteban, P., Cuesta-Albertos, J. A. (2023)
On a projection-based class of uniformity tests on the hypersphere.
\emph{Bernoulli}, 29(1):181--204. \doi{10.3150/21-BEJ1454}.
}
\author{
Eduardo García-Portugués and Paula Navarro-Esteban.
}
