#' Gather League Transactions for Specific Round
#'
#' Given a league ID and round, grab the transaction data concerning that particular league.
#' This includes adds or drops, whether it was a free agent or a waiver, notes, and more.
#'
#' @return Returns a data frame containing information about the round of transactions for that league.
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, September 2021
#' @keywords transactions league
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#' @export
#' @examples
#' \dontrun{get_transactions(688281863499907072, 2)}
#'
#' @param league_id League ID generated by Sleeper (numeric or character)
#' @param round round of transactions to gather (can also be thought of as week depending on when your league processes transactions) (numeric)
#'
get_transactions <- function(league_id, round) {
  # Check if class of round parameter is numeric
  if(!is.numeric(round)) {
    # If not numeric, inform user and halt function
    stop("round parameter must be of type numeric")
  }
  # Query results from API given league ID and round specified
  x <- jsonlite::fromJSON(httr::content(httr::GET(paste0("https://api.sleeper.app/v1/league/", league_id, "/transactions/", round)), as = "text"))
  # Check if returned object is a list
  if(inherits(x, "list")) {
    # If returned object is a list, inform user and return nothing
    message("No data found. Were the league ID and round entered correctly?")
  } else {
    # If returned object is not a list, break out nested data frames
    x_drops <- x$drops
    x_adds <- x$adds
    x_settings <- x$settings
    x_metadata <- x$metadata
    # Remove nested data frames from main query
    x$drops <- NULL
    x$adds <- NULL
    x$settings <- NULL
    x$metadata <- NULL
    # Bind broken out data frames back into main query
    x_fin <- cbind(x, x_drops, x_adds, x_settings, x_metadata)
    # Return final data frame
    return(x_fin)
  }
}

