% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_rejection.R
\name{extrapolate_rejection}
\alias{extrapolate_rejection}
\title{Extrapolate coverage and width using sub-sampled bootstrap confidence
  intervals.}
\usage{
extrapolate_rejection(
  data,
  pvalue_subsamples,
  B_target = Inf,
  alpha = 0.05,
  nested = FALSE,
  format = "wide"
)
}
\arguments{
\item{data}{data frame or tibble containing the simulation results.}

\item{pvalue_subsamples}{list or name of column from \code{data} containing list
of confidence intervals calculated based on sub-samples with different
numbers of replications.}

\item{B_target}{number of bootstrap replications to which the criteria should
be extrapolated, with a default of \code{B = Inf}.}

\item{alpha}{scalar or vector indicating the nominal alpha level(s). Default
value is set to the conventional .05.}

\item{nested}{logical value controlling the format of the output. If
\code{FALSE} (the default), then the results will be returned as a data
frame with rows for each distinct number of bootstraps. If \code{TRUE},
then the results will be returned as a data frame with a single row, with
each performance criterion containing a nested data frame.}

\item{format}{character string controlling the format of the output when
\code{CI_subsamples} has results for more than one type of confidence
interval. If \code{"wide"} (the default), then each performance criterion
will have a separate column for each CI type. If \code{"long"}, then each
performance criterion will be a single variable, with separate rows for
each CI type.}
}
\value{
A tibble containing the number of simulation iterations, performance
  criteria estimate(s) and the associated MCSE.
}
\description{
Given a set of bootstrap confidence intervals calculated across
  sub-samples with different numbers of replications, extrapolates confidence
  interval coverage and width of bootstrap confidence intervals to a
  specified (larger) number of bootstraps. The function also calculates the
  associated Monte Carlo standard errors. The confidence interval percentage
  is based on how you calculated the lower and upper bounds.
}
\examples{

# function to generate data from two distinct populations
dgp <- function(N_A, N_B, shape_A, scale_A, shape_B, scale_B) {
  data.frame(
    group = rep(c("A","B"), c(N_A, N_B)),
      y = c(
        rgamma(N_A, shape = shape_A, scale = scale_A),
        rgamma(N_B, shape = shape_B, scale = scale_B)
      )
  )
}

# function to do a bootstrap t-test
estimator <- function(
    dat,
    B_vals = c(49,59,89,99), # number of booties to evaluate
    pval_reps = 4L
) {
  stat <- t.test(y ~ group, data = dat)$statistic

  # create bootstrap replications under the null of no difference
  boot_dat <- dat
  booties <- replicate(max(B_vals), {
    boot_dat$group <- sample(dat$group)
    t.test(y ~ group, data = boot_dat)$statistic
  })

  # calculate multiple bootstrap p-values using sub-sampling of replicates
  res <- data.frame(stat = stat)

  res$pvalue_subsamples <- bootstrap_pvals(
    boot_stat = booties,
    stat = stat,
    B_vals = B_vals,
    reps = pval_reps,
    enlist = TRUE
  )

  res
}

# create simulation driver
simulate_boot_pvals <- bundle_sim(
  f_generate = dgp,
  f_analyze = estimator
)

# replicate the bootstrap process
x <- simulate_boot_pvals(
  reps = 50L,
  N_A = 20, N_B = 25,
  shape_A = 7, scale_A = 2,
  shape_B = 4, scale_B = 3,
  B_vals = c(49, 99, 149, 199),
  pval_reps = 2L
)

extrapolate_rejection(
  data = x,
  pvalue_subsamples = pvalue_subsamples,
  B_target = 1999,
  alpha = c(.01, .05, .10)
)

extrapolate_rejection(
  data = x,
  pvalue_subsamples = pvalue_subsamples,
  B_target = Inf,
  alpha = c(.01, .05, .10),
  nested = TRUE
)

}
\references{
\insertRef{boos2000MonteCarloEvaluation}{simhelpers}
}
