% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenience-functions.R
\name{transform_forecasts}
\alias{transform_forecasts}
\title{Transform forecasts and observed values}
\usage{
transform_forecasts(data, fun = log_shift, append = TRUE, label = "log", ...)
}
\arguments{
\item{data}{A data.frame or data.table with the predictions and observations.
For scoring using \code{\link[=score]{score()}}, the following columns need to be present:
\itemize{
\item \code{true_value} - the true observed values
\item \code{prediction} - predictions or predictive samples for one
true value. (You only don't need to provide a prediction column if
you want to score quantile forecasts in a wide range format.)}
For scoring integer and continuous forecasts a \code{sample} column is needed:
\itemize{
\item \code{sample} - an index to identify the predictive samples in the
prediction column generated by one model for one true value. Only
necessary for continuous and integer forecasts, not for
binary predictions.}
For scoring predictions in a quantile-format forecast you should provide
a column called \code{quantile}:
\itemize{
\item \code{quantile}: quantile to which the prediction corresponds
}

In addition a \code{model} column is suggested and if not present this will be
flagged and added to the input data with all forecasts assigned as an
"unspecified model").

You can check the format of your data using \code{\link[=check_forecasts]{check_forecasts()}} and there
are examples for each format (\link{example_quantile}, \link{example_continuous},
\link{example_integer}, and \link{example_binary}).}

\item{fun}{A function used to transform both true values and predictions.
The default function is \code{\link[=log_shift]{log_shift()}}, a custom function that is essentially
the same as \code{\link[=log]{log()}}, but has an additional arguments (\code{offset})
that allows you add an offset before applying the logarithm. This is often
helpful as the natural log transformation is not defined at zero. A common,
and pragmatic solution, is to add a small offset to the data before applying
the log transformation. In our work we have often used an offset of 1 but
the precise value will depend on your application.}

\item{append}{Logical, defaults to \code{TRUE}. Whether or not to append a
transformed version of the data to the currently existing data (\code{TRUE}). If
selected, the data gets transformed and appended to the existing data frame,
making it possible to use the outcome directly in \code{\link[=score]{score()}}. An additional
column, 'scale', gets created that denotes which rows or untransformed
('scale' has the value "natural") and which have been transformed ('scale'
has the value passed to the argument \code{label}).}

\item{label}{A string for the newly created 'scale' column to denote the
newly transformed values. Only relevant if \code{append = TRUE}.}

\item{...}{Additional parameters to pass to the function you supplied. For
the default option of \code{\link[=log_shift]{log_shift()}} this could be the \code{offset} argument.}
}
\value{
A \code{data.table} with either a transformed version of the data, or one
with both the untransformed and the transformed data. includes the original
data as well as a transformation of the original data. There will be one
additional column, `scale', present which will be set to "natural" for the
untransformed forecasts.
}
\description{
Function to transform forecasts and true values before scoring.
}
\details{
There are a few reasons, depending on the circumstances, for
why this might be desirable (check out the linked reference for more info).
In epidemiology, for example, it may be useful to log-transform incidence
counts before evaluating forecasts using scores such as the weighted interval
score (WIS) or the continuous ranked probability score (CRPS).
Log-transforming forecasts and observations changes the interpretation of
the score from a measure of absolute distance between forecast and
observation to a score that evaluates a forecast of the exponential growth
rate. Another motivation can be to apply a variance-stabilising
transformation or to standardise incidence counts by population.

Note that if you want to apply a transformation, it is important to transform
the forecasts and observations and then apply the score. Applying a
transformation after the score risks losing propriety of the proper scoring
rule.
}
\examples{

library(magrittr) # pipe operator

# transform forecasts using the natural logarithm
# negative values need to be handled (here by replacing them with 0)
example_quantile \%>\%
  .[, true_value := ifelse(true_value < 0, 0, true_value)] \%>\%
# Here we use the default function log_shift() which is essentially the same
# as log(), but has an additional arguments (offset) that allows you add an
# offset before applying the logarithm.
  transform_forecasts(append = FALSE) \%>\%
  head()

# alternatively, integrating the truncation in the transformation function:
example_quantile \%>\%
 transform_forecasts(
   fun = function(x) {log_shift(pmax(0, x))}, append = FALSE
 ) \%>\%
 head()

# specifying an offset for the log transformation removes the
# warning caused by zeros in the data
example_quantile \%>\%
  .[, true_value := ifelse(true_value < 0, 0, true_value)] \%>\%
  transform_forecasts(offset = 1, append = FALSE) \%>\%
  head()

# adding square root transformed forecasts to the original ones
example_quantile \%>\%
  .[, true_value := ifelse(true_value < 0, 0, true_value)] \%>\%
  transform_forecasts(fun = sqrt, label = "sqrt") \%>\%
  score() \%>\%
  summarise_scores(by = c("model", "scale"))

# adding multiple transformations
example_quantile \%>\%
  .[, true_value := ifelse(true_value < 0, 0, true_value)] \%>\%
  transform_forecasts(fun = log_shift, offset = 1) \%>\%
  transform_forecasts(fun = sqrt, label = "sqrt") \%>\%
  head()
}
\references{
Transformation of forecasts for evaluating predictive
performance in an epidemiological context
Nikos I. Bosse, Sam Abbott, Anne Cori, Edwin van Leeuwen, Johannes Bracher,
Sebastian Funk
medRxiv 2023.01.23.23284722
\doi{https://doi.org/10.1101/2023.01.23.23284722}
\url{https://www.medrxiv.org/content/10.1101/2023.01.23.23284722v1}
}
\author{
Nikos Bosse \email{nikosbosse@gmail.com}
}
\keyword{check-forecasts}
