% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/topic-nse.R
\name{topic-inject}
\alias{topic-inject}
\alias{quasiquotation}
\alias{nse-force}
\alias{nse-inject}
\title{Injecting with \verb{!!}, \verb{!!!}, and glue syntax}
\description{
The injection operators are extensions of R implemented by rlang to modify a piece of code before R processes it. There are two main families:
\itemize{
\item The \link[=dyn-dots]{dynamic dots} operators, \code{\link{!!!}} and \ifelse{html}{\code{\link[=glue-operators]{"\{"}}}{\verb{"\{"}}.
\item The \link[=topic-metaprogramming]{metaprogramming operators} \code{\link{!!}}, \ifelse{html}{\code{\link[=embrace-operator]{\{\{}}}{\verb{\{\{}}, and \ifelse{html}{\code{\link[=glue-operators]{"\{\{"}}}{\verb{"\{\{"}}. Splicing with \code{\link{!!!}} can also be done in metaprogramming context.
}
}
\section{Dots injection}{
Unlike regular \code{...}, \link[=dyn-dots]{dynamic dots} are programmable with injection operators.
\subsection{Splicing with \verb{!!!}}{

For instance, take a function like \code{rbind()} which takes data in \code{...}. To bind rows, you supply them as separate arguments:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rbind(a = 1:2, b = 3:4)
#>   [,1] [,2]
#> a    1    2
#> b    3    4
}\if{html}{\out{</div>}}

But how do you bind a variable number of rows stored in a list? The base R solution is to invoke \code{rbind()} with \code{do.call()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rows <- list(a = 1:2, b = 3:4)

do.call("rbind", rows)
#>   [,1] [,2]
#> a    1    2
#> b    3    4
}\if{html}{\out{</div>}}

Functions that implement dynamic dots include a built-in way of folding a list of arguments in \code{...}. To illustrate this, we'll create a variant of \code{rbind()} that takes dynamic dots by collecting \code{...} with \code{\link[=list2]{list2()}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rbind2 <- function(...) \{
  do.call("rbind", list2(...))
\}
}\if{html}{\out{</div>}}

It can be used just like \code{rbind()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rbind2(a = 1:2, b = 3:4)
#>   [,1] [,2]
#> a    1    2
#> b    3    4
}\if{html}{\out{</div>}}

And a list of arguments can be supplied by \emph{splicing} the list with \code{\link{!!!}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rbind2(!!!rows, c = 5:6)
#>   [,1] [,2]
#> a    1    2
#> b    3    4
#> c    5    6
}\if{html}{\out{</div>}}
}

\subsection{Injecting names with \verb{"\{"}}{

A related problem comes up when an argument name is stored in a variable. With dynamic dots, you can inject the name using \href{https://glue.tidyverse.org/}{glue syntax} with \ifelse{html}{\code{\link[=glue-operators]{"\{"}}}{\verb{"\{"}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{name <- "foo"

rbind2("\{name\}" := 1:2, bar = 3:4)
#>     [,1] [,2]
#> foo    1    2
#> bar    3    4
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{
rbind2("prefix_\{name\}" := 1:2, bar = 3:4)
#>            [,1] [,2]
#> prefix_foo    1    2
#> bar           3    4
}\if{html}{\out{</div>}}
}
}

\section{Metaprogramming injection}{
\link[=topic-data-mask]{Data-masked} arguments support the following injection operators. They can also be explicitly enabled with \code{\link[=inject]{inject()}}.
\subsection{Embracing with \verb{\{\{}}{

The embracing operator \ifelse{html}{\code{\link[=embrace-operator]{\{\{}}}{\verb{\{\{}} is made specially for function arguments. It \link[=topic-defuse]{defuses} the expression supplied as argument and immediately injects it in place. The injected argument is then evaluated in another context such as a \link[=topic-data-mask]{data mask}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Inject function arguments that might contain
# data-variables by embracing them with \{\{ \}\}
mean_by <- function(data, by, var) \{
  data \%>\%
    dplyr::group_by(\{\{ by \}\}) \%>\%
    dplyr::summarise(avg = mean(\{\{ var \}\}, na.rm = TRUE))
\}

# The data-variables `cyl` and `disp` inside the
# env-variables `by` and `var` are injected inside `group_by()`
# and `summarise()`
mtcars \%>\% mean_by(by = cyl, var = disp)
#> # A tibble: 3 x 2
#>     cyl   avg
#>   <dbl> <dbl>
#> 1     4  105.
#> 2     6  183.
#> 3     8  353.
}\if{html}{\out{</div>}}

Learn more about this pattern in \ifelse{html}{\link[=topic-data-mask-programming]{Data mask programming patterns}}{\link[=topic-data-mask-programming]{Data mask programming patterns}}.
}

\subsection{Injecting with \verb{!!}}{

Unlike \code{\link{!!!}} which injects a list of arguments, the injection operator \code{\link{!!}} (pronounced "bang-bang") injects a \emph{single} object. One use case for \verb{!!}  is to substitute an environment-variable (created with \verb{<-}) with a data-variable (inside a data frame).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# The env-variable `var` contains a data-symbol object, in this
# case a reference to the data-variable `height`
var <- data_sym("disp")

# We inject the data-variable contained in `var` inside `summarise()` 
mtcars \%>\%
  dplyr::summarise(avg = mean(!!var, na.rm = TRUE))
#> # A tibble: 1 x 1
#>     avg
#>   <dbl>
#> 1  231.
}\if{html}{\out{</div>}}

Another use case is to inject a variable by value to avoid \link[=topic-data-mask-ambiguity]{name collisions}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{df <- data.frame(x = 1)

# This name conflicts with a column in `df`
x <- 100

# Inject the env-variable
df \%>\%
  dplyr::mutate(x = x / !!x)
#>      x
#> 1 0.01
}\if{html}{\out{</div>}}

Note that in most cases you don't need injection with \verb{!!}. For instance, the \code{\link{.data}} and \code{\link{.env}} pronouns provide more intuitive alternatives to injecting a column name and injecting a value.
}

\subsection{Splicing with \verb{!!!}}{

The splice operator \code{\link{!!!}} of dynamic dots can also be used in metaprogramming context (inside \link[=topic-data-mask]{data-masked} arguments and inside \code{\link[=inject]{inject()}}). For instance, we could reimplement the \code{rbind2()} function presented above using \code{inject()} instead of \code{do.call()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rbind2 <- function(...) \{
  inject(rbind(!!!list2(...)))
\}
}\if{html}{\out{</div>}}

There are two things going on here. We collect \code{...} with \code{\link[=list2]{list2()}} so that the callers of \code{rbind2()} may use \verb{!!!}. And we use \code{inject()} so that \code{rbind2()} itself may use \verb{!!!} to splice the list of arguments passed to \code{rbind2()}.
}
}

\section{Injection in other languages}{
Injection is known as \strong{quasiquotation} in other programming languages and in computer science. \code{expr()} is similar to a quasiquotation operator and \verb{!!} is the unquote operator. These terms have a rich history in Lisp languages, and live on in modern languages like \href{https://docs.julialang.org/en/v1/manual/metaprogramming/}{Julia} and \href{https://docs.racket-lang.org/reference/quasiquote.html}{Racket}. In base R, quasiquotation is performed with \code{\link[=bquote]{bquote()}}.

The main difference between rlang and other languages is that quasiquotation is often implicit instead of explicit. You can use injection operators in any defusing / quoting function (unless that function defuses its argument with a special operator like \code{\link[=enquo0]{enquo0()}}). This is not the case in lisp languages for example where injection / unquoting is explicit and only enabled within a backquote.
}

\section{See also}{
\itemize{
\item \ifelse{html}{\link[=topic-inject-out-of-context]{What happens if I use injection operators out of context?}}{\link[=topic-inject-out-of-context]{What happens if I use injection operators out of context?}}
}
}

\keyword{internal}
