#' Model the Data with Multivariate Adjustment
#'
#' Using the methodology generated by Muoko et. al (see README for
#' full citation), run the modelling with the JAGS sampler. This model
#' uses the calculated attenuation-contamination matrix to adjust
#' the various covariates needed in the model. The function accepts
#' the computed pre-model, the attenuation-contamination coefficient,,
#' a model string, and standard deviations. There is an optional
#' argument for also fitting the univariate model to run further
#' comparisons between the naive, univariate, and multivariate models
#' in later steps.
#'
#' @param formula character The model formula
#' @param data data.frame The data to model
#' @param columns vector The columns that are relevant
#' @param a_c_matrix matrix The attenuation-contamination matrix
#' @param n_burn numeric Number of draws to burn at the start, default = 1000
#' @param n_thin numeric Thinning factor for the draws, default = 1
#' @param n_steps numeric The total number of draws to run, default = 10,000
#' @param seed numeric The random seed to set
#' @param b0 numeric The prior mean of the beta
#' @param capital_b_0 numeric The precision of the beta
#' @param sampler character The sampler to use for the model
#' @param c0 numeric Shape parameter for gamma
#' @param d0 numeric Scale parameter for gamm
#' @param univariate bool Whether or not to run the univariate model,
#'                       default = TRUE
#' @param sds list If you are running the univariate model, the listing of
#'                 standard deviations needs to be applied
#' @param variances list If you are running the univariate model, the listing of
#'                 variances need to be applied
#' @return List with fitted models ready for further analysis
#' @export
#' @import MCMCpack
#'
#' @examples 
#' columns <- c("fruit", "veg", "tobacco")
#' fruit_v_coef <- generate_coefficient(100, 0.3, 0.8, 0.95)
#' veg_v_coef <- generate_coefficient(100, 0.25, 0.75, 0.95)
#' tob_v_coef <- generate_coefficient(100, 0.4, 0.7, 0.95)
#' validity_coefficients <- c(fruit_v_coef, veg_v_coef, tob_v_coef)
#' data <- data.frame(
#'  list(
#'    "BMI" = rnorm(100, mean = 0, sd = 1),
#'    "fruit" = rnorm(100, mean = 0, sd = 1),
#'    "veg" = rnorm(100, mean = 0, sd = 1),
#'    "tobacco" = rnorm(100, mean = 0, sd = 1)
#'  )
#' )
#' output <- acme_model(data, columns)
#' lambda <- attenuation_matrix(
#'   output,
#'   columns,
#'   validity_coefficients,
#' )
#' model_output <- multivariate_model(
#'   "BMI ~ fruit + veg + tobacco",
#'   data = data,
#'   columns = columns,
#'   a_c_matrix = lambda$matrix,
#'   sds = lambda$sds,
#'   variances = lambda$variances,
#'   univariate = TRUE
#' )
multivariate_model <- function(
    formula,
    data,
    columns,
    a_c_matrix,
    n_burn = 1000,
    n_thin = 1,
    n_steps = 10000,
    seed = 42,
    b0 = 0,
    capital_b_0 = 0.000001,
    sampler = "Metropolis",
    c0 = 0.001,
    d0 = 0.001,
    univariate = FALSE,
    sds = NULL,
    variances = NULL) {
  stopifnot(is.character(formula))
  stopifnot(is.data.frame(data))
  stopifnot(is.matrix(a_c_matrix))

  # Check if the correct type
  stopifnot(is.numeric(n_burn))
  stopifnot(is.numeric(n_thin))
  stopifnot(is.numeric(n_steps))
  stopifnot(is.numeric(seed))

  # Check for implausible values
  stopifnot(!(n_burn < 1))
  stopifnot(!(n_thin < 1))
  stopifnot(!(n_thin > 8))
  stopifnot(!(n_steps < 1))

  # Check if univariate arguments make sense
  if (univariate) {
    stopifnot(!is.null(sds))
    stopifnot(!is.null(variances))
  }

  set.seed(seed)

  naive_model <- MCMCpack::MCMCregress(
    formula,
    data = data,
    burnin = n_burn,
    mcmc = n_steps,
    thin = n_thin,
    verbose = 0,
    seed = seed,
    beta.start = NA,
    b0 = b0,
    B0 = capital_b_0,
    sampler = sampler,
    c0 = c0,
    d0 = d0
  )

  n_cols <- length(columns)
  if (univariate) {
    unv_adjust <- list()
    for (i in seq_along(columns)) {
      target <- columns[i]
      temp <- t((
        (sds[[target]])^2 * (t(naive_model[, i + 1])) / variances[[target]]
      ))
      colnames(temp) <- target
      unv_adjust[[i]] <- temp
    }
    beta_adjusted_univ <- data.frame(do.call("cbind", unv_adjust))
  }

  beta_adjusted_triv <- (
    solve(t(a_c_matrix))
    %*% (t(naive_model[, 2:(2 + n_cols - 1)]))
  )
  target_x <- t(beta_adjusted_triv)
  named_list <- list()
  for (i in seq_along(columns)) {
    named_list[[columns[i]]] <- target_x[, i]
  }
  beta_adjusted_triv_named <- data.frame(named_list)
  if (univariate) {
    return(list(
      naive = naive_model,
      univariate = beta_adjusted_univ,
      multivariate = beta_adjusted_triv_named
    ))
  } else {
    return(list(
      naive = naive_model,
      multivariate = beta_adjusted_triv_named
    ))
  }
}
