% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nested_resamples.R
\name{nested_resamples}
\alias{nested_resamples}
\title{Create splits with nested data}
\usage{
nested_resamples(
  data,
  resamples,
  nesting_method = NULL,
  size_action = c("truncate", "recycle", "recycle-random", "combine", "combine-random",
    "combine-end", "error"),
  ...
)
}
\arguments{
\item{data}{A data frame.}

\item{resamples}{An expression, function, formula or string that can
be evaluated to produce an \code{rset} or \code{rsplit} object.}

\item{nesting_method}{A recipe, workflow or \code{NULL}, used to nest \code{data}
if \code{data} is not already nested (see Details).}

\item{size_action}{If different numbers of splits are produced in each
nest, how should sizes be matched? (see Details)}

\item{...}{Extra arguments to pass into \code{resamples}.}
}
\value{
Either an \code{rsplit} object or an \code{rset} object, depending on
\code{resamples}.
}
\description{
Use any rsample split function on nested data, where nests act as
strata. This almost guarantees that every split will contain data from
every nested data frame.
}
\details{
This function breaks down a data frame into smaller, nested data frames.
Resampling is then performed within these nests, and the results are
combined together at the end. This ensures that each split contains
data from every nest. However, this function does not perform any
pooling (unlike \code{\link[rsample:make_strata]{rsample::make_strata()}}), so you may run into issues
if a nest is too small.
}
\section{Nesting Data}{
\code{data} can be nested in several ways:
If \code{nesting_method} is \code{NULL} and \code{data} is grouped (using
\code{\link[dplyr:group_by]{dplyr::group_by()}}), the data will be nested (see \code{\link[tidyr:nest]{tidyr::nest()}}
for how this works).
If \code{data} is not grouped, it is assumed to already be nested, and
\code{nested_resamples} will try to find a column that contains nested data
frames.
If \code{nesting_method} is a workflow or recipe, and the recipe has a step
created using  \code{\link[=step_nest]{step_nest()}}, \code{data} will be nested using the step in
the recipe. This is convenient if you've already created a recipe or
workflow, as it saves a line of code.
}

\section{Resample Evaluation}{
The \code{resamples} argument can take many forms:
\itemize{
\item A function call, such as \code{vfold_cv(v = 5)}. This is similar to the
format of \code{\link[rsample:nested_cv]{rsample::nested_cv()}}.
\item A function, such as \code{rsample::vfold_cv}.
\item A purrr-style anonymous function, which will be converted to a
function using \code{\link[rlang:as_function]{rlang::as_function()}}.
\item A string, which will be evaluated using \code{\link[rlang:exec]{rlang::exec()}}.
}

Every method will be evaluated with \code{data} passed in as the first
argument (with name 'data').
}

\section{Size Matching}{
Before the set of resamples created in each nest can be combined, they
must contain the same number of splits. For most resampling methods,
this will not be an issue. \code{\link[rsample:vfold_cv]{rsample::vfold_cv()}}, for example, reliably
creates the number of splits defined in its 'v' argument. However,
other resampling methods, like \code{\link[rsample:rolling_origin]{rsample::rolling_origin()}}, depend on
the size of their 'data' argument, and therefore may produce different
numbers of resamples when presented with differently sized nests.

The \code{size_action} argument defines many ways of matching the sizes of
resample sets with different numbers of splits. These methods will either try
to reduce the number of splits in each set until each rset is the same
length as the set with the lowest number of splits; or the opposite,
where each rset will have the same number of splits as the largest set.

"truncate", the default, means that all splits beyond the required
length will be removed.

"recycle" means that sets of splits will be extended by repeating
elements until the required length has been reached, mimicking the
process of vector recycling. The advantage of this method is that all
created splits will be preserved.

"recycle-random" is a similar process to recycling, but splits will be
copied at random to spaces in the output, which may be important if
the order of resamples matters. This process is not completely random,
and the program makes sure that every split is copied roughly the same
number of times.

"combine" gets rid of excess splits by combining them with previous ones.
This means the training and testing rows are merged into one split.
Combining is done systematically: if a set of splits needs to be
compacted down to a set of 5, the first split is combined with the
sixth split, then the eleventh, then the sixteenth, etc. This approach
is not recommended, since it is not clear what the benefit of a
combined split is.

"combine-random" combines each split with a random set of other splits,
instead of the systematic process described in the previous method.
Once again, this process is not actually random, and each split will
be combined with roughly the same number of other splits.

"combine-end" combines every excess split with the last non-excess
split.

"error" throws an error if each nest does not produce the same number
of splits.
}

\examples{
\dontshow{if (rlang::is_installed("workflows")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

library(tidyr)
library(recipes)
library(workflows)
library(rsample)
library(dplyr)

nested_data <- example_nested_data \%>\%
  nest(data = -id)

grouped_data <- example_nested_data \%>\%
  group_by(id)

recipe <- recipe(example_nested_data, z ~ .) \%>\%
  step_nest(id)

wf <- workflow() \%>\%
  add_recipe(recipe)

nested_resamples(nested_data, vfold_cv())

nested_resamples(
  group_by(example_nested_data, id),
  ~ initial_split(.)
)

nested_resamples(example_nested_data, bootstraps,
  times = 25, nesting_method = wf
)

# nested nested resamples

nested_resamples(nested_data, nested_cv(
  vfold_cv(),
  bootstraps()
))
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[rsample:initial_split]{rsample::initial_split()}} for an example of the strata
argument.
}
