% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/measure_diffusion.R
\name{measure_diffusion_net}
\alias{measure_diffusion_net}
\alias{net_transmissibility}
\alias{net_recovery}
\alias{net_reproduction}
\alias{net_immunity}
\title{Measures of network diffusion}
\usage{
net_transmissibility(.data)

net_recovery(.data, censor = TRUE)

net_reproduction(.data)

net_immunity(.data, normalized = TRUE)
}
\arguments{
\item{.data}{Network data with nodal changes,
as created by \code{play_diffusion()},
or a valid network diffusion model,
as created by \code{as_diffusion()}.}

\item{censor}{Where some nodes have not yet recovered by the end
of the simulation, right censored values can be replaced by the number of steps.
By default TRUE.
Note that this will likely still underestimate recovery.}

\item{normalized}{Logical scalar, whether the centrality scores are normalized.
Different denominators are used depending on whether the object is one-mode or two-mode,
the type of centrality, and other arguments.}
}
\description{
These functions allow measurement of various features of
a diffusion process at the network level:
\itemize{
\item \code{net_transmissibility()} measures the average transmissibility observed
in a diffusion simulation, or the number of new infections over
the number of susceptible nodes.
\item \code{net_recovery()} measures the average number of time steps
nodes remain infected once they become infected.
\item \code{net_reproduction()} measures the observed reproductive number
in a diffusion simulation as the network's transmissibility over
the network's average infection length.
\item \code{net_immunity()} measures the proportion of nodes that would need
to be protected through vaccination, isolation, or recovery for herd immunity to be reached.
}
}
\section{Transmissibility}{

\code{net_transmissibility()} measures how many directly susceptible nodes
each infected node will infect in each time period, on average.
That is:
\deqn{T = \frac{1}{n}\sum_{j=1}^n \frac{i_{j}}{s_{j}}}
where \eqn{i} is the number of new infections in each time period, \eqn{j \in n},
and \eqn{s} is the number of nodes that could have been infected in that time period
(note that \eqn{s \neq S}, or
the number of nodes that are susceptible in the population).
\eqn{T} can be interpreted as the proportion of susceptible nodes that are
infected at each time period.
}

\section{Recovery time}{

\code{net_recovery()} measures the average number of time steps that
nodes in a network remain infected.
Note that in a diffusion model without recovery, average infection length
will be infinite.
This will also be the case where there is right censoring.
The longer nodes remain infected, the longer they can infect others.
}

\section{Reproduction number}{

\code{net_reproduction()} measures a given diffusion's reproductive number.
Here it is calculated as:
\deqn{R = \min\left(\frac{T}{1/L}, \bar{k}\right)}
where \eqn{T} is the observed transmissibility in a diffusion
and \eqn{L} is the observed recovery length in a diffusion.
Since \eqn{L} can be infinite where there is no recovery
or there is right censoring,
and since network structure places an upper limit on how many
nodes each node may further infect (their degree),
this function returns the minimum of \eqn{R_0}
and the network's average degree.

Interpretation of the reproduction number is oriented around R = 1.
Where \eqn{R > 1}, the 'disease' will 'infect' more and more
nodes in the network.
Where \eqn{R < 1}, the 'disease' will not sustain itself and eventually
die out.
Where \eqn{R = 1}, the 'disease' will continue as endemic,
if conditions allow.
}

\section{Herd immunity}{

\code{net_immunity()} estimates the proportion of a network
that need to be protected from infection for herd immunity
to be achieved.
This is known as the Herd Immunity Threshold or HIT:
\deqn{1 - \frac{1}{R}}
where \eqn{R} is the reproduction number from \code{net_reproduction()}.
The HIT indicates the threshold at which
the reduction of susceptible members of the network means
that infections will no longer keep increasing.
Note that there may still be more infections after this threshold has been reached,
but there should be fewer and fewer.
These excess infections are called the \emph{overshoot}.
This function does \emph{not} take into account the structure
of the network, instead using the average degree.

Interpretation is quite straightforward.
A HIT or immunity score of 0.75 would mean that 75\% of the nodes in the network
would need to be vaccinated or otherwise protected to achieve herd immunity.
To identify how many nodes this would be, multiply this proportion with the number
of nodes in the network.
}

\examples{
  smeg <- generate_smallworld(15, 0.025)
  smeg_diff <- play_diffusion(smeg, recovery = 0.2)
  plot(smeg_diff)
  # To calculate the average transmissibility for a given diffusion model
  net_transmissibility(smeg_diff)
  # To calculate the average infection length for a given diffusion model
  net_recovery(smeg_diff)
  # To calculate the reproduction number for a given diffusion model
  net_reproduction(smeg_diff)
  # Calculating the proportion required to achieve herd immunity
  net_immunity(smeg_diff)
  # To find the number of nodes to be vaccinated
  net_immunity(smeg_diff, normalized = FALSE)
}
\references{
\subsection{On epidemiological models}{

Kermack, William O., and Anderson Gray McKendrick. 1927.
"A contribution to the mathematical theory of epidemics".
\emph{Proc. R. Soc. London A} 115: 700-721.
\doi{10.1098/rspa.1927.0118}
}

\subsection{On the basic reproduction number}{

Diekmann, Odo, Hans J.A.P. Heesterbeek, and Hans J.A.J. Metz. 1990.
"On the definition and the computation of the basic reproduction ratio R0 in models for infectious diseases in heterogeneous populations".
\emph{Journal of Mathematical Biology}, 28(4): 365–82.
\doi{10.1007/BF00178324}

Kenah, Eben, and James M. Robins. 2007.
"Second look at the spread of epidemics on networks".
\emph{Physical Review E}, 76(3 Pt 2): 036113.
\doi{10.1103/PhysRevE.76.036113}
}

\subsection{On herd immunity}{

Garnett, G.P. 2005.
"Role of herd immunity in determining the effect of vaccines against sexually transmitted disease".
\emph{The Journal of Infectious Diseases}, 191(1): S97-106.
\doi{10.1086/425271}
}
}
\seealso{
Other measures: 
\code{\link{measure_attributes}},
\code{\link{measure_central_between}},
\code{\link{measure_central_close}},
\code{\link{measure_central_degree}},
\code{\link{measure_central_eigen}},
\code{\link{measure_closure}},
\code{\link{measure_cohesion}},
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_node}},
\code{\link{measure_features}},
\code{\link{measure_heterogeneity}},
\code{\link{measure_hierarchy}},
\code{\link{measure_holes}},
\code{\link{measure_periods}},
\code{\link{measure_properties}},
\code{\link{member_diffusion}}

Other diffusion: 
\code{\link{make_play}},
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_node}},
\code{\link{member_diffusion}}
}
\concept{diffusion}
\concept{measures}
