% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_power_npe.R
\name{gs_power_npe}
\alias{gs_power_npe}
\title{Group sequential bound computation with non-constant effect}
\usage{
gs_power_npe(
  theta = 0.1,
  theta0 = NULL,
  theta1 = NULL,
  info = 1,
  info0 = NULL,
  info1 = NULL,
  info_scale = c("h0_h1_info", "h0_info", "h1_info"),
  upper = gs_b,
  upar = qnorm(0.975),
  lower = gs_b,
  lpar = -Inf,
  test_upper = TRUE,
  test_lower = TRUE,
  binding = FALSE,
  r = 18,
  tol = 1e-06
)
}
\arguments{
\item{theta}{Natural parameter for group sequential design representing
expected incremental drift at all analyses; used for power calculation.}

\item{theta0}{Natural parameter for null hypothesis,
if needed for upper bound computation.}

\item{theta1}{Natural parameter for alternate hypothesis,
if needed for lower bound computation.}

\item{info}{Statistical information at all analyses for input \code{theta}.}

\item{info0}{Statistical information under null hypothesis,
if different than \code{info};
impacts null hypothesis bound calculation.}

\item{info1}{Statistical information under hypothesis used for
futility bound calculation if different from
\code{info}; impacts futility hypothesis bound calculation.}

\item{info_scale}{Information scale for calculation. Options are:
\itemize{
\item \code{"h0_h1_info"} (default): variance under both null and alternative hypotheses is used.
\item \code{"h0_info"}: variance under null hypothesis is used.
\item \code{"h1_info"}: variance under alternative hypothesis is used.
}}

\item{upper}{Function to compute upper bound.}

\item{upar}{Parameters passed to \code{upper}.}

\item{lower}{Function to compare lower bound.}

\item{lpar}{parameters passed to \code{lower}.}

\item{test_upper}{Indicator of which analyses should include
an upper (efficacy) bound;
single value of \code{TRUE} (default) indicates all analyses; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have an efficacy bound.}

\item{test_lower}{Indicator of which analyses should include a lower bound;
single value of \code{TRUE} (default) indicates all analyses;
single value of \code{FALSE} indicated no lower bound; otherwise,
a logical vector of the same length as \code{info} should
indicate which analyses will have a lower bound.}

\item{binding}{Indicator of whether futility bound is binding;
default of \code{FALSE} is recommended.}

\item{r}{Integer value controlling grid for numerical integration as in
Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally, \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for boundary convergence (on Z-scale).}
}
\value{
A tibble with columns as analysis index, bounds, z,
crossing probability, theta (standardized treatment effect),
theta1 (standardized treatment effect under alternative hypothesis),
information fraction, and statistical information.
}
\description{
Derives group sequential bounds and boundary crossing probabilities for a design.
It allows a non-constant treatment effect over time,
but also can be applied for the usual homogeneous effect size designs.
It requires treatment effect and statistical information at each analysis
as well as a method of deriving bounds, such as spending.
The routine enables two things not available in the gsDesign package:
\enumerate{
\item non-constant effect, 2) more flexibility in boundary selection.
For many applications, the non-proportional-hazards design function
\code{gs_design_nph()} will be used; it calls this function.
Initial bound types supported are 1) spending bounds,
\item fixed bounds, and 3) Haybittle-Peto-like bounds.
The requirement is to have a boundary update method that can
each bound without knowledge of future bounds.
As an example, bounds based on conditional power that require
knowledge of all future bounds are not supported by this routine;
a more limited conditional power method will be demonstrated.
Boundary family designs Wang-Tsiatis designs including the
original (non-spending-function-based) O'Brien-Fleming and Pocock designs
are not supported by \code{gs_power_npe()}.
}
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Extract the length of input info as the number of interim analysis.
   \item Validate if input info0 is NULL, so set it equal to info.
   \item Validate if the length of inputs info and info0 are the same.
   \item Validate if input theta is a scalar, so replicate
   the value for all k interim analysis.
   \item Validate if input theta1 is NULL and if it is a scalar.
   If it is NULL, set it equal to input theta. If it is a scalar,
   replicate the value for all k interim analysis.
   \item Validate if input test_upper is a scalar,
   so replicate the value for all k interim analysis.
   \item Validate if input test_lower is a scalar,
   so replicate the value for all k interim analysis.
   \item Define vector a to be -Inf with
   length equal to the number of interim analysis.
   \item Define vector b to be Inf with
   length equal to the number of interim analysis.
   \item Define hgm1_0 and hgm1 to be NULL.
   \item Define upper_prob and lower_prob to be
   vectors of NA with length of the number of interim analysis.
   \item Update lower and upper bounds using \code{gs_b()}.
   \item If there are no interim analysis, compute probabilities
   of crossing upper and lower bounds
   using \code{h1()}.
   \item Compute cross upper and lower bound probabilities
   using \code{hupdate()} and \code{h1()}.
   \item Return a tibble of analysis number, bound, z-values,
   probability of crossing bounds,
   theta, theta1, info, and info0.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
library(gsDesign)
library(gsDesign2)
library(dplyr)

# Default (single analysis; Type I error controlled)
gs_power_npe(theta = 0) \%>\% filter(bound == "upper")

# Fixed bound
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_b,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF)$upper$bound,
  lower = gs_b,
  lpar = c(-1, 0, 0)
)

# Same fixed efficacy bounds, no futility bound (i.e., non-binding bound), null hypothesis
gs_power_npe(
  theta = rep(0, 3),
  info = (1:3) * 40,
  upar = gsDesign::gsDesign(k = 3, sfu = gsDesign::sfLDOF)$upper$bound,
  lpar = rep(-Inf, 3)
) \%>\%
  filter(bound == "upper")

# Fixed bound with futility only at analysis 1; efficacy only at analyses 2, 3
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_b,
  upar = c(Inf, 3, 2),
  lower = gs_b,
  lpar = c(qnorm(.1), -Inf, -Inf)
)

# Spending function bounds
# Lower spending based on non-zero effect
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# Same bounds, but power under different theta
gs_power_npe(
  theta = c(.15, .25, .35),
  info = (1:3) * 40,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfHSD, total_spend = 0.1, param = -1, timing = NULL)
)

# Two-sided symmetric spend, O'Brien-Fleming spending
# Typically, 2-sided bounds are binding
x <- gs_power_npe(
  theta = rep(0, 3),
  info = (1:3) * 40,
  binding = TRUE,
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL)
)

# Re-use these bounds under alternate hypothesis
# Always use binding = TRUE for power calculations
gs_power_npe(
  theta = c(.1, .2, .3),
  info = (1:3) * 40,
  binding = TRUE,
  upar = (x \%>\% filter(bound == "upper"))$z,
  lpar = -(x \%>\% filter(bound == "upper"))$z
)

# Different values of `r` and `tol` lead to different numerical accuracy
# Larger `r` and smaller `tol` give better accuracy, but leads to slow computation
n_analysis <- 5
gs_power_npe(
  theta = rep(0.1, n_analysis),
  theta0 = NULL,
  theta1 = NULL,
  info = 1:n_analysis,
  info0 = 1:n_analysis,
  info1 = NULL,
  info_scale = "h0_info",
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL),
  lower = gs_b,
  lpar = -rep(Inf, n_analysis),
  test_upper = TRUE,
  test_lower = FALSE,
  binding = FALSE,
  # Try different combinations of (r, tol) with
  # r in 6, 18, 24, 30, 35, 40, 50, 60, 70, 80, 90, 100
  # tol in 1e-6, 1e-12
  r = 6,
  tol = 1e-6
)
}
