% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom.R
\name{g_util}
\alias{g_util}
\alias{g_make_valid}
\alias{g_set_3D}
\alias{g_set_measured}
\alias{g_swap_xy}
\title{Geometry utility functions operating on WKB or WKT}
\usage{
g_make_valid(
  geom,
  method = "LINEWORK",
  keep_collapsed = FALSE,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB",
  quiet = FALSE
)

g_set_3D(
  geom,
  is_3d,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB",
  quiet = FALSE
)

g_set_measured(
  geom,
  is_measured,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB",
  quiet = FALSE
)

g_swap_xy(
  geom,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB",
  quiet = FALSE
)
}
\arguments{
\item{geom}{Either a raw vector of WKB or list of raw vectors, or a
character vector containing one or more WKT strings.}

\item{method}{Character string. One of \code{"LINEWORK"} (the default) or
\code{"STRUCTURE"} (requires GEOS >= 3.10 and GDAL >= 3.4). See Details.}

\item{keep_collapsed}{Logical value, applies only to the STRUCTURE method.
Defaults to \code{FALSE}. See Details.}

\item{as_wkb}{Logical value, \code{TRUE} to return the output geometry in WKB
format (the default), or \code{FALSE} to return as WKT.}

\item{as_iso}{Logical value, \code{TRUE} to export as ISO WKB/WKT (ISO 13249
SQL/MM Part 3), or \code{FALSE} (the default) to export as "Extended WKB/WKT".}

\item{byte_order}{Character string specifying the byte order when output is
WKB. One of \code{"LSB"} (the default) or \code{"MSB"} (uncommon).}

\item{quiet}{Logical value, \code{TRUE} to suppress warnings. Defaults to \code{FALSE}.}

\item{is_3d}{Logical value, \code{TRUE} if the input geometries should have a Z
dimension, or \code{FALSE} to remove the Z dimension.}

\item{is_measured}{Logical value, \code{TRUE} if the input geometries should have
a M dimension, or \code{FALSE} to remove the M dimension.}
}
\value{
A geometry as WKB raw vector or WKT string, or a list/character vector of
geometries as WKB/WKT with length equal to \code{length(geom)}. \code{NULL} is returned
with a warning if WKB input cannot be converted into an OGR geometry object,
or if an error occurs in the call to the underlying OGR API.
}
\description{
These functions operate on input geometries in OGC WKB or WKT format to
perform various manipulations for utility purposes.
}
\details{
These functions use the GEOS library via GDAL headers.

\code{g_make_valid()} attempts to make an invalid geometry valid without losing
vertices. Already-valid geometries are cloned without further intervention.
Wrapper of \code{OGR_G_MakeValid()}/\code{OGR_G_MakeValidEx()} in the GDAL API.
Requires the GEOS >= 3.8 library, check it for the definition of the
geometry operation. If GDAL is built without GEOS >= 3.8, this function
will return a clone of the input geometry if it is valid, or \code{NULL}
(\code{as_wkb = TRUE}) / \code{NA} (\code{as_wkb = FALSE}) if it is invalid.
\itemize{
\item \code{"LINEWORK"} is the default method, which combines all rings into a set
of noded lines and then extracts valid polygons from that linework
(requires GEOS >= 3.10 and GDAL >= 3.4). The \code{"STRUCTURE"} method first
makes all rings valid, then merges shells and subtracts holes from shells to
generate a valid result. Assumes that holes and shells are correctly
categorized.
\item \code{keep_collapsed} only applies to the \code{"STRUCTURE"} method:
\itemize{
\item \code{FALSE} (the default): collapses are converted to empty geometries
\item \code{TRUE}: collapses are converted to a valid geometry of lower dimension
}
}

\code{g_set_3D()} adds or removes the explicit Z coordinate dimension. Removing
the Z coordinate dimension of a geometry will remove any existing Z values.
Adding the Z dimension to a geometry collection, a compound curve, a
polygon, etc. will affect the children geometries.
Wrapper of \code{OGR_G_Set3D()} in the GDAL API.

\code{g_set_measured()} adds or removes the explicit M coordinate dimension.
Removing the M coordinate dimension of a geometry will remove any existing M
values. Adding the M dimension to a geometry collection, a compound curve, a
polygon, etc. will affect the children geometries.
Wrapper of \code{OGR_G_SetMeasured()} in the GDAL API.

\code{g_swap_xy()} swaps x and y coordinates of the input geometry.
Wrapper of \code{OGR_G_SwapXY()} in the GDAL API.
}
\examples{
# g_make_valid() requires GEOS >= 3.8, otherwise is only a validity test
geos_version()

# valid
wkt <- "POINT (0 0)"
g_make_valid(wkt, as_wkb = FALSE)

# invalid to valid
wkt <- "POLYGON ((0 0,10 10,0 10,10 0,0 0))"
g_make_valid(wkt, as_wkb = FALSE)

# invalid - error
wkt <- "LINESTRING (0 0)"
g_make_valid(wkt)  # NULL

pt_xyzm <- g_create("POINT", c(1, 9, 100, 2000))

g_wk2wk(pt_xyzm, as_iso = TRUE)

g_set_3D(pt_xyzm, is_3d = FALSE) |> g_wk2wk(as_iso = TRUE)

g_set_measured(pt_xyzm, is_measured = FALSE) |> g_wk2wk(as_iso = TRUE)

g <- "GEOMETRYCOLLECTION(POINT(1 2),
                         LINESTRING(1 2,2 3),
                         POLYGON((0 0,0 1,1 1,0 0)))"

g_swap_xy(g, as_wkb = FALSE)
}
\seealso{
\code{\link[=g_is_valid]{g_is_valid()}}, \code{\link[=g_is_3D]{g_is_3D()}}, \code{\link[=g_is_measured]{g_is_measured()}}
}
