\name{array_utils}
\alias{g3_array_agg}
\alias{g3_array_combine}
\alias{g3_array_plot}
\concept{G3 utilities}

\title{Gadget3 array-handling utilities}
\description{
  Tools to make munging array reports easier
}

\usage{
g3_array_agg(
        ar,
        margins = NULL,
        agg = c(
            "sum",
            "length_mean", "length_sd",
            "predator_length_mean", "predator_length_sd" ),
        opt_time_split = !("time" \%in\% margins || "time" \%in\% ...names()),
        opt_length_midlen = FALSE,
        ... )

g3_array_combine(
        arrays,
        agg = sum,
        init_val = 0 )

g3_array_plot(
        ar,
        legend = "topright" )
}

\arguments{
  \item{ar}{Input array, e.g. \code{dstart_fish__num} from a model report}
  \item{arrays}{List of input arrays, can be a nested list as generated by \var{cons} in \code{\link{g3_quota_assess}}}
  \item{margins}{dimension names to include in the final array, e.g. \code{c("age", "year")} to generate a report by-year & age. If NULL, no aggregation is done}
  \item{agg}{Function or character. Function to use when aggregating, or name of one of the built-in functions}
  \item{init_val}{The initial value to use when combining arrays}
  \item{opt_time_split}{Boolean, should we split up "time" into separate "year" & "step" dimensions?}
  \item{opt_length_midlen}{Boolean, should we convert "length"}
  \item{legend}{Location of legend, passed to \code{\link{legend}}'s \var{x} parameter}
  \item{...}{Filters to apply to any dimension, including "year" / "step" if \var{opt_time_split} is TRUE. e.g. \code{length = 40, age = 5, step = 1}}
}

\details{
  \code{g3_array_agg} allows you to both filter & aggregate an array at the same time.

  Specifying a filter in \var{...} is simplfied in comparison to a regular R subset:

  \enumerate{
    \item{You can give the dimensions in any order}
    \item{Values are always interpreted, \code{age = 3} will be interpreted as \code{"age3"}, not the third age.}
  }

  For particular dimensions we have extra helpers:

  \describe{
    \item{age}{Numeric ages e.g. \code{age = 5} are converted to "age5", as generated by gadget3}
    \item{length}{Numeric lengths will pick a value within groups, e.g. with lengths "10:20", "20:30", \code{length = 15} will pick the smaller lengthgroup}
  }

  \code{g3_array_combine} generates the union of 2 disjoint arrays,
  so you can combine aggregated output from an immature and mature stock for example.

  \code{g3_array_plot} will plot the contents of an array, for arrays with 2 dimensions or less.
}

\value{
  An array, filtered by \var{...} and aggregated by \var{margins}
}

\examples{
# Generate an array to test with
dn <- list(
    length = c("50:60", "60:70", "70:Inf"),
    age = paste0("age", 0:5),
    time = paste0(rep(1990:1996, each = 2), c("-01", "-02")) )
ar <- array(
    seq_len(prod(sapply(dn, length))),
    dim = sapply(dn, length),
    dimnames = dn)
ar[,,"1994-02", drop = FALSE]
g3_array_plot(ar[,,"1994-02"])
g3_array_plot(ar["50:60","age3",])

# Generate by-year report for ages 2..4
g3_array_agg(ar, c('age', 'year'), age = 2:4)

# ...for only step 1
g3_array_agg(ar, c('age', 'year'), age = 2:4, step = 1)

# Report on smallest length group, for each timestep
g3_array_agg(ar, c('length', 'time'), length = 55)
# Use midlen as the dimension name
g3_array_agg(ar, c('length', 'time'), length = 55, opt_length_midlen = TRUE)

# Combine 2 arrays with disjoint age ranges into one list
g3_array_combine(list(
    g3_array_agg(ar, c('age', 'year'), age = 2:4),
    g3_array_agg(ar / 1000, c('age', 'year'), age = 3:5) ))
}
