% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/writeRaster.r
\name{writeRaster,GRaster,character-method}
\alias{writeRaster,GRaster,character-method}
\alias{writeRaster}
\alias{writeRaster,missing,missing-method}
\title{Save a GRaster to disk}
\usage{
\S4method{writeRaster}{GRaster,character}(
  x,
  filename,
  overwrite = FALSE,
  datatype = NULL,
  byLayer = FALSE,
  names = TRUE,
  levelsExt = NULL,
  compress = "LZW",
  warn = TRUE,
  ...
)

\S4method{writeRaster}{missing,missing}(x, filename)
}
\arguments{
\item{x}{A \code{GRaster} or missing: If missing, a table of supported file types is reported.}

\item{filename}{Character: Path and file name.}

\item{overwrite}{Logical: If \code{FALSE} (default), do not save over existing file(s).}

\item{datatype}{\code{NULL} (default) or character: The datatype of the values stored in non-ASCII rasters. If \code{NULL}, this will be ascertained from the raster, and the function usually does a good job at it. However, you can force it manually, but note that in some cases, trying to save a \code{GRaster} using an inappropriate \code{datatype} for its values can result in an error or in the function exiting without an error but also without having written the raster to disk. The argument can take any of those shown below under the first four columns, but whatever is used, it will be converted to the \strong{GDAL} version.\tabular{lllll}{
   \strong{fasterRaster} \tab \strong{terra} \tab \strong{GRASS} \tab \strong{GDAL} \tab \strong{Values} \cr
   \code{integer} \tab \code{INT1U} \tab \code{CELL} \tab \code{Byte} \tab Integer values from 0 to 255 \cr
   \code{integer} \tab \code{INT2U} \tab \code{CELL} \tab \code{UInt16} \tab Integer values from 0 to 65,534 \cr
   \code{integer} \tab \code{INT2S} \tab \code{CELL} \tab \code{Int16} \tab Integer values from -32,767 to -32,767 \cr
   \code{integer} \tab \code{INT4S} \tab \code{CELL} \tab \code{Int32} \tab Integer values from -2,147,483,647 to 2,147,483,647 \cr
   \code{float} \tab \code{FLT4S} \tab \code{FCELL} \tab \code{Float32} \tab Values from -3.4E+38 to 3.4E+38, including decimal values \cr
   \code{double} \tab \code{FLT8S} \tab \code{DCELL} \tab \code{Float64} \tab Values from -1.79E+308 to 1.79E+308, including decimal values \cr
   \code{factor} \tab \code{INT}* \tab \code{CELL} \tab \verb{INT*} \tab Integer values corresponding to categories \cr
}


\code{*} Depends on the integers (signed/unsigned, range of values). Categorical rasters will have an associated file saved with them that has category values and labels. The file name will be the same as the raster's file name, but end with the extension given by \code{levelsExt} (\code{.csv} by default).}

\item{byLayer}{Logical: If \code{FALSE} (default), multi-layer rasters will be saved in one file. If \code{TRUE}, the each layer will be saved in a separate file. The filename from \code{filename} will be amended so that it ends with \verb{_<name>} (then the file extension), where \verb{<name>} is give by \code{\link[=names]{names()}}. Note that if any characters in raster names will not work in a file name, then the function will fail (e.g., a backslash or question mark).}

\item{names}{Logical: If \code{TRUE} (default), save a file with raster layer names. The file will have the same name as the raster file but end with "\verb{_names.csv}". Currently, the \code{\link[=names]{names()}} attribute of rasters cannot be saved in the raster, which can create confusion when multi-layered rasters are saved. Turning on this option will save the ancillary file with layer names. If it exists, this file will be read by \code{\link[=fast]{fast()}} so layer names are assigned when the raster is read by that function. The absence of a "names" file will not create any issues with this function or \code{\link[=fast]{fast()}}, other than not having the metadata on layer names.}

\item{levelsExt}{Character, logical, or \code{NULL} (default): Name of the file extension for the "levels" file that accompanies a categorical \code{GRaster}. When saving categorical rasters, the raster file is accompanied with a "levels" file that contain information on the levels of the raster. This file is the same as \code{filename}, except it has a different extension. Valid values depend on how many raster layers are saved at a time (case is ignored):
\itemize{
\item DefaultOne raster layer: \code{".csv"}
\item Two or more layers, with at least one categorical raster: \code{".rds"}, \code{".rda"}, \code{".rdat"}, \code{".rdata"}
\item Any: \code{NULL} or \code{TRUE} automatically selects either \code{".csv"} (one raster layer) or \verb{".rds} (two or more)
\item Any: \code{FALSE} disables saving of a levels file.
}}

\item{compress}{Character: Type of compression to use for GeoTIFF files:
\itemize{
\item \code{"LZW"} (default)
\item \code{"DEFLATE"}
\item \code{"PACKBITS"}
\item \code{"LZMA"}
\item \code{NULL}: No compression is used, but the file can still be reduced in size by using zip, gzip, or other compressions.
}}

\item{warn}{Logical: If \code{TRUE} (default), display a warning if the \code{datatype} argument does not match the value given by \code{datatype(x, "GDAL")}, or if the \code{fileExt} argument will not work with the given raster and so has been automatically changed.}

\item{...}{Additional arguments. These can include:
\itemize{
\item \code{bigTiff}: Logical: If \code{TRUE}, and the file format is a GeoTIFF and would be larger than 4 GB (regardless of compression), then the file will be saved in BIGTIFF format.
\item \code{format}: Character, indicating file format. This is usually ascertained from the file extension, but in case this fails, it can be stated explicitly. When using other formats, you may have to specify the \code{createopts} argument, too (see help page for \strong{GRASS} module \code{r.out.gdal}). Two common formats include:
\itemize{
\item \code{"GTiff"} (default): GeoTIFF \code{filename} ends in \code{.tif}.
\item \code{"ASC"}: ASCII \code{filename} ends in \code{.asc}
}
\item Additional arguments to send to \strong{GRASS} modules \code{r.out.gdal} and \code{r.out.ascii}.
\item \code{precision}: Numeric: For ASCII files, you may need to state the number of significant digits. 32-bit values have 7 digits and 64-bit values have 16. So in these cases the argument would be \code{precision=7} or \code{precision=16}.
}}
}
\value{
A \code{GRaster} (invisibly). A raster is also saved to disk.
}
\description{
This function saves a \code{GRaster} to disk directly from a \strong{GRASS} session. It is faster than using \code{\link[=rast]{rast()}}, then saving the output of that to disk (because \code{rast()} actually save the raster to disk, anyway).

The function will attempt to ascertain the file type to be ascertained from the file extension, but you can specify the format using the \code{format} argument (see entry for \code{...}). You can see a list of supported formats by simply using this function with no arguments, as in \code{writeRaster()}, or by consulting the online help page for the \strong{GRASS} module \code{r.out.gdal} (see \code{grassHelp("r.out.gdal")}). Only the \code{GeoTIFF} file format is guaranteed to work for multi-layered rasters.

The function will attempt to optimize the \code{datatype} argument, but this can take a long time. You can speed this up by setting \code{datatype} manually. Note that if you are saving a "stack" of \code{GRaster}s with different \code{datatype}s, the one with the highest information density will be used (e.g., low-bit integer < high-bit integer < floating-point < double-floating point). This can make rasters with lower datatypes much larger on disk. In these cases, it make be best to save rasters with similar \code{datatype}s together.
}
\examples{
if (grassStarted()) {

# Setup
library(terra)

# Example data
madElev <- fastData("madElev")
madChelsa <- fastData("madChelsa")

### What raster formats can we attempt to write?
writeRaster()

### Save GRaster to disk (using temporary file)
elev <- fast(madElev)
filename <- tempfile(fileext = ".tif")
writeRaster(elev, filename)

# Load raster from disk
elev2 <- fast(filename)
elev2

### Save multi-layer GRaster to disk in one file (using temporary file)
chelsa <- fast(madChelsa)
filename <- tempfile(fileext = ".tif")
writeRaster(chelsa, filename)

# Load raster from disk
chelsa2 <- fast(filename)
chelsa2

### Save multi-layer GRaster to disk layer-by-layer (using temporary file)
chelsa <- fast(madChelsa)
filename <- tempfile(fileext = ".tif")
writeRaster(chelsa, filename, byLayer = TRUE)

# Load one of the rasters from disk
filename2 <- sub(filename, pattern = ".tif", replacement = "_bio1.tif")
chelsaBio1 <- fast(filename2)
chelsaBio1

}
}
\seealso{
\code{\link[terra:writeRaster]{terra::writeRaster()}}, \strong{GRASS} module \code{r.out.gdal} (see \code{grassHelp("r.out.gdal")})
}
