% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prep_static_ggspectro.R
\name{prep_static_ggspectro}
\alias{prep_static_ggspectro}
\alias{prepStaticSpec}
\alias{prepStaticGGspec}
\title{Generate ggplot2-based spectrogram(s), which can be passed to paged_spectro}
\usage{
prep_static_ggspectro(
  soundFile,
  destFolder = NULL,
  outFilename = NULL,
  savePNG = FALSE,
  colPal = "inferno",
  crop = NULL,
  bg = NULL,
  filter = NULL,
  xLim = NULL,
  yLim = c(0, 10),
  title = NULL,
  plotLegend = FALSE,
  onlyPlotSpec = TRUE,
  ampTrans = 1,
  resampleRate = 15000,
  min_dB = -30,
  wl = 512,
  ovlp = 90,
  wn = "blackman",
  specWidth = 9,
  specHeight = 3,
  colbins = 30,
  ampThresh = 0,
  bgFlood = FALSE,
  fontAndAxisCol = NULL,
  optim = NULL,
  ...
)
}
\arguments{
\item{soundFile}{should work with URLs, full and relative paths; handles .mp3 and .wav}

\item{destFolder}{path to directory to save output. Needs to be like "figures/spectrograms/" to be relative to working directory. Default=parent folder of soundFile. Specify "wd" to output to the working directory, gotten from [get_wd()]}

\item{outFilename}{name for output PNG. default=NULL will use input name in output filename.}

\item{savePNG}{logical; Save static spectrograms as PNGs? They will be exported to destFolder.}

\item{colPal}{color palette; one of "viridis","magma","plasma","inferno","cividis" from the \code{\link[viridis]{viridis}} package OR a 2 value vector (e.g. c("white","black")), defining the start and end of a custom color gradient}

\item{crop}{subset of recording to include; default crop=NULL will use whole file, up to 10 sec; if a number, interpreted as crop first X.X sec; if c(X1,X2), interpreted as trimming out a specific time interval in sec; if crop=FALSE, will not crop at all, even for recordings over 10 sec.}

\item{bg}{background color (defaults to 1st value of chosen palette)}

\item{filter}{apply a bandpass filter? Defaults to none (NULL). Expects 'c(0,2)' where sound from 0 to 2kHz would be filtered out}

\item{xLim}{the time limit (x-axis width) in seconds for all spectrograms; i.e. page width in seconds for multi-page dynamic spectrograms (defaults to WAV file length, unless file duration >5s). To override the 5s limit, put xLim=Inf or specify the desired spectrogram x-axis limit.}

\item{yLim}{the frequency limits (y-axis); default is c(0,10) aka 0-10kHz}

\item{title}{string for title of plots; default=NULL}

\item{plotLegend}{logical; include a legend showing amplitude colors? default=FALSE}

\item{onlyPlotSpec}{logical; do you want to just plot the spec and leave out the legend, axes, and axis labels? default= TRUE}

\item{ampTrans}{amplitude transform for boosting spectrum contrast; default=1 (actual dB values); specify a decimal number for the lambda value of scales::modulus_trans(); 2.5 is a good place to start. (This amplifies your loud values the most, while not increasing background noise much at all)}

\item{resampleRate}{a number in Hz to downsample audio for spectrogram only. This will simplify audio data and speed up generation of spectrogram. Passed to [tuneR::downsample()]. Default=15000 shaves off a few seconds without losing much quality. Put NULL to keep original sample rate for spectrogram. Audiofile will not be resampled for MP4.}

\item{min_dB}{the minimum decibel (quietest sound) to include in the spec; defaults to -30 (-40 would include quieter sounds; -20 would cut out all but very loud sounds)}

\item{wl}{window length for the spectrogram (low values= higher temporal res; high values= higher freq. res). Default 512 is a good tradeoff; human speech would look better at 1024 or higher, giving higher frequency resolution.}

\item{ovlp}{how much overlap (as percent) between sliding windows to generate spec? Default 90 looks good, but takes longer}

\item{wn}{window name (slight tweaks on algorithm that affect smoothness of output) see \code{\link[seewave]{spectro}}}

\item{specWidth}{what width (in inches) would you like to make your PNG output be, if saving a static spec?}

\item{specHeight}{what height (in inches) would you like to make your PNG output be, if saving a static spec?}

\item{colbins}{default 30: increasing can smooth the color contours, but take longer to generate spec}

\item{ampThresh}{amplitude threshold as a percent to cut out of recording (try 5 to start); default= no filtering (high data loss with this; not recommended; play with min_dB  and ampTrans first)}

\item{bgFlood}{do you want the background color to spill into the axis margins? Default=FALSE (i.e. white margins)}

\item{fontAndAxisCol}{the color of legend text if onlyPlotSpec=TRUE (since margins will be white, with black text); if bgFlood=TRUE, this will be the color of axis margins, labels and legend text. If you don't supply this, it will be picked automatically to be white or black based on supplied bg color}

\item{optim}{NULL by default; this is an experimental feature to simplify the dataframe of the FFT-processed waveform used to generate the spectrogram (currently does nothing)}

\item{...}{Other arguments to be passed for rendering the spec (i.e. to seewave::spectro)}
}
\value{
a list with all spectrogram parameters, segmented WAV files (segWavs) and spectrograms spec; importantly, spec is a list of n=number of "pages"/segments; the first page is displayed by default
}
\description{
Can be used to generate single or segmented static spectrograms. Works as standalone,
but the returned object is also intended to feed into \code{\link{paged_spectro}}. Workflow: 1) use
prep_static_ggspectro to crop, filter, segment and tweak all spectrogram parameters; 2) pass these
settings to \code{\link{paged_spectro}} to generate dynamic spectrogram video.
}
\examples{
\dontrun{
require(dynaSpec)
f <- list.files(pattern=".wav", full.names = TRUE, path = system.file(package="dynaSpec"))

# default behavior should be a decent start for good recordings; doesn't save anything, just plots
prep_static_ggspectro(f[1])

# to use with paged_spectro or to do other stuff, you need to assign the
# resulting object, but it will still always plot the first spec
# let's add axes and boost the signal a smidge
femaleBarnSwallow <- prep_static_ggspectro(f[1],destFolder="wd",
onlyPlotSpec = FALSE, bgFlood=TRUE,ampTrans=2)

# feels like we're missing a little bit of the quieter signals; let's lower
# the minimum amplitude threshold a bit
femaleBarnSwallow<-prep_static_ggspectro(f[1],destFolder="wd",
onlyPlotSpec = FALSE, bgFlood=TRUE,ampTrans=2,min_dB=-35)

#now for a male song
maleBarnSwallow<-prep_static_ggspectro(f[2],destFolder="wd",onlyPlotSpec = FALSE,
bgFlood=TRUE)

#Nice, but the trill is fading out; I'm gonna signal boost and lower the min_dB
maleBarnSwallow<-prep_static_ggspectro(f[2],destFolder="wd",onlyPlotSpec = FALSE,
bgFlood=TRUE,ampTrans=2,min_dB=-40)

#much stronger, now let's combine them
(you need the patchwork package to use the / operator to stack plots)
library(patchwork)
(femaleBarnSwallow$spec[[1]]+ggplot2::xlim(0,5)) /
(maleBarnSwallow$spec[[1]]+ggplot2::xlim(0,5))  +
patchwork::plot_annotation(title="Female and Male barn swallow songs",
caption="Female song (top) is much shorter, but similar
complexity to males. See: MR Wilkins et al. (2020) Animal
 Behaviour 168")

# ggplot2::ggsave("M&F_barn_swallow_song_specs.jpeg",width=11,height=7)

# see more examples at https://marce10.github.io/dynaSpec/
}
}
\references{
Araya-Salas M & Wilkins M R. (2020). *dynaSpec: dynamic spectrogram visualizations in R*. R package version 1.0.0.
}
\seealso{
\code{\link{paged_spectro}}
}
\author{
Matthew R Wilkins (\email{matt@galacticpolymath.com})
}
