#' Step 4: Replaying the experiment with optimal parameters
#'
#' @description
#' This function takes the optimal parameters generated by `fit_p` and applies 
#' them back to each subject's data to generate a new column, `Rob_Choose`. 
#' This allows users to analyze whether the reinforcement learning model can 
#' reproduce the original experimental effects observed in the data.
#'
#' @param data [data.frame]  
#'  This data should include the following mandatory columns: 
#'   \itemize{
#'     \item "sub"
#'     \item "time_line" (e.g., "Block", "Trial")
#'     \item "L_choice"
#'     \item "R_choice"
#'     \item "L_reward"
#'     \item "R_reward"
#'     \item "sub_choose"
#'   }
#'
#' @param result [data.frame]  
#'  Output data generated by the `fit_p()` function. Each row represents 
#'   model fit results for a subject.
#'
#' @param model [function]  
#'   A model function to be applied in evaluating the experimental effect.
#'
#' @param model_name [character]  
#'   A character string specifying the name of the model to extract from 
#'   the result.
#'
#' @param param_prefix [character]  
#'   A prefix string used to identify parameter columns in the 
#'   `result` data (e.g., "param_").
#'
#' @param n_trials [integer]  
#'   Number of total trials in the experimental task.
#'
#' @return A list, where each element is a data.frame representing one subject's
#' results. Each data.frame includes the value update history for each option,
#' the learning rate (\code{eta}), discount rate (\code{gamma}), and other 
#' relevant information used in each update.
#' 
#' @export
#'
rpl_e <- function(
  data, 
  result, 
  model,
  model_name,
  param_prefix, 
  n_trials = NA
) {
  # 事前准备. 探测信息
  info <- suppressWarnings(suppressMessages(detect_information(data = data)))

  Subject <- info[["sub_col_name"]]
  id <- info[["all_ids"]]
  
  fit_model <- "fit_model"
  result <- result[result[[fit_model]] == model_name, ]
  
  res <- list()
  
  for (i in 1:length(id)) {
    
    params <- stats::na.omit(unlist(result[i, grep(param_prefix, names(result))]))
    
    binaryRL.env <- new.env()
    binaryRL.env$data <- data
    binaryRL.env$id <- id[i]
    binaryRL.env$n_params <- length(params)
    binaryRL.env$n_trials <- n_trials
    binaryRL.env$mode <- "replay"
    
    
    obj_func <- model
    
    environment(obj_func) <- binaryRL.env
    res[[i]] <- obj_func(params = params)[[1]]
  }
  
  return(res)
}
