#######################################################
#	apc package
#	Bent Nielsen, 17 September 2016, version 1.2.3
#	Data examples
#######################################################
#	Copyright 2014-2016 Bent Nielsen
#	Nuffield College, OX1 1NF, UK
#	bent.nielsen@nuffield.ox.ac.uk
#
#	This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#######################################################

#############################################################################################
#	SPECIFIC DATA SETS
#############################################################################################

###############################
#	JAPANESE BREAST CANCER DATA
###############################
data.Japanese.breast.cancer	<- function()
#	BN, 24 apr 2015	(17 oct 2013)
#	An example with A,P,C effects
#
#	Taken from table I of
#	Clayton, D. and Schifflers, E. (1987b)
#	Models for temperoral variation in cancer rates. II: age-period-cohort models.
#	Statistics in Medicine 6, 469-481.
#
#	Original Table caption:
#	age-specific mortality rates (per 100,000 person-years observation) of breast cancer in Japan,
#	during the period 1955-1979. Numbers of cases on which rates are based are in parentheses
#	(source: WHO mortality data base).  
{	#	data.Japanese.breast.cancer
v.rates		<- c( 0.44, 0.38, 0.46, 0.55, 0.68,
			 	  1.69, 1.69, 1.75, 2.31, 2.52,
				  4.01, 3.90, 4.11, 4.44, 4.80,
				  6.59, 6.57, 6.81, 7.79, 8.27,
				  8.51, 9.61, 9.96,11.68,12.51,
				 10.49,10.80,12.36,14.59,16.56,
				 11.36,11.51,12.98,14.97,17.79,
				 12.03,10.67,12.67,14.46,16.42,
				 12.55,12.03,12.10,13.81,16.46,
				 15.81,13.87,12.65,14.00,15.60,
				 17.97,15.62,15.83,15.71,16.52)
v.cases		<- c(   88,   78,  101,  127,  179,
				   299,  330,  363,  509,  588,
				   596,  680,  798,  923, 1056,
				   874,  962, 1171, 1497, 1716,
				  1022, 1247, 1429, 1987, 2398,
				  1035, 1258, 1560, 2079, 2794,
				   970, 1087, 1446, 1828, 2465,
				   820,  861, 1126, 1549, 1962,
				   678,  738,  878, 1140, 1683,
				   640,  628,  656,  900, 1162,
				   497,  463,  536,  644,  865)				 
col.names	<- paste(as.character(seq(from=1955,length=5,by=5)),"-",
			 		 as.character(seq(from=1955,length=5,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			age1		=25				,
			per1		=1955			,
			unit		=5				,
			label		="Japanese breast cancer"))
}	#	data.Japanese.breast.cancer

##################################
#	ITALIAN BLADDER CANCER DATA
##################################
data.Italian.bladder.cancer	<- function()
#	BN, 24 apr 2015 (17 oct 2013)
#	An example with A,C effects
#
#	Taken from table IV of
#	Clayton, D. and Schifflers, E. (1987a)
#	Models for temperoral variation in cancer rates. I: age-period and age-cohort models.
#	Statistics in Medicine 6, 449-467.
#
#	Original Table caption:
#	age-specific incidence rates (per 100,000 person-years observation) of bladder cancer in
#	Italian males during the period 1955-1979. Numerators are in parentheses
#	(source of data: WHO mortality database).  
{	#	data.Italian.bladder.cancer
v.rates		<- c( 0.03, 0.03, 0.01, 0.04,  0.12,
				  0.17, 0.18, 0.12, 0.08,  0.09,
				  0.32, 0.31, 0.35, 0.42,  0.32,
				  1.04, 1.05, 0.91, 1.04,  1.27,
				  2.86, 2.52, 2.61, 3.04,  3.16,
				  6.64, 7.03, 6.43, 6.46,  8.47,
				 12.71,13.39,14.59,14.64, 16.38,
				 20.11,23.98,26.69,27.55, 28.53,
				 24.40,33.16,42.12,47.77, 50.37,
				 32.81,42.31,52.87,66.01, 74.64,
				 45.54,47.94,62.05,84.65,104.21)
v.cases		<- c(   3,   3,   1,   4,  12,
  				   16,  17,  11,   8,   8,
				   24,  29,  33,  39,  30,
				   79,  76,  82,  95, 115,
				  234, 185, 183, 267, 285,
				  458, 552, 450, 431, 723,
				  720, 867,1069, 974,1004,
				  890,1230,1550,1840,1811,
				  891,1266,1829,2395,3028,
				  920,1243,1584,2292,3176,
				  831, 937,1285,1787,2659)
col.names	<- paste(as.character(seq(from=1955,length=5,by=5)),"-",
			 		 as.character(seq(from=1955,length=5,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			age1		=25				,
			per1		=1955			,
			unit		=5				,
			label		="Italian bladder cancer"))
}	#	data.Italian.bladder.cancer

##################################
#	BELGIAN LUNG CANCER DATA
##################################
data.Belgian.lung.cancer	<- function(unbalanced=FALSE)
#	BN, 17 oct 2013
#	An example with A,drift effects
#
#	Taken from table VIII of
#	Clayton, D. and Schifflers, E. (1987a)
#	Models for temperoral variation in cancer rates. I: age-period and age-cohort models.
#	Statistics in Medicine 6, 449-467.
#
#	Original Table caption:
#	age-specific mortality rates (per 100,000 person-years observation) of lung cancer in
#	Belgian females during the period 1955-1978. Numerators are shown in parentheses
#	(source of data: WHO mortality database).
#
#	NOTE	The data are unbalanced since the last column only covers 4 years.  This is not used.
#	In:		unbalanced		logical.  If true unbalanced version includind last column
{	#	data.Belgian.lung.cancer
v.rates		<- c( 0.19, 0.13, 0.50, 0.19, 0.70,
				  0.66, 0.98, 0.72, 0.71, 0.57,
				  0.78, 1.32, 1.47, 1.64, 1.32,
				  2.67, 3.16, 2.53, 3.38, 3.93,
				  4.84, 5.60, 4.93, 6.05, 6.83,
				  6.60, 8.50, 7.65,10.59,10.42,
				 10.36,12.00,12.68,14.34,17.95,
				 14.76,16.37,18.00,17.60,23.91,
				 20.53,22.60,24.90,24.33,32.70,
				 26.24,27.70,30.47,36.94,38.47,
				 33.47,33.61,36.77,43.69,45.20)
v.cases		<- c(  3,  2,  7,  3, 10,
				  11, 16, 11, 10,  7,
				  11, 22, 24, 25, 15,
				  36, 44, 42, 53, 48,
				  77, 74, 68, 99, 88,
				 106,131, 99,142,134,
				 157,184,189,180,177,
				 193,232,262,249,239,
				 219,267,323,325,343,
				 223,250,308,412,358,
				 198,214,253,338,312)
col.names	<- c("1955-1959","1960-1964","1965-1969","1970-1974","1975-1978")
row.names	<- paste(as.character(seq(from=25  ,length=11,by=5)),"-",
			 		 as.character(seq(from=25  ,length=11,by=5)+4),sep="")
rates	<- matrix(data=v.rates,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
cases	<- matrix(data=v.cases,nrow=11, ncol=5,byrow=TRUE,dimnames=list(row.names,col.names))
if(unbalanced==FALSE)
	return(apc.data.list(
			response	=cases[,(1:4)]					,
			dose		=cases[,(1:4)]/rates[,(1:4)]	,
			data.format	="AP"							,
			age1		=25								,
			per1		=1955							,
			unit		=5								))
if(unbalanced==TRUE)
	return(apc.data.list(
			response	=cases			,
			dose		=cases/rates	,
			data.format	="AP"			,
			unit		=5				,
			label		="Belgian lung cancer"))
}	#	data.Belgian.lung.cancer

###################################
##	PROSTATE CANCER FOR NONWHITES IN THE US
###################################
data.US.prostate.cancer	<- function()
##	BN, 28 apr 2015
##	An example with over-dispersion
##
##	Taken from table 2 of
##	Holford, T.R. (1983)
##	The estimation of age, period and cohort effects for vital rates.
##	Biometrics 39, 311-324.
##
##	Original Table caption:
##	Number of prostate cancer deathrs and midperiod population for nonwhites in the
##	U.S. by age and period
##	Sources:
##	Cancer deaths: National Center for Health Statistics, 1937-1973
##	Population 1935-60: Grove and Hetzel, 1968
##	Population 1960-69: Bureau of the Census, 1974		
##	Population measured in 1000s
##
{	#	data.US.prostate.cancer
v.deaths	<- c( 177, 271, 312, 382, 321, 305, 308,
				  262, 350, 552, 620, 714, 649, 738,
				  360, 479, 644, 949, 932,1292,1327,
				  409, 544, 812,1150,1668,1958,2153,
				  328, 509, 763,1097,1593,2039,2433,
				  222, 359, 584, 845,1192,1638,2068,
				  108, 178, 285, 475, 742, 992,1374)

v.population<- c( 301, 317, 353, 395, 426, 473, 498,
				  212, 248, 279, 301, 358, 411, 443,
				  159, 194, 222, 222, 258, 304, 341,
				  132, 144, 169, 210, 230, 264, 297,
				   76,  94, 110, 125, 149, 180, 197,
				   37,  47,  59,  71,  91, 108, 118,
				   19,  22,  32,  39,  44,  56,  66)
col.names	<- paste(as.character(seq(from=1935,length=7,by=5)),"-",
			 		 as.character(seq(from=1935,length=7,by=5)+4),sep="")
row.names	<- paste(as.character(seq(from=50  ,length=7,by=5)),"-",
			 		 as.character(seq(from=50  ,length=7,by=5)+4),sep="")
response	<- matrix(data=v.deaths		,nrow=7, ncol=7,byrow=TRUE,dimnames=list(row.names,col.names))
dose		<- matrix(data=v.population	,nrow=7, ncol=7,byrow=TRUE,dimnames=list(row.names,col.names))
return(apc.data.list(
			response	=response		,
			dose		=dose			,
			data.format	="AP"			,
			age1		=50				,
			per1		=1935			,
			unit		=5				,
			label		="US prostate cancer"))
}	#	data.US.prostate.cancer

##################################
#	UK Asbestos data
##################################
data.asbestos	<- function(all.age.groups=FALSE)
#	BN, 17 oct 2013
#
#	Taken from
#	Martinez Miranda, Nielsen and Nielsen (2013)
#	Inference and forecasting in the age-period-cohort model with unknown exposure with
#	an application to mesothelioma mortality.
#	To appear in Journal of the Royal Statistical Society series A
#	
#	update of data from the Health and Safety Executive
{	#	data.asbestos
v.cases	<-c(0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,0	,2	,0	,1	,2	,0	,0	,1	,0	,0	,2	,1	,1	,1	,1	,4	,1	,1	,4	,5	,3	,5	,3	,3	,6	,3	,2	,3	,4	,1	,4	,1	,0	,2	,1	,0	,1	,0	,0	,0	,0	,2	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,1	,2	,0	,3	,0	,1	,3	,4	,1	,1	,2	,6	,1	,1	,3	,3	,5	,3	,4	,1	,5	,3	,8	,3	,4	,4	,5	,3	,1	,3	,2	,2	,3	,1	,0	,3	,1	,4	,2	,0	,1	,1	,3	,1	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,3	,2	,2	,3	,0	,1	,1	,1	,0	,3	,1	,4	,6	,8	,3	,6	,3	,10	,6	,7	,6	,4	,2	,6	,5	,8	,4	,1	,0	,2	,1	,3	,1	,1	,1	,2	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,1	,0	,0	,1	,1	,2	,1	,0	,1	,1	,4	,2	,1	,2	,5	,4	,6	,5	,10	,3	,4	,11	,10	,5	,9	,1	,5	,4	,7	,6	,3	,2	,4	,6	,5	,0	,1	,0	,0	,1	,2	,2	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,1	,1	,3	,0	,1	,0	,3	,3	,4	,3	,2	,4	,3	,5	,5	,1	,1	,10	,7	,4	,7	,5	,2	,5	,13	,1	,5	,3	,4	,0	,6	,5	,1	,4	,2	,2	,1	,1	,1	,2	,1	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,2	,1	,6	,1	,2	,5	,2	,4	,4	,2	,3	,4	,6	,9	,7	,8	,8	,3	,6	,6	,5	,2	,6	,7	,4	,10	,5	,5	,3	,5	,6	,2	,1	,1	,2	,0	,3	,0	,1	,1	,0	,1	,1	,0	,2	,0	,0	,0	,0	,0	,0	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,0	,0	,1	,1	,1	,1	,1	,1	,0	,4	,4	,1	,5	,5	,6	,1	,6	,5	,2	,6	,1	,5	,8	,5	,9	,9	,6	,7	,8	,5	,3	,7	,9	,7	,4	,8	,2	,5	,4	,2	,1	,4	,2	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,0	,1	,3	,4	,3	,5	,2	,6	,2	,5	,3	,4	,4	,11	,3	,5	,10	,10	,3	,6	,11	,7	,8	,6	,6	,4	,9	,10	,7	,5	,2	,3	,2	,0	,4	,0	,0	,2	,2	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,1	,2	,2	,1	,2	,6	,2	,3	,12	,7	,5	,3	,4	,3	,4	,3	,8	,8	,6	,11	,11	,9	,11	,11	,4	,6	,10	,5	,7	,6	,9	,3	,3	,3	,3	,5	,0	,4	,2	,3	,1	,1	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,2	,0	,0	,2	,2	,2	,0	,1	,0	,3	,3	,3	,7	,5	,4	,5	,9	,5	,8	,9	,5	,7	,5	,14	,13	,5	,11	,9	,7	,10	,8	,9	,9	,12	,8	,2	,11	,7	,7	,3	,0	,4	,3	,3	,1	,2	,3	,1	,0	,2	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,2	,0	,3	,2	,2	,1	,3	,3	,5	,6	,1	,5	,7	,5	,6	,5	,6	,5	,11	,9	,4	,10	,4	,9	,9	,9	,14	,13	,10	,7	,6	,8	,10	,10	,8	,7	,7	,9	,8	,2	,4	,2	,2	,1	,3	,2	,1	,1	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,0	,1	,1	,2	,2	,2	,2	,1	,5	,3	,7	,5	,5	,9	,8	,9	,13	,11	,9	,8	,8	,12	,11	,9	,12	,6	,23	,5	,17	,11	,8	,4	,5	,8	,13	,12	,12	,9	,8	,3	,5	,4	,6	,3	,1	,0	,1	,1	,1	,0	,0	,2	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,1	,2	,1	,0	,1	,3	,3	,1	,0	,5	,0	,4	,8	,4	,7	,10	,10	,9	,9	,12	,11	,10	,10	,8	,8	,6	,8	,14	,10	,13	,13	,15	,15	,10	,13	,15	,8	,12	,8	,11	,6	,6	,6	,3	,1	,2	,2	,2	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,0	,0	,3	,1	,2	,1	,1	,6	,2	,7	,3	,5	,5	,6	,11	,11	,13	,8	,8	,13	,12	,17	,9	,15	,8	,6	,10	,13	,17	,16	,14	,12	,11	,10	,9	,12	,8	,4	,9	,5	,7	,7	,4	,0	,1	,1	,2	,1	,3	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,2	,4	,3	,3	,1	,1	,4	,4	,3	,6	,3	,4	,10	,3	,9	,10	,17	,12	,13	,14	,18	,17	,11	,14	,18	,12	,12	,16	,14	,12	,12	,11	,12	,5	,14	,9	,7	,11	,12	,3	,7	,7	,5	,3	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,0	,1	,2	,1	,4	,3	,3	,4	,4	,2	,3	,5	,5	,5	,3	,4	,4	,11	,10	,7	,14	,5	,18	,13	,15	,12	,22	,11	,13	,10	,15	,21	,12	,14	,14	,16	,22	,15	,6	,14	,6	,11	,8	,5	,4	,2	,1	,3	,3	,2	,1	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,3	,0	,1	,2	,1	,5	,6	,3	,6	,4	,4	,5	,10	,7	,9	,13	,10	,12	,16	,14	,21	,21	,18	,12	,16	,11	,11	,5	,20	,24	,14	,21	,11	,15	,20	,14	,17	,11	,9	,7	,7	,9	,6	,12	,2	,3	,1	,3	,0	,2	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,1	,3	,1	,2	,2	,3	,2	,6	,5	,2	,6	,9	,8	,11	,9	,4	,9	,13	,14	,20	,10	,22	,26	,12	,25	,22	,19	,14	,19	,11	,21	,20	,14	,18	,15	,14	,19	,11	,7	,12	,11	,12	,12	,12	,4	,9	,5	,3	,1	,2	,2	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,4	,1	,0	,0	,2	,3	,4	,3	,4	,6	,4	,8	,7	,2	,2	,3	,11	,15	,14	,18	,14	,12	,24	,16	,26	,27	,16	,16	,23	,8	,8	,11	,14	,16	,24	,18	,24	,17	,12	,7	,22	,12	,8	,7	,6	,8	,4	,8	,3	,3	,1	,2	,0	,0	,2	,0	,1	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,3	,2	,1	,2	,2	,2	,5	,4	,7	,9	,4	,4	,4	,6	,10	,9	,16	,9	,12	,13	,20	,24	,18	,19	,27	,25	,19	,25	,25	,21	,16	,23	,19	,25	,20	,13	,20	,18	,15	,14	,14	,12	,6	,9	,9	,9	,2	,4	,2	,1	,2	,0	,0	,1	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,3	,3	,3	,1	,3	,6	,4	,5	,5	,4	,7	,5	,16	,11	,9	,19	,11	,12	,18	,16	,17	,22	,30	,27	,27	,28	,25	,29	,20	,37	,23	,16	,19	,13	,16	,16	,30	,21	,20	,21	,15	,10	,18	,7	,13	,7	,6	,4	,3	,2	,1	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,3	,1	,3	,4	,3	,3	,4	,4	,9	,10	,8	,8	,14	,9	,16	,6	,11	,11	,19	,21	,22	,29	,24	,21	,16	,27	,30	,31	,26	,36	,35	,26	,24	,20	,34	,23	,24	,19	,17	,19	,18	,17	,12	,6	,5	,4	,5	,7	,2	,2	,3	,2	,2	,2	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,3	,2	,2	,2	,1	,5	,4	,2	,3	,8	,6	,3	,7	,14	,11	,10	,13	,12	,19	,18	,19	,23	,21	,39	,24	,33	,22	,25	,29	,29	,38	,30	,29	,17	,25	,15	,12	,30	,27	,23	,18	,15	,15	,16	,12	,7	,7	,9	,4	,4	,3	,2	,0	,2	,0	,0	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,6	,3	,8	,3	,3	,5	,9	,9	,10	,11	,13	,15	,13	,15	,16	,14	,23	,14	,21	,18	,27	,23	,30	,30	,29	,21	,35	,22	,31	,34	,25	,20	,32	,20	,21	,19	,22	,18	,15	,16	,12	,6	,7	,11	,8	,4	,3	,2	,0	,0	,1	,0	,0	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,1	,3	,2	,10	,6	,4	,6	,10	,12	,10	,11	,13	,13	,13	,20	,23	,17	,24	,22	,26	,18	,25	,40	,28	,29	,42	,37	,35	,33	,42	,39	,30	,23	,25	,25	,19	,22	,16	,19	,14	,15	,14	,6	,11	,5	,3	,1	,2	,1	,2	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,3	,1	,1	,4	,2	,10	,2	,3	,7	,14	,13	,14	,12	,16	,21	,23	,22	,15	,21	,22	,30	,26	,30	,32	,18	,40	,27	,37	,37	,30	,34	,46	,29	,32	,34	,22	,34	,24	,38	,28	,22	,17	,14	,8	,8	,9	,5	,7	,5	,3	,2	,0	,2	,2	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,0	,1	,3	,2	,5	,2	,4	,5	,6	,6	,13	,12	,19	,9	,7	,21	,19	,18	,22	,29	,18	,17	,17	,27	,29	,37	,29	,27	,30	,43	,42	,37	,38	,50	,41	,46	,26	,26	,29	,30	,26	,15	,22	,22	,17	,19	,11	,6	,6	,6	,2	,4	,0	,1	,2	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,3	,0	,6	,8	,10	,19	,19	,13	,24	,17	,21	,16	,23	,23	,27	,23	,26	,33	,26	,47	,49	,38	,52	,39	,40	,40	,43	,34	,35	,40	,35	,36	,25	,27	,25	,25	,23	,16	,15	,16	,9	,7	,7	,9	,8	,4	,4	,1	,2	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,2	,8	,3	,4	,8	,6	,11	,11	,12	,15	,18	,13	,22	,22	,25	,31	,26	,35	,28	,29	,27	,31	,45	,51	,48	,40	,44	,55	,54	,32	,43	,47	,52	,30	,30	,26	,29	,26	,15	,19	,13	,6	,7	,11	,3	,3	,5	,6	,5	,0	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,0	,2	,2	,1	,3	,0	,1	,5	,6	,9	,17	,13	,20	,11	,18	,15	,24	,22	,23	,30	,25	,25	,24	,38	,31	,35	,31	,51	,42	,47	,44	,55	,48	,48	,39	,44	,41	,40	,32	,17	,21	,28	,22	,20	,11	,21	,16	,13	,6	,10	,6	,2	,4	,0	,1	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,3	,2	,4	,6	,5	,7	,10	,13	,15	,13	,22	,24	,21	,16	,20	,28	,30	,30	,32	,47	,29	,34	,37	,37	,43	,53	,46	,49	,49	,38	,38	,51	,36	,61	,34	,22	,21	,23	,26	,18	,19	,20	,12	,15	,3	,7	,9	,2	,1	,0	,2	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,2	,0	,0	,0	,1	,1	,2	,0	,0	,2	,3	,4	,2	,7	,13	,10	,15	,15	,17	,24	,27	,24	,23	,26	,28	,24	,26	,42	,30	,29	,40	,40	,53	,46	,44	,54	,42	,50	,69	,50	,49	,38	,64	,44	,39	,30	,31	,23	,28	,23	,26	,13	,10	,7	,7	,3	,5	,4	,4	,3	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,2	,2	,2	,6	,4	,7	,9	,13	,19	,8	,17	,25	,29	,33	,36	,35	,41	,45	,39	,34	,40	,42	,43	,43	,51	,50	,42	,40	,45	,62	,56	,71	,54	,52	,49	,45	,27	,21	,26	,24	,21	,18	,12	,13	,9	,7	,5	,3	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,5	,1	,7	,5	,4	,8	,9	,7	,15	,23	,18	,17	,21	,26	,29	,28	,38	,43	,38	,42	,57	,42	,35	,45	,50	,52	,61	,51	,66	,54	,57	,43	,50	,61	,54	,40	,38	,22	,20	,20	,22	,25	,18	,10	,9	,5	,4	,2	,1	,2	,0	,1,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,0	,1	,2	,1	,6	,3	,3	,4	,5	,11	,10	,20	,13	,25	,25	,32	,30	,35	,30	,40	,36	,43	,41	,55	,41	,47	,50	,54	,51	,70	,49	,57	,57	,62	,63	,72	,57	,59	,58	,55	,52	,24	,18	,22	,24	,21	,20	,7	,8	,6	,3	,4	,1	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,0	,1	,1	,1	,3	,2	,2	,9	,9	,13	,16	,21	,27	,29	,35	,34	,28	,35	,39	,37	,45	,74	,52	,41	,57	,48	,61	,51	,79	,65	,77	,62	,73	,60	,51	,45	,49	,48	,34	,39	,20	,17	,15	,19	,8	,12	,12	,8	,4	,4	,3	,0	,4,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,0	,0	,3	,3	,6	,4	,7	,8	,10	,13	,15	,30	,24	,26	,24	,46	,40	,37	,49	,47	,32	,53	,41	,65	,59	,58	,48	,66	,53	,53	,53	,68	,55	,64	,67	,60	,53	,47	,41	,42	,22	,25	,17	,21	,12	,11	,6	,4	,3	,4	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,0	,2	,4	,3	,3	,2	,6	,5	,2	,7	,6	,14	,7	,17	,26	,26	,25	,40	,30	,49	,41	,47	,59	,56	,49	,50	,64	,68	,64	,54	,74	,62	,69	,61	,59	,56	,72	,67	,53	,46	,37	,52	,39	,22	,14	,12	,13	,13	,10	,5	,7	,4	,1	,4,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,0	,0	,2	,1	,1	,3	,1	,2	,3	,2	,3	,5	,7	,11	,13	,14	,13	,19	,18	,37	,32	,37	,38	,38	,55	,49	,57	,48	,54	,74	,62	,57	,57	,68	,61	,73	,60	,66	,69	,54	,61	,74	,61	,53	,42	,44	,33	,32	,21	,22	,13	,6	,10	,5	,9	,0	,1	,1,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,3	,4	,5	,3	,2	,9	,3	,11	,16	,4	,18	,28	,31	,33	,31	,31	,40	,58	,60	,52	,59	,63	,62	,78	,52	,57	,67	,59	,75	,81	,69	,71	,64	,56	,56	,55	,47	,39	,41	,29	,37	,19	,15	,8	,5	,10	,7	,2	,6	,5,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,1	,0	,1	,0	,0	,4	,0	,0	,3	,1	,6	,2	,3	,5	,4	,9	,8	,16	,14	,27	,52	,49	,46	,39	,46	,41	,65	,45	,62	,76	,80	,58	,61	,51	,75	,67	,62	,85	,84	,65	,70	,55	,60	,58	,45	,47	,38	,36	,31	,9	,11	,10	,6	,3	,5	,3	,9)
col.names	<- c("5-9","10-14","15-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=41, ncol=75, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(5,69)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1967			,
				unit			=1				))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="mesothelioma, UK"))
}	#	data.asbestos						

##################################
#	UK Asbestos data updated to 2013
##################################
data.asbestos.2013	<- function(all.age.groups=FALSE)
#	BN, 30 Apr 2016
#
#	Taken from
#	Martinez-Miranda, Nielsen and Nielsen (2016)
#	A simple benchmark for mesothelioma projection for Great Britain.
#	Occupational and Environmental Medicine 73, 561-563.
#
#	Asbestos induced mesothelioma mortality for men
#	update of data from the Health and Safety Executive
{	#	data.asbestos.2013
v.cases	<-c(0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,1	,2	,0	,3	,0	,1	,3	,4	,1	,1	,2	,6	,1	,1	,3	,3	,5	,3	,4	,1	,5	,3	,8	,3	,4	,4	,5	,3	,1	,3	,2	,2	,3	,1	,0	,3	,1	,4	,2	,0	,1	,1	,3	,1	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,3	,2	,2	,3	,0	,1	,1	,1	,0	,3	,1	,4	,6	,8	,3	,6	,3	,10	,6	,7	,6	,4	,2	,6	,5	,8	,4	,1	,0	,2	,1	,3	,1	,1	,1	,2	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,1	,0	,0	,1	,1	,2	,1	,0	,1	,1	,4	,2	,1	,2	,5	,4	,6	,5	,10	,3	,4	,11	,10	,5	,9	,1	,5	,4	,7	,6	,3	,2	,4	,6	,5	,0	,1	,0	,0	,1	,2	,2	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,1	,1	,3	,0	,1	,0	,3	,3	,4	,3	,2	,4	,3	,5	,5	,1	,1	,10	,7	,4	,7	,5	,2	,5	,13	,1	,5	,3	,4	,0	,6	,5	,1	,4	,2	,2	,1	,1	,1	,2	,1	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,2	,1	,6	,1	,2	,5	,2	,4	,4	,2	,3	,4	,6	,9	,7	,8	,8	,3	,6	,6	,5	,2	,6	,7	,4	,10	,5	,5	,3	,5	,6	,2	,1	,1	,2	,0	,3	,0	,1	,1	,0	,1	,1	,0	,2	,0	,0	,0	,0	,0	,0	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,0	,0	,1	,1	,1	,1	,1	,1	,0	,4	,4	,1	,5	,5	,6	,1	,6	,5	,2	,6	,1	,5	,8	,5	,9	,9	,6	,7	,8	,5	,3	,7	,9	,7	,4	,8	,2	,5	,4	,2	,1	,4	,2	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,0	,1	,3	,4	,3	,5	,2	,6	,2	,5	,3	,4	,4	,11	,3	,5	,10	,10	,3	,6	,11	,7	,8	,6	,6	,4	,9	,10	,7	,5	,2	,3	,2	,0	,4	,0	,0	,2	,2	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,1	,2	,2	,1	,2	,6	,2	,3	,12	,7	,5	,3	,4	,3	,4	,3	,8	,8	,6	,11	,11	,9	,11	,11	,4	,6	,10	,5	,7	,6	,9	,3	,3	,3	,3	,5	,0	,4	,2	,3	,1	,1	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,2	,0	,0	,2	,2	,2	,0	,1	,0	,3	,3	,3	,7	,5	,4	,5	,9	,5	,8	,9	,5	,7	,5	,14	,13	,5	,11	,9	,7	,10	,8	,9	,9	,12	,8	,2	,11	,7	,7	,3	,0	,4	,3	,3	,1	,2	,3	,1	,0	,2	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,2	,0	,3	,2	,2	,1	,3	,3	,5	,6	,1	,5	,7	,5	,6	,5	,6	,5	,11	,9	,4	,10	,4	,9	,9	,9	,14	,13	,10	,7	,6	,8	,10	,10	,8	,7	,7	,9	,8	,2	,4	,2	,2	,1	,3	,2	,1	,1	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,0	,1	,1	,2	,2	,2	,2	,1	,5	,3	,7	,5	,5	,9	,8	,9	,13	,11	,9	,8	,8	,12	,11	,9	,12	,6	,23	,5	,17	,11	,8	,4	,5	,8	,13	,12	,12	,9	,8	,3	,5	,4	,6	,3	,1	,0	,1	,1	,1	,0	,0	,2	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,1	,1	,1	,2	,1	,0	,1	,3	,3	,1	,0	,5	,0	,4	,8	,4	,7	,10	,10	,9	,9	,12	,11	,10	,10	,8	,8	,6	,8	,14	,10	,13	,13	,15	,15	,10	,13	,15	,8	,12	,8	,11	,6	,6	,6	,3	,1	,2	,2	,2	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,0	,0	,3	,1	,2	,1	,1	,6	,2	,7	,3	,5	,5	,6	,11	,11	,13	,8	,8	,13	,12	,17	,9	,15	,8	,6	,10	,13	,17	,16	,14	,12	,11	,10	,9	,12	,8	,4	,9	,5	,7	,7	,4	,0	,1	,1	,2	,1	,3	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,2	,4	,3	,3	,1	,1	,4	,4	,3	,6	,3	,4	,10	,3	,9	,10	,17	,12	,13	,14	,18	,17	,11	,14	,18	,12	,12	,16	,14	,12	,12	,11	,12	,5	,14	,9	,7	,11	,12	,3	,7	,7	,5	,3	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,0	,1	,2	,1	,4	,3	,3	,4	,4	,2	,3	,5	,5	,5	,3	,4	,4	,11	,10	,7	,14	,5	,18	,13	,15	,12	,22	,11	,13	,10	,15	,21	,12	,14	,14	,16	,22	,15	,6	,14	,6	,11	,8	,5	,4	,2	,1	,3	,3	,2	,1	,1	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,3	,0	,1	,2	,1	,5	,6	,3	,6	,4	,4	,5	,10	,7	,9	,13	,10	,12	,16	,14	,21	,21	,18	,12	,16	,11	,11	,5	,20	,24	,14	,21	,11	,15	,20	,14	,17	,11	,9	,7	,7	,9	,6	,12	,2	,3	,1	,3	,0	,2	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,1	,3	,1	,2	,2	,3	,2	,6	,5	,2	,6	,9	,8	,11	,9	,4	,9	,13	,14	,20	,10	,22	,26	,12	,25	,22	,19	,14	,19	,11	,21	,20	,14	,18	,15	,14	,19	,11	,7	,12	,11	,12	,12	,12	,4	,9	,5	,3	,1	,2	,2	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,4	,1	,0	,0	,2	,3	,4	,3	,4	,6	,4	,8	,7	,2	,2	,3	,11	,15	,14	,18	,14	,12	,24	,16	,26	,27	,16	,16	,23	,8	,8	,11	,14	,16	,24	,18	,24	,17	,12	,7	,22	,12	,8	,7	,6	,8	,4	,8	,3	,3	,1	,2	,0	,0	,2	,0	,1	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,3	,2	,1	,2	,2	,2	,5	,4	,7	,9	,4	,4	,4	,6	,10	,9	,16	,9	,12	,13	,20	,24	,18	,19	,27	,25	,19	,25	,25	,21	,16	,23	,19	,25	,20	,13	,20	,18	,15	,14	,14	,12	,6	,9	,9	,9	,2	,4	,2	,1	,2	,0	,0	,1	,0	,0	,0	,1,
			0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,3	,3	,3	,1	,3	,6	,4	,5	,5	,4	,7	,5	,16	,11	,9	,19	,11	,12	,18	,16	,17	,22	,30	,27	,27	,28	,25	,29	,20	,37	,23	,16	,19	,13	,16	,16	,30	,21	,20	,21	,15	,10	,18	,7	,13	,7	,6	,4	,3	,2	,1	,1	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,3	,1	,3	,4	,3	,3	,4	,4	,9	,10	,8	,8	,14	,9	,16	,6	,11	,11	,19	,21	,22	,29	,24	,21	,16	,27	,30	,31	,26	,36	,35	,26	,24	,20	,34	,23	,24	,19	,17	,19	,18	,17	,12	,6	,5	,4	,5	,7	,2	,2	,3	,2	,2	,2	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,3	,2	,2	,2	,1	,5	,4	,2	,3	,8	,6	,3	,7	,14	,11	,10	,13	,12	,19	,18	,19	,23	,21	,39	,24	,33	,22	,25	,29	,29	,38	,30	,29	,17	,25	,15	,12	,30	,27	,23	,18	,15	,15	,16	,12	,7	,7	,9	,4	,4	,3	,2	,0	,2	,0	,0	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,6	,3	,8	,3	,3	,5	,9	,9	,10	,11	,13	,15	,13	,15	,16	,14	,23	,14	,21	,18	,27	,23	,30	,30	,29	,21	,35	,22	,31	,34	,25	,20	,32	,20	,21	,19	,22	,18	,15	,16	,12	,6	,7	,11	,8	,4	,3	,2	,0	,0	,1	,0	,0	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,1	,3	,2	,10	,6	,4	,6	,10	,12	,10	,11	,13	,13	,13	,20	,23	,17	,24	,22	,26	,18	,25	,40	,28	,29	,42	,37	,35	,33	,42	,39	,30	,23	,25	,25	,19	,22	,16	,19	,14	,15	,14	,6	,11	,5	,3	,1	,2	,1	,2	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,3	,1	,1	,4	,2	,10	,2	,3	,7	,14	,13	,14	,12	,16	,21	,23	,22	,15	,21	,22	,30	,26	,30	,32	,18	,40	,27	,37	,37	,30	,34	,46	,29	,32	,34	,22	,34	,24	,38	,28	,22	,17	,14	,8	,8	,9	,5	,7	,5	,3	,2	,0	,2	,2	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,0	,1	,3	,2	,5	,2	,4	,5	,6	,6	,13	,12	,19	,9	,7	,21	,19	,18	,22	,29	,18	,17	,17	,27	,29	,37	,29	,27	,30	,43	,42	,37	,38	,50	,41	,46	,26	,26	,29	,30	,26	,15	,22	,22	,17	,19	,11	,6	,6	,6	,2	,4	,0	,1	,2	,1	,0	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,3	,0	,6	,8	,10	,19	,19	,13	,24	,17	,21	,16	,23	,23	,27	,23	,26	,33	,26	,47	,49	,38	,52	,39	,40	,40	,43	,34	,35	,40	,35	,36	,25	,27	,25	,25	,23	,16	,15	,16	,9	,7	,7	,9	,8	,4	,4	,1	,2	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,2	,8	,3	,4	,8	,6	,11	,11	,12	,15	,18	,13	,22	,22	,25	,31	,26	,35	,28	,29	,27	,31	,45	,51	,48	,40	,44	,55	,54	,32	,43	,47	,52	,30	,30	,26	,29	,26	,15	,19	,13	,6	,7	,11	,3	,3	,5	,6	,5	,0	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,0	,2	,2	,1	,3	,0	,1	,5	,6	,9	,17	,13	,20	,11	,18	,15	,24	,22	,23	,30	,25	,25	,24	,38	,31	,35	,31	,51	,42	,47	,44	,55	,48	,48	,39	,44	,41	,40	,32	,17	,21	,28	,22	,20	,11	,21	,16	,13	,6	,10	,6	,2	,4	,0	,1	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,2	,3	,2	,4	,6	,5	,7	,10	,13	,15	,13	,22	,24	,21	,16	,20	,28	,30	,30	,32	,47	,29	,34	,37	,37	,43	,53	,46	,49	,49	,38	,38	,51	,36	,61	,34	,22	,21	,23	,26	,18	,19	,20	,12	,15	,3	,7	,9	,2	,1	,0	,2	,1	,0,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,2	,0	,0	,0	,1	,1	,2	,0	,0	,2	,3	,4	,2	,7	,13	,10	,15	,15	,17	,24	,27	,24	,23	,26	,28	,24	,26	,42	,30	,29	,40	,40	,53	,46	,44	,54	,42	,50	,69	,50	,49	,38	,64	,44	,39	,30	,31	,23	,28	,23	,26	,13	,10	,7	,7	,3	,5	,4	,4	,3	,0,
			0	,0	,0	,0	,1	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,2	,2	,2	,6	,4	,7	,9	,13	,19	,8	,17	,25	,29	,33	,36	,35	,41	,45	,39	,34	,40	,42	,43	,43	,51	,50	,42	,40	,45	,62	,56	,71	,54	,52	,49	,45	,27	,21	,26	,24	,21	,18	,12	,13	,9	,7	,5	,3	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,0	,5	,1	,7	,5	,4	,8	,9	,7	,15	,23	,18	,17	,21	,26	,29	,28	,38	,43	,38	,42	,57	,42	,35	,45	,50	,52	,61	,51	,66	,54	,57	,43	,50	,61	,54	,40	,38	,22	,20	,20	,22	,25	,18	,10	,9	,5	,4	,2	,1	,2	,0	,1,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,2	,0	,1	,2	,1	,6	,3	,3	,4	,5	,11	,10	,20	,13	,25	,25	,32	,30	,35	,30	,40	,36	,43	,41	,55	,41	,47	,50	,54	,51	,70	,49	,57	,57	,62	,63	,72	,57	,59	,58	,55	,52	,24	,18	,22	,24	,21	,20	,7	,8	,5	,3	,4	,1	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,0	,1	,1	,1	,3	,2	,2	,9	,9	,13	,16	,21	,27	,29	,35	,34	,28	,35	,39	,37	,45	,74	,52	,41	,57	,48	,61	,51	,79	,65	,77	,62	,73	,60	,51	,45	,49	,48	,34	,39	,20	,17	,15	,19	,8	,12	,12	,8	,4	,4	,3	,0	,4,
			1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,0	,0	,3	,3	,6	,4	,7	,8	,10	,13	,15	,30	,24	,26	,24	,46	,40	,37	,48	,47	,32	,53	,41	,65	,59	,58	,48	,66	,53	,53	,53	,68	,55	,64	,67	,60	,53	,47	,41	,43	,22	,25	,17	,21	,12	,11	,6	,4	,3	,4	,0	,2,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,0	,2	,4	,3	,3	,2	,6	,5	,2	,7	,6	,14	,7	,17	,26	,26	,25	,40	,30	,49	,41	,47	,59	,56	,49	,50	,64	,68	,64	,54	,74	,62	,69	,61	,59	,56	,72	,67	,53	,46	,37	,52	,39	,22	,14	,12	,13	,13	,10	,5	,7	,4	,1	,4,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,0	,0	,2	,1	,1	,3	,1	,2	,3	,2	,3	,5	,7	,11	,13	,14	,13	,19	,18	,37	,32	,37	,38	,38	,55	,49	,57	,48	,54	,74	,62	,57	,57	,69	,61	,74	,60	,66	,69	,54	,61	,74	,61	,53	,42	,44	,33	,32	,21	,22	,13	,6	,10	,5	,9	,0	,1	,1,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,1	,1	,1	,3	,4	,5	,3	,2	,9	,3	,11	,16	,4	,18	,28	,31	,33	,31	,31	,40	,58	,60	,52	,60	,63	,62	,78	,52	,57	,67	,59	,75	,81	,69	,71	,64	,56	,57	,55	,47	,39	,41	,29	,37	,19	,15	,8	,5	,10	,7	,2	,6	,5,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,2	,1	,0	,1	,0	,0	,4	,0	,0	,3	,1	,6	,2	,3	,6	,4	,9	,8	,16	,14	,27	,51	,49	,46	,39	,47	,42	,66	,45	,63	,77	,81	,58	,62	,51	,76	,67	,62	,86	,85	,65	,70	,55	,60	,60	,45	,50	,38	,36	,32	,10	,11	,10	,6	,3	,5	,3	,9,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,2	,1	,1	,1	,0	,3	,3	,2	,4	,0	,4	,7	,9	,7	,10	,9	,23	,26	,26	,34	,52	,37	,40	,62	,55	,59	,60	,56	,81	,61	,83	,86	,73	,69	,83	,86	,70	,71	,63	,56	,73	,62	,47	,39	,47	,44	,31	,16	,8	,11	,4	,4	,6	,3	,8,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,2	,1	,4	,0	,4	,5	,8	,4	,10	,11	,17	,17	,22	,22	,26	,40	,36	,48	,48	,63	,50	,63	,52	,64	,80	,66	,93	,86	,72	,89	,67	,88	,87	,73	,61	,71	,77	,63	,43	,44	,40	,35	,33	,23	,13	,6	,1	,4	,3	,4,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,0	,0	,1	,2	,0	,1	,0	,0	,1	,0	,3	,6	,5	,2	,3	,10	,6	,5	,14	,18	,14	,21	,29	,39	,36	,60	,52	,51	,64	,56	,72	,70	,78	,66	,81	,73	,82	,77	,67	,79	,103,81	,74	,67	,63	,47	,61	,44	,39	,42	,31	,26	,9	,6	,4	,5	,1	,5,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,1	,1	,2	,2	,2	,0	,2	,3	,5	,7	,7	,13	,11	,13	,18	,24	,26	,29	,30	,50	,65	,50	,46	,64	,56	,70	,82	,65	,63	,79	,74	,76	,77	,83	,75	,86	,80	,83	,80	,64	,46	,45	,32	,41	,26	,30	,21	,14	,8	,4	,1	,8,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,0	,1	,3	,2	,4	,0	,6	,7	,9	,7	,18	,8	,22	,18	,18	,26	,47	,52	,66	,65	,78	,64	,78	,83	,74	,81	,101,94	,89	,91	,92	,83	,79	,100,98	,69	,81	,56	,47	,56	,35	,31	,26	,26	,17	,8	,9	,3	,4,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,3	,1	,3	,3	,4	,0	,1	,5	,1	,5	,10	,7	,11	,14	,12	,22	,25	,29	,36	,51	,56	,68	,68	,75	,79	,66	,96	,88	,72	,94	,101,101,79	,76	,90	,92	,83	,82	,74	,57	,49	,47	,45	,38	,27	,28	,14	,13	,8	,5	,6)
col.names	<- c("0-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=46, ncol=73, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(3,67)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1968			,
				unit			=1				,
				label			="men, UK mesothelioma 2013 update"))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="men, UK mesothelioma 2013 update, all age groups"))
}	#	data.asbestos.2013


########################
#	men
data.asbestos.2013.men	<- function(all.age.groups=FALSE)
#	BN, 17 Sep 2016
{	#	data.asbestos.2013.men
	return(data.asbestos.2013(all.age.groups))
}	#	data.asbestos.2013.men

########################
#	women
data.asbestos.2013.women	<- function(all.age.groups=FALSE)
#	BN, 17 Sep 2016
#	Asbestos induced mesothelioma mortality for women
#	update of data from the Health and Safety Executive
{	#	data.asbestos.2013
v.cases	<-c(0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,1	,1	,2	,1	,2	,1	,1	,2	,1	,0	,0	,1	,2	,0	,0	,1	,0	,0	,0	,4	,1	,4	,1	,1	,1	,1	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,0	,1	,0	,0	,0	,1	,1	,3	,0	,2	,3	,2	,3	,2	,1	,1	,0	,1	,0	,2	,0	,2	,0	,1	,1	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,1	,1	,2	,1	,1	,0	,1	,1	,0	,2	,1	,0	,4	,2	,1	,2	,0	,1	,2	,1	,1	,1	,2	,0	,2	,1	,2	,4	,0	,0	,0	,1	,1	,1	,1	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,2	,1	,1	,1	,1	,1	,3	,1	,1	,0	,4	,1	,1	,2	,0	,0	,2	,0	,3	,0	,2	,0	,1	,1	,0	,0	,0	,3	,0	,0	,1	,0	,1	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,1	,2	,0	,1	,0	,0	,0	,1	,1	,1	,0	,0	,0	,2	,2	,1	,2	,1	,2	,1	,2	,1	,2	,1	,4	,1	,0	,0	,4	,0	,2	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,2	,2	,0	,1	,1	,1	,2	,0	,0	,2	,2	,2	,2	,0	,1	,3	,1	,1	,0	,1	,5	,1	,2	,0	,1	,1	,0	,0	,2	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,3	,1	,1	,2	,2	,3	,0	,3	,2	,1	,2	,0	,2	,1	,1	,0	,4	,5	,3	,3	,3	,1	,1	,1	,2	,1	,1	,1	,1	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,1	,0	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,1	,0	,2	,2	,0	,0	,0	,0	,2	,1	,1	,5	,1	,0	,3	,1	,1	,2	,5	,1	,2	,1	,4	,2	,2	,0	,1	,1	,0	,0	,0	,2	,0	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,3	,1	,2	,2	,0	,2	,3	,2	,0	,0	,2	,0	,2	,5	,3	,1	,4	,2	,1	,5	,0	,3	,1	,2	,1	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,2	,2	,3	,1	,1	,1	,5	,2	,1	,1	,2	,0	,4	,2	,3	,2	,1	,1	,1	,1	,6	,0	,3	,0	,2	,1	,1	,1	,2	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,0	,0	,1	,0	,0	,1	,0	,0	,3	,1	,2	,1	,2	,2	,3	,4	,1	,3	,4	,4	,3	,2	,2	,0	,1	,2	,5	,1	,2	,2	,2	,0	,2	,0	,2	,0	,0	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,1	,0	,0	,0	,1	,0	,2	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,0	,1	,0	,1	,0	,1	,3	,1	,0	,2	,3	,2	,4	,3	,0	,3	,2	,5	,6	,2	,5	,2	,6	,2	,4	,0	,2	,3	,6	,2	,1	,2	,0	,3	,1	,1	,0	,2	,1	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,2	,0	,3	,0	,0	,1	,0	,0	,0	,2	,0	,0	,0	,0	,5	,2	,2	,3	,1	,5	,1	,3	,3	,1	,4	,3	,1	,5	,4	,2	,8	,5	,1	,8	,4	,2	,3	,3	,1	,1	,3	,3	,1	,0	,1	,0	,1	,1	,0	,0	,0	,0	,1	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,2	,0	,0	,1	,1	,0	,0	,0	,1	,1	,3	,1	,5	,1	,2	,2	,0	,3	,1	,2	,4	,3	,2	,0	,4	,0	,6	,2	,1	,3	,1	,2	,2	,4	,0	,2	,2	,0	,3	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,1	,0	,0	,1	,0	,0	,0	,0	,1	,1	,1	,0	,1	,1	,3	,1	,2	,2	,4	,3	,4	,3	,4	,5	,3	,4	,1	,0	,8	,3	,3	,0	,5	,4	,4	,3	,3	,1	,0	,2	,0	,0	,1	,1	,0	,1	,1	,1	,0	,0	,0	,2	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,1	,0	,0	,2	,0	,1	,0	,1	,1	,1	,1	,2	,2	,1	,5	,2	,2	,1	,3	,2	,5	,4	,1	,3	,3	,6	,3	,5	,3	,4	,3	,4	,0	,3	,2	,4	,1	,2	,1	,1	,1	,1	,1	,1	,0	,0	,0	,0	,0	,1	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,1	,1	,1	,0	,0	,0	,0	,0	,2	,1	,1	,0	,0	,1	,0	,1	,0	,0	,1	,1	,1	,5	,4	,3	,3	,6	,5	,3	,6	,3	,3	,4	,3	,5	,1	,3	,3	,0	,2	,2	,1	,0	,1	,1	,1	,1	,1	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,1	,1	,2	,0	,0	,0	,0	,0	,1	,0	,2	,1	,1	,1	,2	,7	,3	,0	,2	,5	,5	,7	,0	,5	,0	,5	,2	,2	,5	,1	,3	,3	,2	,2	,2	,0	,1	,0	,1	,0	,1	,1	,0	,0	,1	,0	,2	,0	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,1	,2	,1	,0	,0	,1	,1	,2	,1	,0	,0	,0	,2	,2	,0	,1	,0	,0	,2	,2	,1	,4	,3	,2	,3	,4	,4	,4	,2	,5	,1	,6	,2	,3	,5	,5	,3	,1	,5	,3	,4	,0	,3	,2	,0	,1	,1	,2	,2	,0	,0	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,1	,0	,2	,1	,0	,1	,0	,0	,0	,0	,1	,1	,1	,3	,4	,3	,2	,2	,7	,1	,4	,1	,4	,2	,5	,3	,4	,4	,3	,8	,5	,4	,2	,5	,2	,2	,4	,0	,4	,0	,2	,2	,1	,0	,0	,0	,1	,0	,0	,1	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,0	,1	,0	,0	,0	,0	,2	,1	,4	,0	,2	,1	,0	,1	,1	,3	,5	,3	,4	,4	,1	,4	,6	,5	,3	,7	,5	,3	,9	,3	,4	,1	,3	,5	,3	,4	,1	,3	,1	,2	,1	,0	,2	,0	,0	,1	,1	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,3	,1	,1	,2	,1	,1	,0	,2	,2	,1	,3	,5	,3	,1	,3	,2	,2	,3	,8	,4	,4	,2	,7	,3	,14	,4	,3	,2	,6	,5	,4	,4	,9	,1	,4	,2	,2	,1	,0	,3	,1	,1	,0	,0	,0	,1	,1	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,1	,1	,1	,0	,3	,1	,0	,1	,1	,3	,1	,1	,2	,2	,2	,2	,4	,0	,4	,1	,4	,8	,3	,7	,4	,6	,1	,4	,3	,6	,1	,5	,6	,2	,7	,1	,3	,1	,2	,1	,4	,0	,1	,2	,0	,0	,0	,1	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,0	,0	,2	,2	,1	,2	,3	,3	,0	,2	,4	,2	,1	,1	,2	,0	,4	,1	,1	,1	,5	,4	,1	,2	,4	,4	,9	,7	,7	,9	,5	,9	,6	,10	,4	,7	,6	,2	,1	,3	,1	,2	,2	,3	,1	,1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,3	,1	,0	,2	,1	,0	,2	,1	,2	,3	,1	,0	,2	,2	,0	,5	,4	,2	,5	,6	,5	,8	,5	,6	,8	,5	,7	,5	,6	,3	,4	,5	,4	,4	,1	,5	,2	,3	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,0	,0	,0	,0	,3	,3	,0	,0	,1	,3	,2	,1	,3	,2	,3	,1	,2	,3	,1	,3	,3	,2	,2	,5	,0	,5	,5	,9	,5	,5	,4	,9	,9	,6	,6	,5	,4	,5	,3	,5	,1	,5	,1	,3	,0	,1	,0	,0	,1	,0	,1	,0	,0	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,2	,0	,0	,0	,2	,2	,3	,2	,1	,2	,0	,1	,2	,2	,4	,3	,1	,1	,1	,5	,5	,5	,6	,5	,2	,4	,6	,8	,9	,3	,9	,7	,4	,3	,4	,8	,7	,6	,2	,1	,0	,3	,4	,3	,1	,2	,0	,0	,1	,0	,0	,0	,0	,0,
			1	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,1	,1	,0	,1	,2	,2	,1	,1	,1	,7	,2	,2	,7	,3	,5	,3	,8	,1	,3	,1	,3	,5	,2	,4	,6	,6	,4	,4	,5	,13	,8	,6	,6	,6	,5	,4	,6	,6	,4	,3	,4	,4	,1	,2	,2	,0	,2	,0	,0	,1	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,1	,0	,1	,2	,2	,1	,4	,2	,1	,4	,4	,2	,2	,1	,2	,2	,4	,4	,2	,3	,6	,9	,4	,7	,5	,5	,9	,8	,6	,6	,3	,2	,4	,7	,6	,2	,5	,3	,4	,1	,3	,1	,0	,1	,0	,0	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,1	,0	,1	,1	,2	,2	,2	,3	,1	,5	,3	,2	,10	,4	,1	,1	,4	,7	,5	,4	,7	,4	,6	,7	,6	,8	,8	,13	,8	,8	,6	,5	,4	,3	,4	,4	,0	,4	,5	,2	,0	,0	,0	,2	,0	,0	,1	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,1	,2	,1	,3	,2	,2	,1	,2	,3	,2	,1	,4	,4	,4	,7	,3	,5	,2	,5	,2	,5	,7	,5	,9	,7	,9	,8	,5	,7	,10	,8	,10	,7	,5	,4	,5	,6	,5	,9	,3	,5	,1	,1	,1	,1	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,2	,1	,1	,2	,2	,4	,2	,2	,4	,4	,7	,3	,2	,8	,7	,3	,6	,7	,7	,4	,5	,3	,10	,11	,3	,10	,7	,12	,13	,6	,10	,10	,10	,7	,7	,3	,4	,7	,3	,3	,1	,1	,2	,0	,0	,0	,0	,0	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,2	,1	,0	,0	,1	,0	,0	,4	,4	,1	,3	,1	,3	,7	,7	,3	,7	,3	,7	,4	,6	,4	,5	,6	,6	,2	,7	,13	,5	,9	,8	,6	,3	,11	,10	,12	,8	,7	,6	,5	,8	,3	,7	,5	,1	,5	,0	,1	,0	,0	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,3	,0	,1	,2	,1	,1	,1	,2	,0	,6	,6	,4	,8	,8	,2	,5	,5	,5	,4	,8	,5	,5	,7	,8	,4	,9	,8	,7	,12	,6	,4	,14	,7	,12	,5	,10	,7	,14	,9	,5	,10	,5	,3	,5	,2	,5	,2	,2	,1	,1	,1	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,1	,0	,0	,0	,1	,0	,0	,0	,0	,2	,2	,0	,1	,3	,1	,4	,5	,4	,8	,6	,7	,2	,7	,10	,4	,4	,10	,3	,9	,12	,7	,6	,3	,14	,10	,12	,9	,13	,7	,8	,11	,14	,6	,12	,10	,6	,4	,7	,3	,4	,3	,5	,0	,1	,1	,2	,0	,1,
			0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,3	,1	,1	,3	,0	,3	,3	,0	,2	,2	,3	,2	,3	,2	,8	,4	,7	,3	,9	,7	,13	,14	,6	,11	,6	,11	,5	,4	,9	,8	,10	,6	,9	,15	,10	,10	,14	,12	,9	,11	,5	,3	,2	,3	,5	,4	,1	,0	,1	,0	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,0	,0	,0	,2	,1	,2	,1	,1	,2	,1	,1	,0	,6	,3	,0	,13	,9	,9	,5	,11	,10	,5	,12	,10	,5	,8	,5	,8	,7	,13	,6	,6	,16	,8	,13	,13	,11	,14	,11	,8	,10	,4	,5	,7	,5	,0	,2	,2	,2	,2	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,3	,0	,2	,4	,1	,1	,4	,4	,2	,5	,4	,6	,4	,7	,6	,4	,5	,7	,9	,9	,13	,9	,3	,11	,4	,8	,13	,9	,10	,11	,4	,13	,14	,10	,9	,7	,12	,8	,10	,6	,3	,4	,1	,0	,3	,2	,2	,2	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,1	,0	,1	,1	,0	,0	,0	,2	,3	,0	,1	,4	,2	,3	,4	,9	,12	,5	,11	,12	,11	,8	,10	,10	,9	,11	,11	,11	,8	,8	,9	,9	,10	,5	,8	,14	,8	,9	,10	,6	,15	,17	,7	,8	,3	,3	,1	,0	,2	,0	,1	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,1	,0	,1	,1	,0	,1	,1	,0	,1	,1	,1	,1	,2	,1	,2	,3	,2	,4	,4	,6	,6	,8	,6	,8	,9	,11	,7	,2	,15	,7	,11	,14	,6	,9	,13	,14	,17	,19	,13	,15	,11	,10	,5	,7	,13	,10	,11	,12	,5	,5	,3	,5	,1	,1	,0	,2	,1,
			0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,2	,2	,0	,1	,3	,1	,2	,2	,1	,3	,3	,2	,4	,6	,12	,6	,8	,11	,10	,11	,11	,9	,13	,14	,15	,13	,12	,13	,16	,12	,11	,10	,16	,18	,11	,23	,12	,11	,5	,7	,12	,5	,8	,6	,2	,3	,4	,1	,0	,0,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,0	,0	,0	,0	,0	,1	,0	,1	,1	,1	,0	,2	,3	,0	,1	,2	,2	,2	,0	,4	,10	,6	,11	,4	,10	,10	,15	,9	,13	,8	,9	,11	,14	,15	,9	,15	,13	,8	,16	,19	,15	,9	,12	,10	,12	,13	,7	,10	,10	,16	,11	,9	,1	,3	,2	,1	,2	,1,
			1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,1	,1	,0	,2	,1	,1	,3	,4	,2	,1	,4	,0	,7	,7	,3	,9	,4	,6	,11	,10	,7	,10	,14	,16	,12	,8	,13	,18	,19	,17	,12	,15	,9	,12	,12	,9	,19	,14	,4	,14	,12	,10	,19	,7	,8	,6	,2	,2	,2	,2	,1	,2,
			0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,1	,0	,1	,0	,2	,0	,2	,1	,1	,2	,2	,1	,5	,1	,2	,6	,5	,5	,6	,3	,6	,5	,4	,12	,12	,10	,14	,14	,9	,9	,15	,14	,16	,12	,12	,13	,17	,16	,13	,12	,11	,7	,10	,9	,8	,9	,10	,8	,3	,3	,4	,2	,0	,1,
			0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,1	,0	,0	,0	,0	,2	,4	,0	,1	,1	,1	,2	,2	,2	,4	,1	,4	,2	,6	,5	,9	,7	,12	,13	,10	,12	,5	,7	,12	,12	,15	,15	,14	,15	,15	,20	,23	,15	,19	,15	,17	,10	,13	,15	,11	,11	,6	,7	,6	,4	,1	,3	,0	,4,
			0	,0	,0	,0	,0	,0	,1	,0	,0	,1	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,0	,2	,1	,0	,0	,2	,7	,2	,1	,1	,5	,1	,1	,2	,6	,5	,4	,3	,3	,4	,7	,14	,15	,13	,10	,17	,13	,15	,19	,11	,18	,12	,16	,23	,11	,18	,18	,18	,12	,5	,14	,14	,13	,5	,9	,5	,0	,7	,3	,3	,3	,2)
col.names	<- c("0-19","20-24",as.character(seq(25,94)),"95+")
cases	<- matrix(data=v.cases,nrow=46, ncol=73, byrow=TRUE,dimnames=list(NULL,col.names))
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=cases[,seq(3,67)]	,
				data.format		="PA"			,
				age1			=25				,
				per1			=1968			,
				unit			=1				,
				label			="women, UK mesothelioma 2013 update"))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases			,
				data.format		="PA"			,
				label			="women, UK mesothelioma 2013 update, all age groups"))
}	#	data.asbestos.2013.women


###############################
#	MOTOR DATA
###############################
data.loss.VNJ	<- function()
#	BN, 24 Apr 2015 (6 Feb 2015)
#	A Chain-Ladder with A,C effects
#
#	Taken from tables 1,2 of
#	Verrall R, Nielsen JP, Jessen AH (2010)
#	Prediction of RBNS and IBNR claims using claim amounts and claim counts
#	ASTIN Bulletin 40, 871-887
#
#	Also analysed in 
#
#	Martinez Miranda MD, Nielsen B, Nielsen JP and Verrall R (2011)
#	Cash flow simulation for a model of outstanding liabilities based on claim amounts and claim numbers
#	ASTIN Bulletin 41, 107-129
#
#	Kuang D, Nielsen B, Nielsen JP (2015)
#	The geometric chain-ladder
#	Scandinavian Acturial Journal, to appear
#
#	Data from Codan, Danish subsiduary of Royal & Sun Alliance 
#	Portfolio of motor policies: third party liability
#	Units in years
#	X Paid run-off triangle
#	N Number of reported claims
{	#	data.loss.VNJ
#	dimension
k		<- 10
#	Number of reported claims
Nvec	<- c(	6238  , 831   , 49    , 7     , 1 , 1 , 2 , 1 , 2 , 3 ,
				7773  , 1381  , 23    , 4     , 1 , 3 , 1 , 1 , 3 ,  
				10306 , 1093  , 17    , 5     , 2 , 0 , 2 , 2 ,    
				9639  , 995   , 17    , 6     , 1 , 5 , 4 ,    
				9511  , 1386  , 39    , 4     , 6 , 5 ,     
				10023 , 1342  , 31    , 16    , 9 ,  
				9834  , 1424  , 59    , 24    ,   
				10899 , 1503  , 84    ,       
				11954 , 1704  ,
				10989 )
#	X	as a vector
Xvec	<- c(   451288 , 339519 , 333371 , 144988 , 93243  , 45511  , 25217 , 20406 , 31482 , 1729 ,
			    448627 , 512882 , 168467 , 130674 , 56044  , 33397  , 56071 , 26522 , 14346 ,     
			    693574 , 497737 , 202272 , 120753 , 125046 , 37154  , 27608 , 17864 ,       
			    652043 , 546406 , 244474 , 200896 , 106802 , 106753 , 63688 ,       
			    566082 , 503970 , 217838 , 145181 , 165519 , 91313  ,      
			    606606 , 562543 , 227374 , 153551 , 132743 ,        
			    536976 , 472525 , 154205 , 150564 ,        
			    554833 , 590880 , 300964 ,          
			    537238 , 701111 ,           
			    684944 )     
return(c(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			time.adjust =1							,
			label		="loss VNJ"					),			
			list(
			counts		=vector.2.triangle(Nvec,k)	)))
}	#	data.loss.VNJ

###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.BZ	<- function()
#	BN, 24 Apr 2014 (7 Feb 2015)
#	A Loss Triangle with A,P,C effects
#
#	Taken from table 3.5 of
#   Barnett, G. and Zehnwirth, B. (2000). Best estimates for reserves.
#	Proc. Casualty Actuar. Soc. 87, 245--321.
#
#	Also analysed in 
#
#	Kuang D, Nielsen B and Nielsen JP (2011)
#	Forecasting in an extended chain-ladder-type model
#	Journal of Risk and Insurance 78, 345-359
#
#	BZ write: "loss development array with a major trend change between payment years 1984 and 1985"
#	Time in years
#	X Paid run-off triangle
{	#	data.loss.BZ
#	dimension
k		<- 11
#	X	as a vector
Xvec	<- c(   153638,	188412,	134534,	 87456,	 60348,	42404,	31238,	21252,	16622,	14440,	12200,
			    178536,	226412,	158894,	104686,	 71448,	47990,	35576,	24818,	22662,	18000,        
			    210172,	259168,	188388,	123074,	 83380,	56086,	38496,	33768,	27400,                
			    211448,	253482,	183370,	131040,	 78994,	60232,	45568,	38000,                        
			    219810,	266304,	194650,	120098,	 87582,	62750,	51000,                                
			    205654,	252746,	177506,	129522,	 96786,	82400,                                        
			    197716,	255408,	194648,	142328,	105600,                                               
			    239784,	329242,	264802,	190400,                                                       
			    326304,	471744,	375400,                                                               
			    420778,	590400,                                                                       
				496200)
Exposure	<- c(     2.2,    2.4,	   2.2,	   2.0,	   1.9,	  1.6,    1.6,    1.8,    2.2,    2.5,    2.6)				
return(c(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			coh1		=1977						,
			time.adjust =1							,
			label		="loss BZ"					),
			exposure	=Exposure					))
}	#	data.loss.BZ

###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.TA	<- function()
#	BN, 24 Apr 2015 (18 Mar 2015)
#	A Loss Triangle with A,C effects and over-dispersion
#
#	Attributed to
#	Taylor and Ashe
#
##	Analysed in
#
#	Verrall, R.J. (1991)
#	On the estimation of reserves from loglinear models
#	Insurance: Mathematics and Economics 10, 75-80
#
#	England, P., Verrall, R.J. (1999)
#	Analytic and bootstrap estimates of prediction errors in claims reserving
#	Insurance: Mathematics and Economics 25, 281-293
#
#	X Paid run-off triangle
{	#	data.loss.BZ
#	dimension
k		<- 10
#	X	as a vector
Xvec	<- c(	357848,	766940, 610542, 482940, 527326, 574398, 146342, 139950, 227229,  67948,
				352118, 884021, 933894,1183289, 445745, 320996, 527804, 266172, 425046,
				290507,1001799, 926219,1016654, 750816, 146923, 495992, 280405,
				310608,1108250, 776189,1562400, 272482, 352053, 206286,
				443160, 693190, 991983, 769488, 504851, 470639,
				396132, 937085, 847498, 805037, 705960,
				440832, 847631,1131398,1063269,
				359480,1061648,1443370,
				376686, 986608,
				344014)
return(apc.data.list(
			response	=vector.2.triangle(Xvec,k)	,
			data.format	="CL"						,
			time.adjust =1							,
			label		="loss TA"					))
}	#	data.loss.TA

################################
##	LOSS TRIANGLE DATA
################################
#data.loss.Greek	<- function()
##	BN, 29 jan 2018
##	A Loss Triangle with A,C effects, over-dispersion
##	Paid and incurred
##
##	Used and analysed in
##
##	Margraf, C. and Nielsen, B.
##	A likelihood approach to Bornhuetter-Ferguson Analysis.
##	mimeo, Nuffield College
##
##	amounts in Euros
##	
##	Paid run-off triangle, cumulative
##	Incurred run-off triangle, cumulative
#{	#	data.loss.Greek
##	dimension
#k		<- 9
##	Paid and incurred as vectors
#Paid.vec<- c(	34492471, 47124007,   55244404,   59817460,   62550940,   66042036,   69311560,   70992659,  72265079,
#				39467733, 54003286,   61349336,   69986825,   76412887,   81768759,   86684598,   90726054,          
#				38928855, 57087550,   65905902,   77128507,   84158380,   92436441,   97838371,                     
#				34202332, 50932726,   60560484,   68566905,   76409739,   82082804,                                
#				35657409, 52397264,   59849582,   66698806,   72724524,                                           
#				25404394, 37040589,   42371049,   50709319,                                                      
#				21268516, 31311410,   35973015,                                                                 
#				17404447, 27786399,                                                                             
#				17676374)                                                                                       
#Incu.vec<- c(	54018141, 56699807,   60273204,   61112600,   63729660,   67142341,   69733859,   71980196,   72738376,
#				68706483, 70534436,   70254136,   75919965,   77900147,   83401774,   88690144,   92171660,           
#				64613205, 72600950,   76163387,   82388057,   87424383,   96246891,  102854340,                       
#				58071632, 66701421,   69420629,   75280537,   81978240,   89923269,                                   
#				60368719, 67868349,   72528239,   80726223,   85339588,                                               
#				47282519, 56488940,   60896832,   65900623,                                                           
#				49905225, 54801141,   60026903,                                                                       
#				48425940, 52652928,                                                                                   
#				47449977)
##	Paid and incurred as matrices				
#Paid.mat.cum	<- vector.2.triangle(Paid.vec,k)					
#Incu.mat.cum	<- vector.2.triangle(Incu.vec,k)
##	Get incrementat triangles
#Paid.mat.inc	<- Paid.mat.cum
#Incu.mat.inc	<- Incu.mat.cum
#for(col in k:2)
#{
#	Paid.mat.inc[,col]	<- Paid.mat.inc[,col]-Paid.mat.inc[,col-1]
#	Incu.mat.inc[,col]	<- Incu.mat.inc[,col]-Incu.mat.inc[,col-1]
#}
#
#
#return(c(apc.data.list(
#			response		=Paid.mat.inc			,
#			data.format		="CL"					,
#			coh1			=2005					,
#			time.adjust 	=0						,
#			label			="loss Greek"			),
#		list(
#			paid			=Paid.mat.inc			,
#			incurred		=Incu.mat.inc			,
#			response.cum	=Paid.mat.cum			,
#			incurred.cum	=Incu.mat.cum			)
#			))
#}	#	data.loss.Greek


###############################
#	LOSS TRIANGLE DATA
###############################
data.loss.XL	<- function()
#	BN, 5 feb 2018
#	A Loss Triangle with A,C effects, log normal
#	Paid
#
#	Used and analysed in
#
#	Kuang, D. and Nielsen, B.
#	Generalized log-normal chain-ladder.
#	mimeo, Nuffield College
#
#	amounts in 1000 USD
#	
#	Paid run-off triangle, cumulative
#	Incurred run-off triangle, cumulative
{	#	data.loss.XL
#	dimension
k		<- 20
#	Paid and incurred as vectors
Paid.vec<- c(  2185 ,  13908 , 44704 , 56445 , 67313 , 62830 , 72619 , 42511 , 32246 , 51257 , 11774 , 21726 , 10926 , 4763  , 3580  , 4777  , 1070 ,  1807  , 824   , 1288 ,            
			   3004 ,  17478 , 49564 , 55090 , 75119 , 66759 , 76212 , 62311 , 31510 , 15483 , 23970 , 8321  , 15027 , 3247  , 8756  , 14364 , 3967 ,  3858  , 4643  ,                  
			   5690 ,  28971 , 55352 , 63830 , 71528 , 73549 , 72159 , 37275 , 38797 , 27264 , 28651 , 14102 , 8061  , 17292 , 10850 , 10732 , 4611 ,  4608  ,                         
			   9035 ,  29666 , 47086 , 41100 , 58533 , 80538 , 70521 , 40192 , 27613 , 13791 , 17738 , 20259 , 12123 , 6473  , 3922  , 3825  , 3082 ,                                 
			   7924 ,  38961 , 41069 , 64760 , 64069 , 61135 , 62109 , 52702 , 36100 , 18648 , 32572 , 17751 , 18347 , 10895 , 2974  , 5828  ,                                       
			   7285 ,  25867 , 44375 , 58199 , 61245 , 48661 , 57238 , 29667 , 34557 , 8560  , 12604 , 8683  , 9660  , 4687  , 1889  ,                                             
			   3017 ,  22966 , 62909 , 54143 , 72216 , 58050 , 29522 , 25245 , 19974 , 16039 , 8083  , 9594  , 3291  , 2016  ,                                                   
			   1752 ,  25338 , 56419 , 75381 , 64677 , 58121 , 38339 , 21342 , 14446 , 13459 , 6364  , 6326  , 6185  ,                                                           
			   1181 ,  24571 , 66321 , 65515 , 62151 , 43727 , 29785 , 23981 , 12365 , 12704 , 12451 , 8272  ,                                                               
			   1706 ,  13203 , 40759 , 57844 , 48205 , 50461 , 27801 , 21222 , 14449 , 10876 , 8979  ,                                                                    
			   623  ,  14485 , 27715 , 52243 , 60190 , 45100 , 31092 , 22731 , 19950 , 18016 ,                       
			   338  ,  6254  , 24473 , 32314 , 35698 , 25849 , 30407 , 15335 , 15697 ,                                     
			   255  ,  3842  , 14086 , 26177 , 27713 , 15087 , 17085 , 12520 ,                                         
			   258  ,  7426  , 22459 , 28665 , 32847 , 28479 , 24096 ,                                               
			   1139 ,  10300 , 19750 , 32722 , 41701 , 29904 ,                                                    
			   381  ,  5671  , 34139 , 33735 , 33191 ,                                                           
			   605  ,  11242 , 24025 , 32777 ,                                                               
			   1091 ,  9970  , 31410 ,                                         
			   1221 ,  8374  ,                                               
			   2458 )                                                        
#	Paid as matrices				
Paid.mat	<- vector.2.triangle(Paid.vec,k)					
return(apc.data.list(
			response		=Paid.mat				,
			data.format		="CL"					,
			age1			=1997					,
			coh1			=1997					,
			time.adjust 	=1997					,
			label			="loss, US casualty, XL Group"	)
			)
}	#	data.loss.XL


###############################
#	AIDS reports in England and Wales
###############################
data.aids	<- function(all.age.groups=FALSE)
#	BN, 7 Feb 2016
#	Numbers of AIDS reports in England and Wales to the end of 1992 by quarter
#	
#	Attributed to
#	De Angelis and Gilks (1994)
#
#	Analysed in
#
#	Davison, A.C. and Hinkley, D.V. (1997) Bootstrap methods and their applications, Cambridge UP
{	#	data.aids
	#	data for coh 1983:3 to 1989:1
	v.cases		<- c(	2,	6,	0,	1,	1,	0,	0,	1,	0,	0,	0,	0,	0,	0,	1,
						2,	7,	1,	1,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
						4,	4,	0,	1,	0,	2,	0,	0,	0,	0,	2,	1,	0,	0,	0,
						0, 10,	0,	1,	1,	0,	0,	0,	1,	1,	1,	0,	0,	0,	0,
						6, 17,	3,	1,	1,	0,	0, 	0,	0,	0,	0,	1,	0,	0,	1,
						5, 22, 	1,	5,	2,	1,	0,	2,	1,	0,	0,	0,	0,	0,	0,
						4, 23, 	4,	5, 	2,	1,	3,	0,	1,	2,	0,	0,	0,	0,	2,
					   11, 11, 	6,	1,	1,	5,	0,	1,	1,	1,	1,	0,	0,	0,	1,
					    9, 22,	6,	2,	4,	3,	3,	4,	7,	1,	2,	0,	0,	0,	0,
						2, 28,	8,	8,	5,	2,	2,	4,	3,	0,	1,	1,	0,	0,	1,
						5, 26, 14,	6,	9,	2,	5,	5,	5,	1,	2,	0,	0,	0,	2,
						7, 49, 17, 11, 	4, 	7, 	5, 	7,	3,	1,	2,	2, 	0,	1,	4,
					   13, 37, 21,	9,	3,	5, 	7,	3,	1,	3,	1,	0,	0,	0,	6,
					   12, 53, 16, 21,	2,	7,	0,	7,	0,	0,	0,	0,	0,	1,	1,
					   21, 44, 29, 11,	6,	4,	2,	2,	1,	0,	2,	0,	2,	2, 	8,
					   17, 74, 13, 13,	3,	5,	3,	1,	2,	2,	0,	0,	0,	3,	5,
					   36, 58, 23, 14,  7,	4,	1,	2,	1,	3,	0,	0,	0,	3,	1,
					   28, 74, 23, 11,	8,	3,	3,	6,	2,	5,	4,	1,	1,	1,	3,
					   31, 80, 16,	9,	3,	2,	8,	3,	1,	4,	6,	2,	1,	2,	6,
					   26, 99, 27, 	9,	8, 11,	3,	4,	6,	3,	5,	5,	1,	1, 	3,
					   31, 95, 35, 13, 18,	4,	6,	4,	4,	3,	3,	2,	0,	3,	3,
					   36, 77, 20, 26, 11,	3,	8,	4,	8,	7,	1,	0,	0,	2,	2,
					   32, 92, 32, 10, 12, 19, 12,	4,	3,	2, 	0,	2,	2, 	0,	2,
					   15, 92, 14, 27, 22, 21, 12,	5,	3,	0,	3,	3,	0,	1,	1,
					   34,104, 29, 31, 18,	8,	6,	7, 	3,	8,	0,	2,	1,	2, NA, 
					   38,101, 34, 18, 	9, 15, 	6,	1,	2,	2,	2,	3,	2, NA, NA, 
					   31,124, 47, 24, 11, 15,	8,	6,	5,	3,	3,	4, NA, NA, NA, 
					   32,132, 36, 10,	9,	7,	6,	4,	4,	5,	0, NA, NA, NA, NA, 
					   49,107, 51, 17, 15,  8,	9,	2,	1,	1, NA, NA, NA, NA, NA, 
					   44,153, 41, 16, 11,	6,	5,	7,	2, NA, NA, NA, NA, NA, NA, 
					   41,137, 29, 33,	7, 11,	6,	4,	3, NA, NA, NA, NA, NA, NA, 
					   56,124, 39, 14, 12, 	7, 10,	1, NA, NA, NA, NA, NA, NA, NA, 
					   53,175, 35, 17, 13, 11, 	2, NA, NA, NA, NA, NA, NA, NA, NA, 
					   63,135, 24, 23, 12,	1, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   71,161, 48, 25, 	5, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   95,178, 39,	6, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
					   76,181, 16, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA,
					   67, 66, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA)
col.names	<- c("0*",as.character(seq(1,13)),"14+")
cases.all	<- matrix(data=v.cases,nrow=38, ncol=15, byrow=TRUE,dimnames=list(NULL,col.names))
cases.clean	<- cases.all
for(row in 0:7)
	cases.clean[38-row,2+row]	<- NA
if(all.age.groups==FALSE)
	return(apc.data.list(
				response		=t(cases.clean)		,
				data.format		="trap"				,
				age1			=0					,
				coh1			=1983.5				,
				unit			=1/4				,
				per.zero		=0					,
				per.max			=38					,
				label			="UK AIDS - clean"	,
				))
if(all.age.groups==TRUE)
	return(apc.data.list(
				response		=cases.all			,
				data.format		="CA"				,
				age1			=0					,
				coh1			=1983.5				,
				unit			=1/4				,
				label			="UK AIDS - all: last column reporting delay >= 14, last diagonal: incomplete count",
				))	
}	#	data.aids
# 	apc.fit.table(data.Aids(),"od.poisson.response")
# 	fit <- apc.fit.model(data.Aids(),"poisson.response","AC")
#	forecast <- apc.forecast.ac(fit)
#	data.sums.coh <- apc.data.sums(data.Aids())$sums.coh
#	forecast.total <- forecast$response.forecast.coh
#	forecast.total[,1]	<- forecast.total[,1]+data.sums.coh[26:38]
#	plot(seq(1983.5,1992.75,by=1/4),data.sums.coh,xlim=c(1988,1993),ylim=c(200,600),main="Davison, Hinkley, Fig 7.6, parametric version")
#	apc.polygon(forecast.total,x.origin=1989.5,unit=1/4)


###############################
#	2-SAMPLE DATA from Riebler and Held 2010
###############################
#	BN, 17 September 2016
#	from Riebler and Held 2010

data.RH.mortality.dk	<- function()
{	#	data.RH.mortality.dk
v.dk.counts <- c(
3890 ,	3201 ,	2182 ,	1521 ,	1118 ,	1193 ,	1092 ,	895	 ,
272	 ,	309	 ,  311	 ,  241	 ,  158	 ,  119	 ,  105	 ,  74	 ,
201	 ,	246	 ,  227	 ,  193	 ,  186	 ,  128	 ,  108	 ,  95	 ,
373	 ,	390	 ,  372	 ,  336	 ,  331	 ,  282	 ,  240	 ,  185	 ,
335	 ,	391	 ,  429	 ,  387	 ,  374	 ,  375	 ,  308	 ,  267	 ,
423	 ,	424	 ,  483	 ,  462	 ,  488	 ,  434	 ,  415	 ,  361	 ,
672	 ,	580	 ,  614	 ,  719	 ,  748	 ,  663	 ,  555	 ,  585	 ,
1101 ,	1036 ,  901	 ,  892	 ,  1183 ,  1062 ,  996	 ,  942	 ,
1675 ,	1708 ,  1579 ,  1464 ,  1537 ,  1891 ,  1690 ,  1582 ,
2420 ,	2555 ,  2612 ,  2360 ,  2382 ,  2435 ,  2806 ,  2552 ,
3652 ,	3666 ,  3954 ,  3736 ,  3605 ,  3621 ,  3660 ,  4066 ,
5215 ,	5486 ,  5349 ,  5475 ,  5475 ,  5447 ,  5175 ,  5037 ,
7475 ,	7786 ,  8032 ,  7641 ,  8059 ,  8037 ,  7666 ,  7433 ,
10744, 	11341,	11028,	11042,	10928,	11419,	11504,	11021,
14707, 	15671,	15278,	15431,	15989,	15594,	16452,	15779,
18286,	18431,	19037,	20005,	20838,	21795,	21649,	21481,
17340,	18713,	19073,	21094,	24073,	25564,	27956,	26699)

v.dk.pop <- c(
909800 ,984000,	902510,	846103,	710529,	661516,	758503 ,837141 ,
901700 ,911300,	984994,	905633,	850189,	715153,	671309 ,775431 ,
956100 ,903300,	912269,	986488,	908632,	852904,	723344 ,686158 ,
1006100,954200,	904464,	914379,	990757,	914700,	862084 ,738989 ,
797500 ,996200,	957068,	909483,	920173,	997808,	928639 ,892000 ,
713900 ,792000,	992707,	954616,	908331,	922223,	1004548,954111 ,
709700 ,712400,	792252,	990075,	952219,	907818,	926494 ,1021125,
760700 ,707000,	711888,	790006,	986867,	950211,	909956 ,939409 ,
779000 ,756200,	702990,	707044,	785013,	981000,	947270 ,914657 ,
756200 ,768100,	746815,	693291,	697918,	775923,	971539 ,943327 ,
756500 ,741500,	752512,	730916,	678757,	683613,	762084 ,959231 ,
685900 ,733700,	719039,	729062,	707870,	656576,	662825 ,744259 ,
608400 ,655500,	701139,	686911,	695512,	674532,	626303 ,635704 ,
507700 ,562200,	609373,	654229,	640884,	647717,	628252 ,582828 ,
386200 ,441800,	496205,	543840,	586717,	575366,	580188 ,561017 ,
268600 ,301300,	356068,	408000,	453701,	492961,	483697 ,486525 ,
145500 ,173400,	207032,	253754,	297238,	336778,	369061 ,361484 )

col.names	<- c(paste(as.character(seq(1960,1995,by=5)),as.character(seq(1964,1999,by=5)),sep="-"))
row.names	<- c(paste(as.character(seq(0,80,by=5)),as.character(seq(4,84,by=5)),sep="-"))

return(apc.data.list(
			response 	= matrix(data=v.dk.counts, nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			dose 		= matrix(data=v.dk.pop	 , nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			data.format	="AP"			,
			age1		=0				,
			per1		=1960			,
			unit		=5				,
			label		="RH mortality Denmark"))
}	#	data.RH.mortality.dk

data.RH.mortality.no	<- function()
{	#	data.RH.mortality.no
v.no.counts <- c(
2840 ,	2557 ,	1878 ,	1354 ,	1093 ,	1127 ,	950  ,	647  ,
246	 ,  228	 ,  242	 ,  184	 ,  116	 ,  102	 ,  106	 ,  83	 ,
175	 ,  174	 ,  162	 ,  171	 ,  128	 ,  95	 ,  85	 ,  73	 ,
230	 ,  280	 ,  247	 ,  256	 ,  261	 ,  251	 ,  205	 ,  199	 ,
209	 ,  254	 ,  247	 ,  273	 ,  229	 ,  276	 ,  234	 ,  209	 ,
214	 ,  219	 ,  289	 ,  289	 ,  254	 ,  286	 ,  303	 ,  281	 ,
355	 ,  267	 ,  320	 ,  350	 ,  360	 ,  403	 ,  410	 ,  400	 ,
627	 ,  509	 ,  438	 ,  416	 ,  563	 ,  558	 ,  566	 ,  559	 ,
1027 ,  913	 ,  757	 ,  627	 ,  658	 ,  910	 ,  912	 ,  899	 ,
1500 ,  1576 ,  1398 ,  1110 ,  982	 ,  1142 ,  1369 ,  1382 ,
2172 ,  2213 ,  2262 ,  2116 ,  1665 ,  1546 ,  1575 ,  2129 ,
3338 ,  3363 ,  3278 ,  3386 ,  3005 ,  2668 ,  2256 ,  2335 ,
5162 ,  5207 ,  4909 ,  4943 ,  4893 ,  4543 ,  3869 ,  3262 ,
7574 ,  7985 ,  7666 ,  7668 ,  7272 ,  7340 ,  6686 ,  5452 ,
10690,	11575,	12336,	11289,	11184,	11361,	11279,	9669 ,
13580,	14392,	15794,	16700,	16163,	16759,	16829,	16135,
14362,	14942,	16533,	18298,	19524,	20959,	22093,	22035)

v.no.pop <- c(
748876,	783556,	789040,	692243,	623821,	639147,	715764,	736098,
746805,	748073,	785553,	795060,	698077,	631011,	642790,	726749,
746991,	748742,	749658,	789153,	798502,	702194,	639877,	650683,
712240,	741735,	749907,	751872,	793545,	804705,	721477,	650225,
545465,	719251,	736965,	750894,	757556,	804345,	817266,	738758,
488599,	543814,	713489,	740375,	756612,	769602,	810871,	838049,
529956,	482970,	542226,	716570,	743565,	764545,	776659,	824076,
603561,	521154,	481973,	544204,	717998,	746045,	763994,	783355,
648664,	589043,	518559,	481586,	543408,	717093,	747004,	766289,
613973,	646268,	584627,	515283,	478966,	540241,	695692,	745169,
581830,	605701,	637047,	577247,	509106,	472977,	524105,	688829,
536976,	571711,	592874,	623648,	565379,	498482,	464017,	515832,
490135,	518834,	552400,	573254,	603628,	546792,	488078,	451985,
408079,	462531,	489158,	522196,	543438,	573257,	528830,	466228,
310512,	367706,	414868,	443133,	476199,	498316,	524441,	489160,
219036,	252507,	299993,	343106,	374704,	407047,	426695,	456943,
129532,	148848,	174326,	212774,	251131,	280834,	306362,	330722)

col.names	<- c(paste(as.character(seq(1960,1995,by=5)),as.character(seq(1964,1999,by=5)),sep="-"))
row.names	<- c(paste(as.character(seq(0,80,by=5)),as.character(seq(4,84,by=5)),sep="-"))

return(apc.data.list(
			response 	= matrix(data=v.no.counts, nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			dose 		= matrix(data=v.no.pop	 , nrow =17, ncol=8, byrow=TRUE, dimnames=list(row.names,col.names)) ,
			data.format	="AP"			,
			age1		=0				,
			per1		=1960			,
			unit		=5				,
			label		="RH mortality Norway"))
}	#	data.RH.mortality.no
