% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computePES}
\alias{computePES}
\title{A Vector Summary of the Persistent Entropy Summary Function}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N}, \code{computePES()} vectorizes the persistent entropy summary (PES) function \deqn{S(t)=-\sum_{i=1}^N \frac{l_i}{L}\log_2{(\frac{l_i}{L}})\bold 1_{[b_i,d_i]}(t),} where \eqn{l_i=d_i-b_i} and \eqn{L=\sum_{i=1}^Nl_i}. Points of \eqn{D} with infinite death value are ignored
}

\usage{
computePES(D, homDim, scaleSeq)
}

\arguments{
\item{D}{matrix with three columns containing the dimension, birth and death values respectively}

\item{homDim}{homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.)}

\item{scaleSeq}{numeric vector of increasing scale values used for vectorization}

}

\value{
A numeric vector whose elements are the average values of the persistent entropy summary function computed between each pair of consecutive scale points of \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}}:
\deqn{\Big(\frac{1}{\Delta t_1}\int_{t_1}^{t_2}S(t)dt,\frac{1}{\Delta t_2}\int_{t_2}^{t_3}S(t)dt,\ldots,\frac{1}{\Delta t_{n-1}}\int_{t_{n-1}}^{t_n}S(t)dt\Big),}
where \eqn{\Delta t_k=t_{k+1}-t_k}
}

\author{Umar Islambekov}

\references{
1. Atienza, N., Gonzalez-Díaz, R., & Soriano-Trigueros, M. (2020). On the stability of persistent entropy and new summary functions for topological data analysis. Pattern Recognition, 107, 107509.
}

\examples{
N <- 100 
set.seed(123)
# sample N points uniformly from unit circle and add Gaussian noise
X <- TDA::circleUnif(N,r=1) + rnorm(2*N,mean = 0,sd = 0.2)

# compute a persistence diagram using the Rips filtration built on top of X
D <- TDA::ripsDiag(X,maxdimension = 1,maxscale = 2)$diagram 

scaleSeq = seq(0,2,length.out=11) # sequence of scale values

# compute PES for homological dimension H_0
computePES(D,homDim=0,scaleSeq)

# compute PES for homological dimension H_1
computePES(D,homDim=1,scaleSeq)
}
