% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_fields.R
\name{check_fields}
\alias{check_fields}
\title{Validate SHARK data fields for a given datatype}
\usage{
check_fields(
  data,
  datatype,
  level = "error",
  stars = 1,
  bacterioplankton_subtype = "abundance",
  field_definitions = .field_definitions
)
}
\arguments{
\item{data}{A data frame containing SHARK data to be validated.}

\item{datatype}{A string giving the SHARK datatype to validate against.
Must exist as a name in the provided \code{field_definitions}.}

\item{level}{Character string, either \code{"error"} or \code{"warning"}.
If \code{"error"}, only required fields are validated. If \code{"warning"},
recommended fields are also checked and reported as warnings.}

\item{stars}{Integer. Maximum number of "\emph{" levels to include.
Default = 1 (only single "}").
For example, \code{stars = 2} includes "\emph{" and "**",
\code{stars = 3} includes "}", "\strong{", and "}*".}

\item{bacterioplankton_subtype}{Character. For "Bacterioplankton" only: either
"abundance" (default) or "production". Ignored for other datatypes.}

\item{field_definitions}{A named list of field definitions. Each element
should contain two character vectors: \code{required} and \code{recommended}.
Defaults to the package's built-in \code{SHARK4R:::.field_definitions}.
Alternatively, the latest definitions can be loaded directly from the
official SHARK4R GitHub repository using
\code{\link{load_shark4r_fields}()}.}
}
\value{
A tibble with the following columns:
\describe{
\item{level}{Either \code{"error"} or \code{"warning"}.}
\item{field}{The name of the field that triggered the check.}
\item{row}{Row number(s) in \code{data} where the issue occurred, or \code{NA}
if the whole field is missing.}
\item{message}{A descriptive message explaining the problem.}
}
The tibble will be empty if no problems are found.
}
\description{
This function checks a SHARK data frame against the required and recommended
fields defined for a specific datatype. It verifies that all required fields
are present and contain non-empty values. If \code{level = "warning"}, it
also checks for recommended fields and empty values within them.
}
\details{
Note: A single "*" marks required fields in the standard SHARK template.
A double "**" is often used to specify columns required for \strong{national monitoring only}.
For more information, see:
https://www.smhi.se/data/hav-och-havsmiljo/datavardskap-oceanografi-och-marinbiologi/leverera-data

Field definitions for SHARK data can be loaded in two ways:
\enumerate{
\item \strong{From the SHARK4R package bundle (default):}
The package contains a built-in object, \code{.field_definitions},
which stores required and recommended fields for each datatype.

\item \strong{From GitHub (latest official version):}
To use the most up-to-date field definitions, you can load them directly from the
\href{https://github.com/nodc-sweden/SHARK4R-statistics}{SHARK4R-statistics} repository:
\preformatted{
    defs <- load_shark4r_fields()
    check_fields(my_data, "Phytoplankton", field_definitions = defs)
    }
}

\strong{Delivery-format (all-caps) data:}
If the column names in \code{data} are all uppercase (e.g. SDATE), \code{check_fields()} assumes
the dataset follows the official SHARK delivery template. In this case:
\itemize{
\item Required fields are determined from the delivery template using
\code{\link{get_delivery_template}()} and \code{\link{find_required_fields}()}.
\item Recommended fields are ignored because the delivery templates do not define them.
\item The function validates that all required columns exist and contain non-empty values.
}
This ensures that both internal \code{SHARK4R} datasets (with camelCase or snake_case columns)
and official delivery files (ALL_CAPS columns) are validated correctly using the appropriate rules.

Stars in the template

Leading asterisks in the delivery template indicate required levels:

\itemize{
\item \emph{*} = standard required column
\item * = required for national monitoring
\item Other symbols = additional requirement level
}

The \code{stars} parameter in \code{check_fields()} controls how many levels of required
columns to include.
}
\examples{
# Example 1: Using built-in field definitions for "Phytoplankton"
df_phyto <- data.frame(
  visit_date = "2023-06-01",
  sample_id = "S1",
  scientific_name = "Skeletonema marinoi",
  value = 123
)

# Check fields
check_fields(df_phyto, "Phytoplankton", level = "warning")

\donttest{
# Example 2: Load latest definitions from GitHub and use them
defs <- load_shark4r_fields(verbose = FALSE)

# Check fields using loaded field definitions
check_fields(df_phyto, "Phytoplankton", field_definitions = defs)
}

# Example 3: Custom datatype with required + recommended fields
defs <- list(
  ExampleType = list(
    required = c("id", "value"),
    recommended = "comment"
  )
)

# Example data
df_ok <- data.frame(id = 1, value = "x", comment = "ok")

# Check fields using custom field definitions
check_fields(df_ok, "ExampleType", level = "warning", field_definitions = defs)

}
\seealso{
\code{\link{load_shark4r_fields}} for fetching the latest field definitions from GitHub,
\code{\link{get_delivery_template}} for downloading delivery templates from SMHI's website.
}
