#' Visualize Walks
#'
#' @family Visualization Functions
#'
#' @author Antti Lennart Rask
#'
#' @description visualize_walks() visualizes the output of the random walk
#' functions in the RandomWalker package, resulting in one or more ggplot2 plots
#' put together in a patchwork composed of 1 or more patches.
#'
#' @details visualize_walks() generates visualizations of the random walks
#' generated by the  random walk functions in the RandomWalker package. These
#' are the functions at the moment of writing:
#' - brownian_motion()
#' - discrete_walk()
#' - geometric_brownian_motion()
#' - random_normal_drift_walk()
#' - random_normal_walk()
#' - rw30()
#'
#' It is possible there are more when you read this, but you can check the rest
#' of the documentation for the current situation.
#'
#' The visualization function is meant to be easy to use. No parameters needed,
#' but you can set the `.alpha` if the default value of 0.7 isn't to your
#' liking.
#'
#' You can combine this function with many tidyverse functions (either before or
#' after). There's one example below.
#'
#' @param .data The input data. Assumed to be created by one of the random walk
#' functions in the RandomWalker package, but can be any data frame or tibble
#' that contains columns `walk_number`, `x`, and one or more numeric columns
#' like `x`, `cum_sum`, `cum_prod`, `cum_min`, `cum_max` and `cum_mean`, for
#' instance.
#' @param .alpha The alpha value for all the line charts in the visualization.
#' Values range from 0 to 1. Default is 0.7.
#'
#' @return A patchwork composed of 1 or more patches
#'
#' @examples
#' # Generate random walks and visualize the result
#' set.seed(123)
#' rw30() |>
#'  visualize_walks()
#'
#' # Set the alpha value to be other than the default 0.7
#' set.seed(123)
#' rw30() |>
#'  visualize_walks(.alpha = 0.5)
#'
#' # Use the function with an input that has alternatives for y
#' set.seed(123)
#' random_normal_walk() |>
#'  visualize_walks()
#'
#' # Use the pluck function from purrr to pick just one visualization
#' set.seed(123)
#' random_normal_walk() |>
#'  visualize_walks() |>
#'  purrr::pluck(2)
#'
#' @name visualize_walks
NULL
#' @rdname visualize_walks
#' @export
visualize_walks <- function(.data, .alpha = 0.7) {

  # Retrieve the attributes of the data
  atb <- attributes(.data)

  # Function to generate a plot for a given variable
  generate_plot <- function(y_var) {

    # Convert y_label to a pretty format if it's not "y"
    y_label_pretty <- if (y_var == "y") "y" else convert_snake_to_title_case(y_var)

    # Create a ggplot object
    ggplot2::ggplot(.data, ggplot2::aes(x = x, y = get(y_var), color = walk_number)) +
      # Plot lines with some transparency
      ggplot2::geom_line(alpha = .alpha) +
      # Use a minimal theme
      ggplot2::theme_minimal() +
      # Remove the legend
      ggplot2::theme(legend.position = "none") +
      # Set the labels for the plot
      ggplot2::labs(title = y_label_pretty, x = "Step", y = NULL)
  }

  # Identify variables to plot, excluding 'walk_number' and 'x'
  plot_vars <- setdiff(atb$names, c("walk_number", "x"))

  # Generate a list of plots for each variable
  plots <- lapply(plot_vars, generate_plot)

  # Generate a subtitle for the combined plot based on the function name
  .subtitle <- paste0("Function: ", if (atb$fns == "rw30") "rw30" else convert_snake_to_title_case(atb$fns))

  # Generate the plot caption dynamically based on attributes
  .caption <- generate_caption(atb)

  # Define annotations for the combined plot
  plot_annotations <- patchwork::plot_annotation(
    title    = paste0(atb$num_walks, " Random Walks"),
    subtitle = .subtitle,
    caption  = .caption
  )

  # Combine the individual plots into a single plot, or return the single plot with annotations
  combined_plot <- if (length(plots) > 1) {
    patchwork::wrap_plots(plots) + plot_annotations
  } else {
    plots[[1]] + plot_annotations
  }

  # Return the final combined plot
  return(combined_plot)
}
