% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/search.shift.R
\name{search.shift}
\alias{search.shift}
\title{Locating shifts in phenotypic evolutionary rates}
\usage{
search.shift(RR, status.type = c("clade", "sparse"),node = NULL, state
  = NULL, cov = NULL, nrep = 1000, f = NULL)
}
\arguments{
\item{RR}{an object fitted by the function \code{\link{RRphylo}}.}

\item{status.type}{whether the \code{"clade"} or \code{"sparse"} condition
must be tested.}

\item{node}{under the \code{"clade"} condition, the node/s (clades) to be
tested for the rate shift. If \code{node} is left unspecified, the function
performs under the 'auto-recognize' feature, meaning it will automatically
test individual clades for deviation of their rates from the background
rate of the rest of the tree (see details).}

\item{state}{the named vector of states for each tip, to be provided under
the \code{"sparse"} condition.}

\item{cov}{the covariate vector to be indicated if its effect on rate values
must be accounted for. Contrary to \code{\link{RRphylo}}, \code{cov} needs to be
as long as the number of tips of the tree.}

\item{nrep}{the number of simulations to be performed for the rate shift
test, by default \code{nrep} is set at 1000.}

\item{f}{the size of the smallest clade to be tested. By default, nodes
subtending to one tenth of the tree tips are tested.}
}
\value{
Under \code{"clade"} case without specifying nodes (i.e.
  'auto-recognize') a list including:

\strong{$all.clades} for each detected node, the data-frame includes
  the average rate difference (computed as the mean rate over all branches
  subtended by the node minus the average rate for the rest of the tree) and
  the probability that it do represent a real shift. Probabilities are
  contrasted to simulations shuffling the rates across the tree branches for
  a number of replicates specified by the argument \code{nrep}. Note that the
  p-values refer to the number of times the real average rates are larger (or
  smaller) than the rates averaged over the rest of the tree, divided by the
  number of simulations. Hence, large rates are significantly larger than the
  rest of the tree (at alpha = 0.05), when the probability is > 0.975; and
  small rates are significantly small for p < 0.025.

\strong{$single.clades} the same as with 'all.clades' but restricted
  to the largest/smallest rate values along a single lineage (i.e. nested
  clades with smaller rate shifts are excluded).

Under \code{"clade"} condition by specifying the \code{node}
  argument:

\strong{$all.clades.together} if more than one node is tested, this
  specifies the average rate difference and the significance of the rate
  shift, by considering all the specified nodes as evolving under a single
  rate. As with the 'auto-recognize' feature, large rates are significantly
  larger than the rest of the tree (at alpha = 0.05), when the probability is
  > 0.975; and small rates are significantly small for p < 0.025.

\strong{$single.clades} gives the significance for individual clades
  tested individually against the rest of the tree (\strong{$singles}) and by
  excluding the rate values of other shifting clades from the rate vector of the rest of
  the tree (\strong{$no.others})

Under the \code{"sparse"} condition:

\strong{$state.results} for each state, the data-frame includes the
  average rate difference (computed as the mean rate over all leaves evolving
  under a given state, minus the average rate for each other state or the
  rest of the tree) and the probability that the shift is real. Large rates
  are significantly larger (at alpha = 0.05), when the probability is >
  0.975; and small rates are significantly small for p < 0.025. States are
  compared pairwise.

Under all circumstances, if \code{'cov'} values are provided to the
  function, \code{search.shift} returns as \strong{$rates} object the vector of
  residuals of \code{\link{RRphylo}} rates versus \code{cov} regression.

The output always has an attribute "Call" which returns an unevaluated call to the function.
}
\description{
The function \code{search.shift} (\cite{Castiglione et al.
  2018}) tests whether individual clades or group of tips dispersed through
  the phylogeny evolve at different \code{\link{RRphylo}} rates as compared
  to the rest of the tree.
}
\details{
Under the 'auto-recognize' mode, \code{search.shift} automatically
tests individual clades (ranging in size from one half of the tree down to
\code{f} tips) for deviation of their rates from the background rate of the
rest of the tree. An inclusive clade with significantly high rates is likely
to include descending clades with similarly significantly high rates. Hence,
under 'auto-recognize' \code{search.shift} scans clades individually and
selects only the node subtending to the highest difference in mean absolute
rates as compared to the rest of the tree. If the argument \code{node}
(\code{"clade"} condition) is provided, the function computes the difference
between mean rate values of each clade and the rest of the tree, and compares
it to a random distribution of differences generated by shuffling rates
across tree branches. Additionally, if more than one \code{node} is
indicated, the rate difference for one clade is additionally computed by
excluding the rate values of the others from the rate vector of the rest of
the tree. Also, all the clades are considered as to be under a common rate
regime and compared as a single group to the rest of the tree.
}
\examples{
\dontrun{
data("DataOrnithodirans")
DataOrnithodirans$treedino->treedino
DataOrnithodirans$massdino->massdino
DataOrnithodirans$statedino->statedino
cc<- 2/parallel::detectCores()

RRphylo(tree=treedino,y=massdino,clus=cc)->dinoRates

# Case 1. Without accounting for the effect of a covariate

# Case 1.1 "clade" condition
# with auto-recognize
search.shift(RR=dinoRates,status.type="clade")->SSauto
# testing two hypothetical clades
search.shift(RR=dinoRates,status.type="clade",node=c(696,746))->SSnode

# Case 1.2 "sparse" condition
# testing the sparse condition.
search.shift(RR=dinoRates,status.type= "sparse",state=statedino)->SSstate


# Case 2. Accounting for the effect of a covariate

# Case 2.1 "clade" condition
search.shift(RR=dinoRates,status.type= "clade",cov=massdino)->SSauto.cov

# Case 2.2 "sparse" condition
search.shift(RR=dinoRates,status.type="sparse",state=statedino,cov=massdino)->SSstate.cov
    }
}
\references{
Castiglione, S., Tesone, G., Piccolo, M., Melchionna, M.,
  Mondanaro, A., Serio, C., Di Febbraro, M., & Raia, P.(2018). A new method
  for testing evolutionary rate variation and shifts in phenotypic evolution.
  \emph{Methods in Ecology and Evolution}, 9:
  974-983.doi:10.1111/2041-210X.12954
}
\seealso{
\href{../doc/search.shift.html}{\code{search.shift} vignette}

\code{\link{overfitSS}}; \href{../doc/overfit.html#overfitSS}{\code{overfitSS} vignette}

\code{\link{plotShift}}; \href{../doc/Plotting-tools.html#plotShift}{\code{plotShift} vignette}
}
\author{
Pasquale Raia, Silvia Castiglione, Carmela Serio, Alessandro
  Mondanaro, Marina Melchionna, Mirko Di Febbraro, Antonio Profico, Francesco
  Carotenuto
}
