% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/characterize_gaussian_fits.R
\name{characterize_gaussian_fits}
\alias{characterize_gaussian_fits}
\title{Characterize the orientation of fitted 2D-Gaussians}
\usage{
characterize_gaussian_fits(
  fit_objects_list = NULL,
  data = NULL,
  constrain_amplitude = FALSE,
  ...
)
}
\arguments{
\item{fit_objects_list}{A list of outputs from \code{fit_gaussian_2D()}. See
Details for more. This is the preferred input object for this function.}

\item{data}{A data.frame that contains the raw data (generally rectilinearly
gridded data, but this is not a strict requirement). Columns must be named
\code{"X_values"}, \code{"Y_values"} and \code{"response"}. See
\code{fit_gaussian_2D()} for details.}

\item{constrain_amplitude}{Default FALSE; should the amplitude of the
Gaussian be set to the maximum value of the \code{"response"} variable
(\code{TRUE}), or should the amplitude fitted by \code{stats::nls()}
(\code{FALSE})? See \code{fit_gaussian_2D()} for details.}

\item{...}{Additional arguments that can be passed to
\code{fit_gaussian_2D()} if data are supplied.}
}
\value{
A list with the following:
\itemize{
\item{"model_comparison"} {A model comparison output (i.e. what is produced
by \code{compare_gaussian_fits()}), which indicates the relative preference
of each of the three models.}
\item{"Q_table"} {A data.frame that provides information on the value of Q
from the best-fitting model, along with the 5-95\% confidence intervals of
this estimate.}
\item{"r_i"} {A numeric, the correlation of the data with the independent
(Q = -1) prediction.}
\item{"r_s"} {A numeric, the correlation of the data with the diagonally-
oriented (Q = 0) prediction.}
\item{"r_is"} {A numeric, the correlation between the independent
(Q = -1) prediction and the the diagonally-oriented (Q = 0) prediction.}
\item{"R_indp"} {A numeric, partial correlation of the response variable
with the independent (Q = -1) prediction.}
\item{"R_diag"} {A numeric, partial correlation of the response variable
with the diagonally-oriented (Q = 0) prediction.}
\item{"ZF_indp"} {A numeric, the Fisher Z-transform of the R_indp
coefficient. See Winship et al. 2006 for details.}
\item{"ZF_diag"} {A numeric, the Fisher Z-transform of the R_diag
coefficient. See Winship et al. 2006 for details.}
\item{"Z_diff"} {A numeric, the Z-difference between ZF_indp and ZF_diag.
See Winship et al. 2006 for details.}

}
}
\description{
The orientation and partial correlations of Gaussian data are analyzed
according to Levitt et al. 1994 and Priebe et al. 2003. Features include
computation of partial correlations between response variables and
independent and diagonally-tuned predictions, along with Z-difference
scoring.
}
\details{
This function accepts either a list of objects output from
\code{fit_gaussian_2D()} (preferred) or a data.frame that contains the raw
data.

The supplied fit_objects_list must be a list that contains objects returned
by \code{fit_gaussian_2D()}. This list must contain exactly three models. All
three models must have been run using \code{method = "elliptical_log"}. The
models must be: 1) one in which orientation is unconstrained, 2) one in which
orientation is constrained to Q = 0 (i.e. a diagonally-oriented Gaussian),
and 3) one in which orientation is constrained to Q = -1 (i.e. a
horizontally-oriented Gaussian). See this function's Examples for guidance.

Should raw data be provided instead of the fit_objects_list, the
\code{characterize_gaussian_fits()} runs \code{fit_gaussian_2D()} internally.
This is generally not recommended, as difficulties in fitting models via
\code{stats::nls()} are more easily troubleshot by the optional arguments in
\code{fit_gaussian_2D()}. Nevertheless, supplying raw data instead of a list
of fitted models is feasible, though your mileage may vary.
}
\examples{
if (interactive()) {
  library(gaussplotR)

  ## Load the sample data set
  data(gaussplot_sample_data)

  ## The raw data we'd like to use are in columns 1:3
  samp_dat <-
    gaussplot_sample_data[,1:3]

  ## Fit the three required models
  gauss_fit_uncn <-
    fit_gaussian_2D(
      samp_dat,
      method = "elliptical_log",
      constrain_amplitude = FALSE,
      constrain_orientation = "unconstrained"
    )

  gauss_fit_diag <-
    fit_gaussian_2D(
      samp_dat,
      method = "elliptical_log",
      constrain_amplitude = FALSE,
      constrain_orientation = 0
    )

  gauss_fit_indp <-
    fit_gaussian_2D(
      samp_dat,
      method = "elliptical_log",
      constrain_amplitude = FALSE,
      constrain_orientation = -1
    )

  ## Combine the outputs into a list
  models_list <-
    list(
      gauss_fit_uncn,
      gauss_fit_diag,
      gauss_fit_indp
    )

  ## Now characterize
  out <-
    characterize_gaussian_fits(models_list)
  out

  ## Alternatively, the raw data itself can be supplied.
  ## This is less preferred, as fitting of models may fail
  ## internally.
  out2 <-
    characterize_gaussian_fits(data = samp_dat)

  ## This produces the same output, assuming models are fit without error
  identical(out, out2)
}
}
\references{
Levitt JB, Kiper DC, Movshon JA. Receptive fields and functional architecture
of macaque V2. J Neurophysiol. 1994 71:2517–2542.

Priebe NJ, Cassanello CR, Lisberger SG. The neural representation of speed in
macaque area MT/V5. J Neurosci. 2003 Jul 2;23(13):5650-61. doi:
10.1523/JNEUROSCI.23-13-05650.2003.

Winship IR, Crowder N, Wylie DRW. Quantitative reassessment of speed tuning
in the accessory optic system and pretectum of pigeons. J Neurophysiol. 2006
95(1):546-551. doi: 10.1152/jn.00921.2005
}
\author{
Vikram B. Baliga
}
