#' @title Totally balanced check
#' @description This function checks if the given game is totally balanced and computes its totally balanced cover.
#' @param v A characteristic function, as a vector.
#' @param game A logical value. By default, \code{game=FALSE}. If set to \code{TRUE}, the totally balanced cover of the game is also returned.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return \code{TRUE} if the game is totally balanced, \code{FALSE} otherwise. If \code{game=TRUE}, the totally balanced cover of the game is also returned.
#' @details A game \eqn{v \in G^{N}} is totally balanced if all of its subgames are balanced
#' (the subgame of each coalition \eqn{S \in 2^{N}} with respect to \eqn{v} is defined by
#' \eqn{v_S(T)=v(T)} for all \eqn{T\in 2^S}).
#' @examples
#' totallybalancedcheck(c(0,0,0,0,1/2,0,0,1/2,0,1/2,1/2,1/2,1/2,1/2,1))
#' totallybalancedcheck(c(0,0,0,0,1,1,0,1,0,0,1,1,1,1,2),game=TRUE)
#' @references Maschler, M., Solan, E., & Zamir, S. (2013). \emph{Game Theory}. Cambridge University Press.
#' @seealso \link{balancedcheck}, \link{subgame}
#' @export

totallybalancedcheck <- function(v, game = FALSE, binary = FALSE, tol = 100*.Machine$double.eps) {

  nC <- length(v)
  n <- log2(nC + 1)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  v.copia <- v # guardamos v tal como nos lo dieron, nos vendrá bien al final
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  Indis <- 2^(0:(n - 1)) # Posiciones de las coaliciones individuales en binario
  coaliNonIndi <- c(1:nC)
  coaliNonIndi <- coaliNonIndi[-Indis] # Posiciones del resto de las coaliciones

  # Salidas, por defecto v totalmente equilibrado
  teqv <- TRUE
  teqenvelope <- v

  # Resolvemos, uno por uno, todos los simplex, que están en balancedcheck

  ## sustituyo subxogos por nuestros subgames, eliminar comentarios que siguen si todo va bien

  for (ii in coaliNonIndi) {
    #sub_result <- subxogos(v, ii, n)
    v0ii <- subgame(v,ii,binary=TRUE)

    #v0ii <- sub_result$subxog
    simp_result <- balancedcheck(v=v0ii,game=TRUE,binary=TRUE,tol=tol)
    bii <- simp_result$check
    if (bii == FALSE) {
      jugadores <- paste("{", paste(getcoalition(ii), collapse = ","), "},", sep = "")
      if (binary == FALSE) {
        juego <- paste("[", paste(bin2lex(v0ii), collapse = ","), "],", sep = "")
      } else {
        juego <- paste("[", paste(v0ii, collapse = ","), "],", sep = "")
      }
      message(juego, " the subgame of coalition",jugadores," is not balanced.\n\n")
      teqv <- FALSE
      if (game == FALSE) { # si solo nos piden devolver el check, ya paramos aquí
        return(check = teqv)
      } else {
        v0iiS <- simp_result$game
        teqenvelope[ii] <- v0iiS[length(v0iiS)]
      }
    }
  }

  # Salidas si no llegamos a parar antes.
  if (game == FALSE) {
    return(check = teqv)
  }
  if (binary == FALSE) {
    teqenvelope <- bin2lex(teqenvelope)
  }
  return(list(check = teqv, game = teqenvelope))
}
