#' @title Shapley value
#' @description Given a game, this function computes its Shapley value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return The Shapley value of the game, as a vector.
#' @details Given \eqn{v\in G^N}, the Shapley value of each player \eqn{i \in N} can be defined as
#' \deqn{Sh_{i}(v) = \sum_{S \subset N \setminus \{i\}} \frac{s!(n-s-1)!}{n!} (v(S \cup \{i\})-v(S)).}
#' It is also possible to compute it as
#' \deqn{Sh_{i}(v) = \sum_{\emptyset \neq S \subset N} M_{i,S} v(S),}
#' where \eqn{M_{i,S} = \frac{(s-1)!(n-s)!}{n!}} if \eqn{i \in S} and \eqn{M_{i,S} = -\frac{s!(n-s-1)!}{n!}} if \eqn{i \notin S}.
#' @examples
#' shapleyvalue(c(0,0,3,0,3,8,6,0,6,9,15,8,16,17,20), binary=TRUE)
#' shapleyvalue(claimsgame(E=69.420,d=runif(10,5,10)))
#' @references Le Creurer, I. J., Mirás Calvo, M. A., Núñez Lugilde, I., Quinteiro Sandomingo, C., & Sánchez Rodríguez, E. (2024). On the computation of the Shapley value and the random arrival rule. Available at \url{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=4293746}.
#' @references Shapley, L. S. (1953). A value for n-person games. \emph{Contribution to the Theory of Games}, 2.
#' @seealso \link{marginalvector}
#' @export

shapleyvalue=function(v,binary=FALSE)
{# Principio de la función

  ################################
  ###Comprobación datos entrada###
  ################################

  nC=length(v)#Número de coaliciones.
  n=log(nC+1)/log(2)#Número de jugadores.
  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ######Cuerpo de la función######
  ################################
  #Caso |N|=1
  if(n==1)
  {
    Sh=v[1]
    return(Sh)
  }
  #Caso |N|=2
  if(n==2)
  {
    m=c(max(0,v[3]-v[2]),max(0,v[3]-v[1]))
    V=matrix(c(m[1],v[3]-m[1],v[3]-m[2],m[2]),ncol=2,byrow=T)
    Sh=colSums(V)/2
    return(Sh)
  }
  #Ahora vamos al caso general.
  #Primero hacemos el Algoritmo 1 del artículo del cálculo del valor de Shapley.
  #Calculamos los valores necesarios para la matriz.
  #Necesitamos K=round(n/2) valores.

  mitad = ceiling(n/2);
  valores = 1/n*rep(1,mitad);

  for (ii in 2:mitad)
  {#El primero vale 1/n y el resto se calculan de forma recursiva.
    valores[ii]=valores[ii-1]*(ii-1)/(n-ii+1);
  }

  #Pasamos al algoritmo 2.


  #Ahora, por dualidad, conseguimos todos los valores de la matriz de Shapley
  if (n/2==round(n/2))
  {#Si |N| es par.
    valores=c(valores,sort(valores));
  }else
  {
    valores=c(valores,sort(valores[1:(length(valores)-1)]));
  }

  #Ahora ya podemos calcular el valor de Shapley

  Sh= rep(0,n) #Lo inicializamos como vector de 0's.
  for (S in 1:(2^(n-1)-1))
  {#Trabajamos con la mitad de las coaliciones. El resto por dualidad.
    coalicion=as.numeric(intToBits(S)[1:n])#Coaliciones en binario.
    a = sum(coalicion);#Número de jugadores en la coalición
    coalicion = coalicion*valores[a];
    coalicion[coalicion==0]=-valores[a+1];#Dualidad
    Sh = Sh+(v[S]-v[2^n-1-S])*coalicion;#Formula obtenida en el paper.
  }
  Sh=Sh+v[nC]/n; #Valor final.

  ################################
  ######Salidas de la función#####
  ################################
  return(Sh)
}# Fin de la función
