#' @title Plot core set
#' @description Given a game with two, three or four players, this function plots its core set and set of imputations.
#' @param v A characteristic function, as a vector. The game represented by \code{v} is assumed to be a profit game (i.e., a game in which a greater allocation is a more desirable allocation), not a cost game (i.e., a game in which a smaller allocation is a more desirable allocation).
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param imputations A logical value. By default, \code{imputations=TRUE}. When set to \code{imputations=FALSE}, the set of imputations is not drawn.
#' @param projected A logical value. By default, \code{projected=FALSE}. When set to \code{projected=TRUE}, for games with three or four players the function draws a projection of the core set (and a projection of the set of imputations, as long as \code{imputations=TRUE}) instead of a full-dimensional representation.
#' @param solutions Optional. A character vector containing a solution or a series of solutions to be added to the plot. Valid solutions:
#'        \code{"corecenter"}, \code{"nucleolus"}, \code{"nucleoluspc"}, \code{"shapleyvalue"}, \code{"tauvalue"}.
#' @param allocations Optional. A matrix containing an allocation or a series of allocations to be added to the plot. The matrix should have as many columns as players in \code{v} and as many rows as allocations are introduced, so that each row contains an allocation.
#' @param color The color in which the core set is to be drawn. By default, \code{color="blue"}.
#' @return A core set plot with the specified features.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @examples
#' v1 <- claimsgame(E=8,d=c(3,5,6))
#' plotcoreset(v1,solutions=c("nucleolus","shapleyvalue"))
#'
#' v2 <- c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' plotcoreset(v2,solutions=c("corecenter","nucleoluspc"))
#' @seealso \link{plotcoresets}
#' @export

plotcoreset <- function(v,binary=FALSE,imputations=TRUE,projected=FALSE,solutions=NULL,allocations=NULL,color="blue") {

  if (length(v) %in% c(3,7,15) == FALSE) {
    stop("'v' is not a characteristic function of a game with two, three or four players.")
  }

  if (binary==TRUE) { # Las funciones que vamos a usar funcionan en lexicográfico.
    v<-bin2lex(v)
  }

  ###################
  ### 2 jugadores ###
  ###################

  if (length(v)==3){
    return(plot2real(v=v,imputations=imputations,solutions=solutions,allocations=allocations,color=color))
  }

  ###################
  ### 3 jugadores ###
  ###################

  if (length(v)==7) {
    if (projected==TRUE) {
      return(plot3proj(v=v,imputations=imputations,solutions=solutions,allocations=allocations,color=color))
    } else {
      return(plot3real(v=v,imputations=imputations,solutions=solutions,allocations=allocations,color=color))
    }
  }

  ###################
  ### 4 jugadores ###
  ###################

  if (length(v)==15) {
    if (projected==TRUE) {
      return(plot4proj(v=v,imputations=imputations,solutions=solutions,allocations=allocations,color=color))
    } else {
      return(plot4real(v=v,imputations=imputations,solutions=solutions,allocations=allocations,color=color))
    }
  }

}
