#' @title Minimal rights vector
#' @description This function computes the minimal rights vector of a game.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return The minimal rights vector.
#' @details Given \eqn{v\in G^N}, the utopia payoff of player \eqn{i\in N} is defined as
#' \eqn{M_i(N,v)=v(N)-v(N\backslash i)}.
#'
#' The minimal right of player \eqn{i\in N} is defined as \eqn{m_i(N,v)=\max_{S:i\in S}(v(S)-\sum_{j\in S\backslash i}M_j(N,v))}.
#' @examples
#' v <- c(0, 0, 0, 1, 1, 1, 2)
#' minimalrightsvector(v)
#' convexcheck(v)
#' minimalrightsvector(v) == c(v[1],v[2],v[3])
#' w <- c(0,0,0,4,7,6,10)
#' convexcheck(w)
#' minimalrightsvector(w) == c(w[1],w[2],w[3])
#' @seealso \link{utopiapayoffsvector}
#' @export
minimalrightsvector <- function(v, binary = FALSE) { # Principio de la función

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  m <- c() # Minimal rights.
  M <- c() # Utopia payoffs.

  ################################
  ###### Cuerpo de la función######
  ################################

  # Utopia Payoffs:
  for (ii in 1:n) {
    M[ii] <- v[nC] - v[nC - 2^(ii - 1)] # Mi=v(N)-v(N\ i)
  }

  # Minimal rights:

  m <- v[2.^(seq(1:n) - 1)] # Vector de v(i).

  for (S in 1:nC) {
    num <- S # Primero calculo el vector en con los jugadores pertenecientes a la coalición en posición S.
    J <- floor(log2(num)) + 1
    Sjug <- rep(0, n) # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 == 1) {
        Sjug[ii] <- 1
      }
      num <- num %/% 2
    }

    SMii <- M * Sjug # Pagos de utopia de los jugadores en la coalición S.
    for (kk in which(Sjug > 0)) { # max(v(S)-\sum_{j in S\i}M_j(N,v)))
      m[kk] <- max(m[kk], v[S] - sum(SMii) + SMii[kk])
    }
  }

  ################################
  ###### Salidas de la función#####
  ################################
  return(m)
} # Fin de la función
