#' @title Essential check
#' @description This function checks if the given game is essential.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return \code{TRUE} if the game is essential, \code{FALSE} otherwise.
#' @details A game \eqn{v\in G^N} is essential if its set of imputations is non-empty, that is,
#' if \eqn{v(N)\ge \sum_{i \in N} v(i)}.
#' @examples
#' v <- c(0, 0, 0, 2, 3, 4, 1)
#' essentialcheck(v, binary = TRUE)
#' essentialcheck(v, binary = FALSE)
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # essential cost game
#' essentialcheck(-cost.v)
#' @seealso \link{degeneratecheck}
#' @export

essentialcheck <- function(v, binary = FALSE) {

  # Comprobaciones iniciales.
  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  # Definimos variables que necesitamos.
  checkR <- TRUE # juego esencial hasta que se demuestre lo contrario
  tol <- 100*.Machine$double.eps

  # Si el juego se introduce en lexicográfico, comprobamos directamente en lexicográfico.
  # Si el juego se introduce en binario, comprobamos directamente en binario.
  if (binary == FALSE) {
    if (v[nC] - sum(v[1:n]) < -tol) {
      checkR <- FALSE
    }
  } else {
    if (v[nC] - sum(v[2.^(seq(1, n)-1)]) < -tol) {
      checkR <- FALSE
    }
  }

  return(checkR)
} # Fin de la función
