#' @title Binary order position to lexicographic order position
#' @description Given the binary order position of a coalition, this function returns the corresponding lexicographic order position.
#' @param n Number of players, as an integer.
#' @param Nbin A binary order position, as an integer between 1 and \eqn{2^{\code{n}}-1}.
#' @return The corresponding lexicographic order position, as an integer between 1 and \eqn{2^{\code{n}}-1}.
#' @details The binary order position of a coalition \eqn{S\in 2^N} is given by \eqn{\sum_{i\in S} 2^{i-1}}. Lexicographic order arranges coalitions in ascending order according to size, and applies lexicographic order to break ties among coalitions of the same size.
#' @examples
#' codebin2lex(5, 4)
#' @seealso \link{bin2lex}, \link{codelex2bin}, \link{lex2bin}
#' @export

codebin2lex <- function(n, Nbin) {

  ################################
  ### Comprobación datos entrada###
  ################################

  if (is.null(Nbin) | is.character(Nbin)) { # Metemos esto para evitar error en additivecheck, belong2core.
    return(Nbin)
  }
  if (Nbin > (2^n - 1)) {
    stop("'Nbin' cannot be greater than 2^n-1.", call. = F)
  }

  ################################
  ##### Creación de variables######
  ################################

  J <- floor(log2(Nbin)) + 1 # numero de jugadores efectivos
  coalition <- rep(0, n) # Inicializo la coalición
  Nlex <- 0 # inicializo el codigo lexicografico
  CC <- Nbin

  ################################
  ###### Cuerpo de la función######
  ################################

  # Obtengo la coalicion (algoritmo de paso de decimal a binario)

  for (ii in 1:J)
  {
    if (CC %% 2 == 1) {
      coalition[ii] <- 1
    }
    CC <- (CC - (CC %% 2)) / 2
  }

  jug <- sum(coalition) # Numero de jugadores en la coalicion
  # Las coaliciones con menos jugadores la preceden en el orden lexicografico
  # Sumo el número de coaliciones con menos jugadores.

  if (jug > 1) {
    for (J in 1:(jug - 1))
    {
      Nlex <- Nlex + choose(n, J)
    }
  }
  # Si no tiene un uno en la posicion ii, le anteceden todas las que si lo tengan
  ii <- 0 # posicion

  while (jug > 0) {
    ii <- ii + 1
    # Miro si en la posición ii va un cero o un uno. Si fuese un cero, el numero de la coalicion
    # sería mayor que "Nlex" más el numero de coaliciones de tamaño n-ii con jug-1 unos y el resto
    # ceros. Permutaciones con repeticion de n-ii objetos (ceros y unos)  con (jug-1) y
    # (n-ii-jug+1) repeticiones.
    tope <- factorial(n - ii) / (factorial(jug - 1) * factorial(n - ii - jug + 1))
    # Si el jugador ii NO esta en la coalicion, todas las coaliciones del tamaño restante
    # que tengan un uno en ese jugador irán delante en el orden lexiografico.
    # Si el jugador está, estoy en el tramo adecuado, entonces, elimino al jugador y sigo.
    if (coalition[ii] == 0) {
      Nlex <- Nlex + tope
    } else {
      jug <- jug - 1
    }
  }

  # Reajusto la posición.
  Nlex <- Nlex + 1
  ################################
  ###### Salidas de la función#####
  ################################

  return(Nlex)
} # Fin de la función
