#'Divides the univariate time series data into subcomponents
#'
#' @param data A numeric vector containing the time series values.
#'
#' @return A data frame containing the decomposed IMFs and residual.
#'
#' @examples
#' # Example usage:
#' sample_data <- rnorm(3000)
#' result <- RAD(sample_data)
#' head(result)
#'
#' @importFrom stats cor mad spline
#' @importFrom utils tail
#' @export


RAD <- function(data) {
  if (!is.numeric(data) || length(data) < 2) {
    stop("Input must be a numeric vector with at least two values.")
  }

  iteration <- 1
  current_data <- data
  original_data <- data
  imfs <- list()
  correlations <- numeric()
  iteration_data <- data.frame(matrix(ncol = 0, nrow = length(data)))
  best_iteration <- NULL

  calculate_weighted_mean <- function(y, weights) {
    n <- length(y)
    weighted_means <- numeric(length = n - 1)
    for (i in 1:(n - 1)) {
      weighted_means[i] <- sum(y[i:(i + 1)] * weights[i:(i + 1)]) / sum(weights[i:(i + 1)])
    }
    return(weighted_means)
  }

  repeat {
    mad_value <- mad(current_data)
    weights <- abs(current_data / mad_value)
    weighted_means <- calculate_weighted_mean(current_data, weights)

    calculate_mean_and_append <- function(data) {
      mean_last_two <- mean(tail(data, 2))
      data <- c(data, mean_last_two)
      return(data)
    }
    result <- calculate_mean_and_append(weighted_means)

    x <- seq_along(result)
    spline_interp <- spline(x, result, n = length(x))

    imf <- current_data - spline_interp$y
    imfs[[iteration]] <- imf

    corr <- cor(imf, original_data)
    correlations <- c(correlations, corr)

    # **Stopping Condition**:
    if (iteration > 1 && abs(corr) > abs(correlations[iteration - 1])) {
      best_iteration <- iteration - 1

      for (i in 1:best_iteration) {
        iteration_data[paste("IMF", i)] <- imfs[[i]]
      }

      residual_name <- "Residual"
      iteration_data[residual_name] <- original_data - rowSums(as.data.frame(imfs[1:best_iteration]))

      rownames(iteration_data) <- NULL
      return(iteration_data)
    }

    # **Extra stopping rule for continuous decrease (after first IMF)**:
    if (iteration == 1) {
      iteration_data[["IMF 1"]] <- imfs[[1]]
      iteration_data[["Residual"]] <- original_data - imfs[[1]]
      rownames(iteration_data) <- NULL
      return(iteration_data)
    }

    current_data <- current_data - imf
    iteration <- iteration + 1
  }
}
