#' Step 4 of the Hartigan and Wong algorithm
#'
#' This function implements the Step 4 of the Hartigan and Wong (Hartigan and Wong, 1979) algorithm without imposing that the use of the euclidean distance  and without imposing that the centers of the groups are calculated by averaging the points. This function allows other distances to be used and allows the centers of the groups to be calculated in different ways.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param centers Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#' @param grouping List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#' @param LIVE_SET_original Vector that contains the groups that have been modified in the previous Step 6. The Step 6 is described in Hartigan and Wong (1979).
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param centers_function Function. This function designs how the centers of the groups will be calculated. It must have as input \code{data} and \code{grouping} and as output a matrix that has the centers. This matrix will have as many rows as centers.
#' @param Ic12_change Matrix. The first row contains the IC1 of each point. The second column contains the IC2 of each point. IC1 and IC2 are the closets and second closest cluster centers.
#' @param index Number. When a point is reallocated, index becomes zero.
#'
#'
#'
#' @return Returns a list with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated. Matrix with \code{dim(centers)[1]} centers of \code{dim(centers)[2]} dimensions.
#'   \item  IC1andIC2: the information of the IC1 and IC2 updated. Matrix.  The first row contains the IC1 of each point. The second column contains the IC2 of each point. IC1 and IC2 are the closets and second closest cluster centers.
#'   \item  grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'   \item Live_set: Vector. Contains the groups that have been modified during the Step 4.
#'   \item  no_Change: vector with the points that do not change its group. More specifically, contains the row of the matrix \code{data} where these points are.
#'   }
#'
#'
#'
#'
#' @examples
#'
#'set.seed(231)
#'data1=gtools::rdirichlet(10,c(1,1,4,4,20,20))
#'data=t(data1)
#'k=3
#'seed=5
#'
#'
#'
#'if(!is.null(seed)){
#'  set.seed(seed)
#'}
#'centers <- data[sample(1:nrow(data), k), ]
#'
#'#We calculate the distance between each row of the data matrix and the centers
#'Dist_e_cent=matrix(0,dim(data)[1],dim(centers)[1])
#'for (i in 1:(dim(data)[1])){
#'for (j in 1:(dim(centers)[1])){
#'  Dist_e_cent[i,j]=Euclideandistance(data[i,],centers[j,])
#'}
#'}
#'
#'
#'Ic12=Dist_IC1_IC2(Dist_e_cent)
#'Ic12_change=Ic12
#'Group=Ic12[,1]
#'grouping<-list()
#'for(i in 1:(max(Group))){
#'grouping[[i]]=which(Group==i)
#'}
#'
#'#Update the clusters centers.
#'centers=centers_function_mean(data, grouping)
#'
#'#Live set.
#'LIVE_SET_original1=c(1:length(grouping))
#'
#'index=0
#'
#'P1=Step4(data,
#'         centers,
#'         grouping,
#'         LIVE_SET_original1,
#'         Euclideandistance,
#'         centers_function_mean,
#'         Ic12_change,
#'         index)
#'
#'
#'
#'@references  Hartigan, J. A., & Wong, M. A. (1979). Algorithm AS 136: A k-means clustering algorithm. Journal of the royal statistical society. series c (applied statistics), 28(1), 100-108.
#'
#' @export


Step4<-function(data, centers, grouping, LIVE_SET_original, distance, centers_function, Ic12_change, index ){

k=dim(centers)[1]
no_Change=c()

for (ii in 1:(dim(data)[1])){
  index=index+1

  IC1=Ic12_change[ii,1]
  IC2=Ic12_change[ii,2]

  if(length(grouping[[IC1]])==1){#If the group has only one point, the point does not change.
    no_Change=add_unique_numbers2(no_Change,ii)
    if(index==dim(data)[1]){
      Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
      names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
      return(Step4_output)
    }
  }else{ #If the group has more than one point
  R1=length(grouping[[IC1]])*distance(data[ii,],centers[IC1,])^2/(length(grouping[[IC1]])-1)
  R2=length(grouping[[IC2]])*distance(data[ii,],centers[IC2,])^2/(length(grouping[[IC2]])+1)

  for(L in 1:k){
    if(!(IC1 %in% LIVE_SET_original) && !(L %in% LIVE_SET_original) || L==IC1  || L==IC2){

              if(R2<R1){
                old=IC1
                new=IC2
                index=0
                Ic12_change[ii,2]=old
                Ic12_change[ii,1]=new
                for(i in 1:length(grouping)){
                  grouping[[i]]=grouping[[i]][grouping[[i]]!=ii]
                }
                grouping[[new]]=add_unique_numbers2(grouping[[new]], ii)
                centers=centers_function(data, grouping)
                LIVE_SET_original=add_unique_numbers(LIVE_SET_original,old,new)
              }else{
                no_Change=add_unique_numbers2(no_Change,ii)

                      if(index==dim(data)[1]){
                      Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
                      names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
                      return(Step4_output)
                      }
                }




    }else{

    RL=length(grouping[[L]])*distance(data[ii,],centers[L,])^2/(length(grouping[[L]])+1)
        if(RL<R2){
          R2=RL
          IC2=L
                    if(R2<R1){
                      old=IC1
                      new=IC2
                      index=0
                      Ic12_change[ii,2]=old
                      Ic12_change[ii,1]=new
                      for(i in 1:length(grouping)){
                        grouping[[i]]=grouping[[i]][grouping[[i]]!=ii]
                      }
                      grouping[[new]]=add_unique_numbers2(grouping[[new]], ii)
                      centers=centers_function(data, grouping)
                      LIVE_SET_original=add_unique_numbers(LIVE_SET_original,old,new)
                    }else{
                      no_Change=add_unique_numbers2(no_Change,ii)
                      Ic12_change[ii,2]=IC2

                                  if(index==dim(data)[1]){
                                    Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
                                    names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
                                    return(Step4_output)
                                  }

                    }


        }else{
                  #Recalculate with R2
                  if(R2<R1){
                    old=IC1
                    new=IC2
                    index=0
                    Ic12_change[ii,2]=old
                    Ic12_change[ii,1]=new
                    for(i in 1:length(grouping)){
                      grouping[[i]]=grouping[[i]][grouping[[i]]!=ii]
                    }
                    grouping[[new]]=add_unique_numbers2(grouping[[new]], ii)
                    centers=centers_function(data, grouping)
                    LIVE_SET_original=add_unique_numbers(LIVE_SET_original,old,new)
                  }else{
                    no_Change=add_unique_numbers2(no_Change,ii)

                              if(index==dim(data)[1]){
                                Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
                                names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
                                return(Step4_output)
                              }
                  }
        }




       }
    }

            if(index==dim(data)[1]){
              Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
              names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
              return(Step4_output)
            }

  }

}

Step4_output=list(centers, Ic12_change, grouping, LIVE_SET_original, no_Change,index)
names(Step4_output)<-c("centers", "IC1andIC2", "grouping", "Live_set", "no_Change", "index")
return(Step4_output)

}


