% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit-action-model.R
\name{add_model}
\alias{add_model}
\alias{remove_model}
\alias{update_model}
\title{Add a model to a workflow}
\usage{
add_model(x, spec, ..., formula = NULL)

remove_model(x)

update_model(x, spec, ..., formula = NULL)
}
\arguments{
\item{x}{A workflow.}

\item{spec}{A parsnip model specification.}

\item{...}{These dots are for future extensions and must be empty.}

\item{formula}{An optional formula override to specify the terms of the
model. Typically, the terms are extracted from the formula or recipe
preprocessing methods. However, some models (like survival and bayesian
models) use the formula not to preprocess, but to specify the structure
of the model. In those cases, a formula specifying the model structure
must be passed unchanged into the model call itself. This argument is
used for those purposes.}
}
\value{
\code{x}, updated with either a new or removed model.
}
\description{
\itemize{
\item \code{add_model()} adds a parsnip model to the workflow.
\item \code{remove_model()} removes the model specification as well as any fitted
model object. Any extra formulas are also removed.
\item \code{update_model()} first removes the model then adds the new specification to
the workflow.
}
}
\details{
\code{add_model()} is a required step to construct a minimal workflow.
}
\section{Indicator Variable Details}{
Some modeling functions in R create indicator/dummy variables from
categorical data when you use a model formula, and some do not. When you
specify and fit a model with a \code{workflow()}, parsnip and workflows match
and reproduce the underlying behavior of the user-specified model’s
computational engine.
\subsection{Formula Preprocessor}{

In the \link[modeldata:Sacramento]{modeldata::Sacramento} data set of real
estate prices, the \code{type} variable has three levels: \code{"Residential"},
\code{"Condo"}, and \code{"Multi-Family"}. This base \code{workflow()} contains a
formula added via \code{\link[=add_formula]{add_formula()}} to predict property
price from property type, square footage, number of beds, and number of
baths:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(123)

library(parsnip)
library(recipes)
library(workflows)
library(modeldata)

data("Sacramento")

base_wf <- workflow() \%>\%
  add_formula(price ~ type + sqft + beds + baths)
}\if{html}{\out{</div>}}

This first model does create dummy/indicator variables:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{lm_spec <- linear_reg() \%>\%
  set_engine("lm")

base_wf \%>\%
  add_model(lm_spec) \%>\%
  fit(Sacramento)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## == Workflow [trained] ================================================
## Preprocessor: Formula
## Model: linear_reg()
## 
## -- Preprocessor ------------------------------------------------------
## price ~ type + sqft + beds + baths
## 
## -- Model -------------------------------------------------------------
## 
## Call:
## stats::lm(formula = ..y ~ ., data = data)
## 
## Coefficients:
##      (Intercept)  typeMulti_Family   typeResidential  
##          32919.4          -21995.8           33688.6  
##             sqft              beds             baths  
##            156.2          -29788.0            8730.0
}\if{html}{\out{</div>}}

There are \strong{five} independent variables in the fitted model for this
OLS linear regression. With this model type and engine, the factor
predictor \code{type} of the real estate properties was converted to two
binary predictors, \code{typeMulti_Family} and \code{typeResidential}. (The third
type, for condos, does not need its own column because it is the
baseline level).

This second model does not create dummy/indicator variables:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rf_spec <- rand_forest() \%>\%
  set_mode("regression") \%>\%
  set_engine("ranger")

base_wf \%>\%
  add_model(rf_spec) \%>\%
  fit(Sacramento)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## == Workflow [trained] ================================================
## Preprocessor: Formula
## Model: rand_forest()
## 
## -- Preprocessor ------------------------------------------------------
## price ~ type + sqft + beds + baths
## 
## -- Model -------------------------------------------------------------
## Ranger result
## 
## Call:
##  ranger::ranger(x = maybe_data_frame(x), y = y, num.threads = 1,      verbose = FALSE, seed = sample.int(10^5, 1)) 
## 
## Type:                             Regression 
## Number of trees:                  500 
## Sample size:                      932 
## Number of independent variables:  4 
## Mtry:                             2 
## Target node size:                 5 
## Variable importance mode:         none 
## Splitrule:                        variance 
## OOB prediction error (MSE):       7058847504 
## R squared (OOB):                  0.5894647
}\if{html}{\out{</div>}}

Note that there are \strong{four} independent variables in the fitted model
for this ranger random forest. With this model type and engine,
indicator variables were not created for the \code{type} of real estate
property being sold. Tree-based models such as random forest models can
handle factor predictors directly, and don’t need any conversion to
numeric binary variables.
}

\subsection{Recipe Preprocessor}{

When you specify a model with a \code{workflow()} and a recipe preprocessor
via \code{\link[=add_recipe]{add_recipe()}}, the \emph{recipe} controls whether dummy
variables are created or not; the recipe overrides any underlying
behavior from the model’s computational engine.
}
}

\examples{
library(parsnip)

lm_model <- linear_reg()
lm_model <- set_engine(lm_model, "lm")

regularized_model <- set_engine(lm_model, "glmnet")

workflow <- workflow()
workflow <- add_model(workflow, lm_model)
workflow

workflow <- add_formula(workflow, mpg ~ .)
workflow

remove_model(workflow)

fitted <- fit(workflow, data = mtcars)
fitted

remove_model(fitted)

remove_model(workflow)

update_model(workflow, regularized_model)
update_model(fitted, regularized_model)
}
