% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{graph.param.estimator}
\alias{graph.param.estimator}
\title{Graph parameter estimator}
\usage{
graph.param.estimator(
  G,
  model,
  parameters = NULL,
  eps = 0.01,
  bandwidth = "Silverman",
  eigenvalues = NULL,
  spectra = NULL,
  classic = FALSE
)
}
\arguments{
\item{G}{the undirected graph (igraph type) or its adjacency matrix. The
adjacency matrix of an unweighted graph contains only 0s and 1s, while the
weighted graph may have nonnegative real values that correspond to the
weights of the edges.}

\item{model}{either a string or a function:

A string that indicates one of the following models: "ER" (Erdos-Renyi random
graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
(Watts-Strogatz model), and "BA" (Barabasi-Albert model).

A function that returns a graph (represented by its adjacency matrix)
generated by a graph model. It must contain two arguments: the first one
corresponds to the graph size and the second to the parameter of the model.}

\item{parameters}{numeric vector containing the values that that will be
considered for the parameter estimation. The 'graph.param.estimator' will
return the element of 'parameter' that minimizes the Kullback-Leiber divergence.
If the user does not provide the argument 'parameters', and 'model' is an
array, then the values considered for the parameter estimation are the
rownames converted to a numeric vector. If 'model' is a string, then
default values are used for the predefined models ("ER", "GRG", "KR", "WS",
and "BA"). The default 'parameter' argument corresponds to a sequence from

0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in
which the parameter corresponds to the probability to connect a pair of
vertices;

0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in
which the parameter corresponds to the radius used to construct the geometric
graph in a unit square;

0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in
which the parameter of the model corresponds to the degree 'k' of a regular
graph;

0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which
the parameter corresponds to the probability to reconnect a vertex;

and 0 to 3 with step 'eps' for the "BA" model (Barabasi-Albert model), in
which the parameter corresponds to the scaling exponent.}

\item{eps}{precision of the grid (default is 0.01) when 'classic' is TRUE.}

\item{bandwidth}{string showing which criterion is used to choose the
bandwidth during the spectral density estimation. Choose between the
following criteria: "Silverman" (default), "Sturges", "bcv", "ucv" and "SJ".
"bcv" is an abbreviation of biased cross-validation, while "ucv" means
unbiased cross-validation. "SJ"  implements the methods of Sheather & Jones
(1991) to select the bandwidth using pilot estimation of derivatives.}

\item{eigenvalues}{optional parameter. It contains the eigenvalues of matrix
G. Then, it can be used when the eigenvalues of G were previously computed.
If no value is passed, then the eigenvalues of G will be computed by
'graph.param.estimator'.}

\item{spectra}{optional parameter containing the precomputed spectrum of the
model. It is a three-dimensional array in which the first dimension
corresponds to all parameters that will be explored in the grid, the second
dimension has the same size of the given graph, and the third one corresponds
to graphs randomly generated by the model. Thus, the position (i,j,k)
contains the j-th eigenvalue of the k-th graph generated with the i-th
parameter. The attribute 'rownames' of the array corresponds to the
parameters converted to string. If spectra is NULL (default), then 'model' is
used to generate random graphs and their spectra are computed automatically.}

\item{classic}{logical. If FALSE (default) parameter is estimated using
ternary search. If TRUE parameter is estimated using grid search.}
}
\value{
A list containing:
\item{param}{the parameter estimate. For the "ER", "GRG", "KR", "WS", and "BA"
models, the parameter corresponds to the probability to connect a pair of
vertices, the radius used to construct the geometric graph in a unit square,
the degree k of a regular graph, the probability to reconnect a vertex, and
the scaling exponent, respectively.}
\item{KLD}{the Kullback-Leibler
divergence between the observed graph and the graph model with the estimated
parameter.}
}
\description{
\code{graph.param.estimator} estimates the parameter that best approximates
the model to the observed graph according to the Graph Information Criterion
(GIC).
}
\examples{
set.seed(1)
G <- igraph::sample_gnp(n=50, p=0.5)

# Using a string to indicate the graph model
result1 <- graph.param.estimator(G, "ER", eps=0.25)
result1

\dontrun{
# Using a function to describe the graph model
# Erdos-Renyi graph
set.seed(1)
model <- function(n, p) {
  return(igraph::sample_gnp(n, p))
}
result2 <- graph.param.estimator(G, model,  seq(0.2, 0.8, 0.1))
result2
}

}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
Discriminating Different Classes of Biological Networks by Analyzing the
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
*21*, 65-66.

Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
selection method for kernel density estimation.
_Journal of the Royal Statistical Society series B_, 53, 683-690.
http://www.jstor.org/stable/2345597.
}
\keyword{parameter_estimation}
