% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statGraph.R
\name{graph.model.selection}
\alias{graph.model.selection}
\title{Graph model selection}
\usage{
graph.model.selection(
  A,
  models = NULL,
  parameters = NULL,
  eps = 0.01,
  bandwidth = "Silverman",
  eigenvalues = NULL
)
}
\arguments{
\item{A}{the adjacency (symmetric) matrix of an undirected graph. For an
unweighted graph it contains only 0s and 1s. For a weighted graph, it
contains real values that correspond to the weights of the edges.}

\item{models}{either a vector of strings, a list of functions or a list of
arrays describing graph models:

A vector of strings containing some of the following models: "ER" (Erdos-Renyi
random graph), "GRG" (geometric random graph), "KR" (k regular random graph),
"WS" (Watts-Strogatz model), and "BA" (Barabasi-Albert model).

A list of functions. Each function returns a graph (represented by its
adjacency matrix) generated by a graph model and has two arguments: the graph
size and the model parameter, in this order.

A list of arrays. Each elememt of the list is a three-dimensional array
containing the precomputed spectrum of each model. Let M be a graph model.
For each parameter p considered for M, the array of model M contains the
eigenvalues of graphs randomly generated by M with parameter p. The position
(i,j,k) of the array contains the j-th eigenvalue of the k-th graph
that generated by M with the i-th parameter. The attribute 'rownames' of
the array corresponds to the parameters converted to string.

If the argument "models" is NULL, then the "ER", "WS", and "BA" models will
be considered for the model selection.}

\item{parameters}{list of numeric vectors. Each vector contains the values
that will be considerated for the parameter estimation of the corresponding
model.
If the user does not provide the argument 'parameters', then default values
are used for the predefined models ("ER", "GRG", "KR", "WS", and "BA").
The default vector corresponds to a sequence from

0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in
which the parameter corresponds to the probability to connect a pair of
vertices;

0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in
which the parameter corresponds to the radius used to contruct the geometric
graph in a unit square;

0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in
which the parameter of the model corresponds to the degree 'k' of a regular
graph;

0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which
the parameter corresponds to the probability to reconnect a vertex;

and 0 to 3 with step 'eps' for the "BA" model (Barabasi-Albert model), in
which the parameter corresponds to the scaling exponent.}

\item{eps}{precision of the grid (default is 0.01).}

\item{bandwidth}{string indicating which criterion will be used to choose
the bandwidth for the spectral density estimation. The available criteria are
"Silverman" (default) and "Sturges".}

\item{eigenvalues}{optional parameter. It contains the eigenvalues of matrix
A. Then, it can be used when the eigenvalues of A were previously computed.
If no value is passed, then the eigenvalues of A will be computed by
'graph.model.selection'.}
}
\value{
A list containing:
\item{model}{the indice(s) or name(s) of the selected model(s), i. e. the
model(s) that minimize(s) the Graph Information Criterion (GIC).}
\item{estimates}{a matrix in which each row corresponds to a model, the
column "p" corresponds to the parameter estimate, and the column "GIC"
corresponds to the Graph Information Criterion (GIC), i. e. the
Kullback-Leibler divergence between the observed graph and the model.}
}
\description{
\code{graph.model.selection} selects the graph model that best approximates the
observed graph according to the Graph Information Criterion (GIC).
}
\examples{

require(igraph)
A <- as.matrix(get.adjacency(erdos.renyi.game(30, p=0.5)))
# Using strings to indicate the graph models
result1 <- graph.model.selection(A, models=c("ER", "WS"), eps=0.5)
result1
# Using functions to describe the graph models
# Erdos-Renyi graph
model1 <- function(n, p) {
   return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
}
# Watts-Strougatz graph
model2 <- function(n, pr, K=8) {
    return(as.matrix(get.adjacency(watts.strogatz.game(1, n, K, pr))))
}
parameters <- list(seq(0, 1, 0.5), seq(0, 1, 0.5))
result2 <- graph.model.selection(A, list(model1, model2), parameters)
result2
}
\references{
Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
Discriminating Different Classes of Biological Networks by Analyzing the
Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
doi:10.1371/journal.pone.0049949.

Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.

Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
*21*, 65-66.
}
\keyword{model_selection}
