#' Graph spectral entropy
#'
#' 'graph.entropy' returns the spectral entropy of a given undirected graph.
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative
#' real values that correspond to the weights of the edges.
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth during the spectral density estimation. The available criteria
#' are "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, if the eigenvalues of matrix A have already been computed, this
#' parameter can be used instead of A. If no value is passed, then the
#' eigenvalues of A will be computed by 'graph.entropy'.
#'
#' @return a real number corresponding to the graph spectral entropy.
#'
#' @keywords spectral_entropy
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' G <- erdos.renyi.game(100, p=0.5)
#' A <- as.matrix(get.adjacency(G))
#' entropy <- graph.entropy(A)
#' entropy
#'
#' @export
graph.entropy <- function(A=NULL, bandwidth="Silverman", eigenvalues=NULL) {
  if (is.null(eigenvalues))
    f <- spectralDensity(A, bandwidth=bandwidth)
  else
    f <- gaussianDensity(eigenvalues, bandwidth=bandwidth)
  if (sum(is.na(f)) > 0)
    return(NA)
  y <- f$y
  n <- length(y)
  i <- which(y != 0)
  y[i] <- y[i]*log(y[i])
  return(-trapezoidSum(f$x, y))
}

#' Graph Information Criterion (GIC)
#'
#' 'GIC' returns the Kullback-Leibler divergence between an undirected graph and
#' a given graph model.
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative
#' real values that correspond to the weights of the edges.
#'
#' @param model either a list, a string, a function or a matrix describing a
#' graph model:
#'
#' A list that represents the spectral density of a model. It contains the
#' components "x" and "y", which are numeric vectors of the same size. The "x"
#' component contains the points at which the density was computed and the "y"
#' component contains the observed density.
#'
#' A string that indicates one of the following models: "ER" (Erdos-Renyi random
#' graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
#' (Watts-Strogatz model), and "BA" (Barabási-Albert model). When the argument
#' 'model' is a string, the user must also provide the model parameter by using
#' the argument 'p'.
#'
#' A function that returns a graph (represented by its adjacency matrix)
#' generated by a graph model. It must contain two arguments: the first one
#' correponds to the graph size and the second to the parameter of the model.
#' The model parameter will be provided by the argument 'p' of the 'GIC'
#' function.
#'
#' A matrix containing the spectrum of the model. Each column contains the
#' eigenvalues of a graph generated by the model. To estimate the spectral
#' density of the model, the method will estimate the density of the values of
#' each column, and then will take the average density.
#'
#' @param p the model parameter. The user must provide a valid parameter if the
#' argument 'model' is a string or a function.
#' For the predefined models ("ER", "GRG", "KR", "WS", and "BA"), the parameter
# 'p' corresponds to:
#
#' the probability to connect a pair of vertices, for the "ER" model
#' (Erdos-Renyi random graph);
#'
#' the radius used to contruct the geometric graph in a unit square, for the
#' "GRG" model (geometric random graph);
#'
#' the degree 'k' of a regular graph, for the "KR" model (k regular random
#' graph);
#'
#' the probability to reconnect a vertex, for the "WS" model (Watts-Strogatz
#' model);
#'
#' and the scaling exponent, for the "BA" model (Barabási-Albert model).
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed.
#' If no value is passed, then the eigenvalues of A will be computed by 'GIC'.
#'
#' @return A real number corresponding to the Kullback-Leibler divergence
#' between the observed graph and the graph model.
#'
#' @keywords graph_information_criterion
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(100, p=0.5)))
#' # Using a string to indicate the graph model
#' result1 <- GIC(A, "ER", 0.5)
#' result1
#'
#' # Using a function to describe the graph model
#' # Erdos-Renyi graph
#' model <- function(n, p) {
#'    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' }
#' result2 <- GIC(A, model, 0.5)
#' result2
#' @export
GIC <- function(A, model, p=NULL, bandwidth="Silverman", eigenvalues=NULL) {
  # TODO: implement differente distances
  dist <- "KL"
  if (is.null(eigenvalues))
    eigenvalues <- (as.numeric(eigen(A, only.values = TRUE,
                                     symmetric=TRUE)$values)/sqrt(nrow(A)))
  if (class(model) == "list") {
    f2 <- model
    f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x),
                          bandwidth=bandwidth, npoints=1024)
  }
  else if (class(model) == "matrix") {
    f2 <- nDensities(model, from=min(eigenvalues),
                     to=max(eigenvalues), bandwidth=bandwidth,
                     npoints=1024)
    if (sum(is.na(f2)) > 0)
      return(Inf)
    f2 <- list("x"=f2$x, "y"=rowMeans(f2$densities))
    f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x),
                          bandwidth=bandwidth, npoints=1024)
  }
  else {
    fun <- model
    if (class(model) == "character") {
      if (model == "WS")
        fun <- WSfun(as.integer(sum(A)/(2*ncol(A))))
      else fun <- matchFunction(model)
    }
    f2 <- modelSpectralDensity(fun, ncol(A), p, from=min(eigenvalues),
                               to=max(eigenvalues), bandwidth=bandwidth,
                               npoints=1024)
    if (sum(is.na(f2)) > 0)
      return(Inf)
    f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x),
                          bandwidth=bandwidth, npoints=1024)
  }
  if (sum(is.na(f1)) > 0)
    return(Inf)
  if (sum(is.na(f2)) > 0)
    return(Inf)
  if(dist == "KL") out <- KL(f1,f2)
  else if(dist == "L2") out <- distance(f1,f2)
  return (out)
}

#' Graph parameter estimator
#'
#' 'graph.param.estimator' estimates the parameter that best approximates the
#' model to the observed graph according to the Graph Information Criterion
#' (GIC).
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative
#' real values that correspond to the weights of the edges.
#'
#' @param model either a string or a function:
#'
#' A string that indicates one of the following models: "ER" (Erdos-Renyi random
#' graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS"
#' (Watts-Strogatz model), and "BA" (Barabási-Albert model).
#'
#' A function that returns a graph (represented by its adjacency matrix)
#' generated by a graph model. It must contain two arguments: the first one
#' correponds to the graph size and the second to the parameter of the model.
#'
#' @param spectra optional parameter containing the precomputed spectrum of the
#' model. It is a three-dimensional array in which the first dimension
#' corresponds to all parameters that will be explored in the grid, the second
#' dimension has the same size of the given graph, and the third one corresponds
#' to graphs randomly generated by the model. Thus, the position (i,j,k)
#' contains the j-th eigenvalue of the k-th graph generated with the i-th
#' parameter. The attribute 'rownames' of the array corresponds to the
#' parameters converted to string. If spectra is NULL (default), then 'model' is
#' used to generate random graphs and their spectra are computed automatically.
#'
#' @param parameters numeric vector containing the values that that will be
#' considerated for the parameter estimation. The 'graph.param.estimator' will
#' return the element of 'parameter' that minimizes the Graph Information
#' Criterion (GIC).
#' If the user does not provide the argument 'parameters', and 'model' is an
#' array, then the values considered for the parameter estimation are the
#' rownames converted to a numeric vector. If 'model' is a string, then
#' default values are used for the predefined models ("ER", "GRG", "KR", "WS",
#' and "BA"). The default 'parameter' argument corresponds to a sequence from
#'
#' 0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in
#' which the parameter corresponds to the probability to connect a pair of
#' vertices;
#'
#' 0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in
#' which the parameter corresponds to the radius used to contruct the geometric
#' graph in a unit square;
#'
#' 0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in
#' which the parameter of the model corresponds to the degree 'k' of a regular
#' graph;
#'
#' 0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which
#' the parameter corresponds to the probability to reconnect a vertex;
#'
#' and 0 to 3 with step 'eps' for the "BA" model (Barabási-Albert model), in
#' which the parameter corresponds to the scaling exponent.
#'
#' @param eps precision of the grid (default is 0.01) when 'classic' is TRUE.
#'
#' @param bandwidth string indicating which criterion will be used to choose the
#' bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed.
#' If no value is passed, then the eigenvalues of A will be computed by
#' 'graph.param.estimator'.
#'
#' @param classic logical. If FALSE (default) parameter is estimated using ternary search.
#' If TRUE parameter is estimated using grid search.
#'
#' @return A list containing:
#' \item{p}{the parameter estimate. For the "ER", "GRG", "KR", "WS", and "BA"
#' models, the parameter corresponds to the probability to connect a pair of
#' vertices, the radius used to contruct the geometric graph in a unit square,
#' the degree k of a regular graph, the probability to reconnect a vertex, and
#' the scaling exponent, respectively.}
#' \item{KL}{the Graph Information Criterion (GIC), i. e. the Kullback-Leibler
#' divergence between the observed graph and the graph model with the estimated
#' parameter.}
#'
#' @keywords parameter_estimation
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.5)))
#'
#' # Using a string to indicate the graph model
#' result1 <- graph.param.estimator(A, "ER", eps=0.25)
#' result1
#'
#' ## Using a function to describe the graph model
#' ## Erdos-Renyi graph
#' # model <- function(n, p) {
#' #    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' # }
#' # result2 <- graph.param.estimator(A, model,  seq(0.2, 0.8, 0.1))
#' # result2
#' @export
graph.param.estimator <- function(A, model, parameters=NULL, eps=0.01,
                                  bandwidth="Silverman", eigenvalues=NULL,
                                  spectra = NULL, classic = FALSE) {
  n <- ncol(A)

  if (class(model) == "function" && is.null(parameters)) {
    stop("It is necessary to enter the parameters that will be evaluated.")
  }

  if(class(model) == "function" && classic == FALSE){
    warning("The ternary search only works for 'model' = \"ER\",\"GRG\",\"KR\",\"BA\" or \"WS\".\nChanging for classic = TRUE.")
    classic = TRUE
  }

  if (is.null(eigenvalues))
    eigenvalues <- (as.numeric(eigen(A, only.values = TRUE)$values)/
                      sqrt(nrow(A)))

  if(classic)
  {
    if (!is.null(spectra)) {
      if (is.null(parameters)) parameters <- as.numeric(rownames(spectra))
    }
    else if (is.null(parameters)) {
      if (model == "GRG") parameters <- seq(sqrt(2),0.1, -eps)
      else if (model == "BA") parameters <- seq(3,0.1, -eps)
      else if (model == "KR") parameters <- as.integer(seq(0, 1, eps)*n)
      else parameters <- seq(1, 0.1, -eps)
    }
    pmin <- -1
    klmin <- Inf
    y = c()
    for (p in parameters){
      if (!is.null(spectra)) kl <- GIC(A, spectra[as.character(p),,], p, bandwidth, eigenvalues=eigenvalues)
      else kl <- GIC(A, model, p, bandwidth, eigenvalues=eigenvalues)
      y = append(y,kl)
      if (kl < klmin) {
        klmin <- kl
        pmin <- p
      }
    }
    out <- list("p"=pmin, "GIC"=klmin)
  }
  else
  {
    if (model == "ER") intervals = list(list("lo" = 0,"hi" = 0.4),list("lo" = 0.4,"hi" = 0.6),list("lo" = 0.6,"hi" = 1))
    else if (model == "WS") intervals = list(list("lo" = 0,"hi" = 0.4),list("lo" = 0.4,"hi" = 1))
    else if (model == "BA") intervals = list(list("lo" = 0,"hi" = 1.25),list("lo" = 1.25,"hi" = 2.25),list("lo" = 2.25,"hi" = 3))
    else if (model == "KR") intervals = list(list("lo" = 0,"hi" = n))
    else if (model == "GRG") intervals = list(list("lo" = 0,"hi" = 0.8),list("lo" = 0.8,"hi" = 1.4))
    kl_res   = Inf
    for(interval in intervals)
    {
      lo = interval$lo
      hi = interval$hi
      while(TRUE)
      {
        if(abs(lo - hi) < eps || lo > hi)
        {
          pmin <- round(((lo+hi)/2), digits = 3)
          if(model == "KR") pmin = as.integer(round(pmin))
          if (!is.null(spectra)) klmin <- GIC(A, spectra[as.character(pmin),,], pmin, bandwidth, eigenvalues=eigenvalues)
          else klmin <- GIC(A, model, pmin, bandwidth, eigenvalues=eigenvalues)
          break
        }

        leftThird  <- round(((2*lo + hi)/3), digits=3)
        rightThird <- round(((lo + 2*hi)/3), digits=3)

        if(model == "KR")
        {
          leftThird  = as.integer(round(leftThird))
          rightThird = as.integer(round(rightThird))
        }
        if (!is.null(spectra)){
          cost1 <- GIC(A, spectra[as.character(leftThird),,], leftThird, bandwidth, eigenvalues=eigenvalues)
          cost2 <- GIC(A, spectra[as.character(rightThird),,], rightThird, bandwidth, eigenvalues=eigenvalues)
        }
        else{
          cost1 <- GIC(A, model, leftThird, bandwidth, eigenvalues=eigenvalues)
          cost2 <- GIC(A, model, rightThird, bandwidth, eigenvalues=eigenvalues)
        }
        if(cost1 <= cost2) hi <- rightThird
        else lo <- leftThird
      }
      if(klmin < kl_res)
      {
        pmin_res = pmin
        kl_res   = klmin
      }
    }
    out <- list("p" = pmin_res, "GIC" = kl_res)
  }
  return(out)
}

#' Graph model selection
#'
#' 'graph.model.selection' selects the graph model that best approximates the
#' observed graph according to the Graph Information Criterion (GIC).
#'
#' @param A the adjacency (symmetric) matrix of an undirected graph. For an
#' unweighted graph it contains only 0s and 1s. For a weighted graph, it
#' contains real values that correspond to the weights of the edges.
#'
#' @param models either a vector of strings, a list of functions or a list of
#' arrays describing graph models:
#'
#' A vector of strings cotaining some of the following models: "ER" (Erdos-Renyi
#' random graph), "GRG" (geometric random graph), "KR" (k regular random graph),
#' "WS" (Watts-Strogatz model), and "BA" (Barabási-Albert model).
#'
#' A list of functions. Each function returns a graph (represented by its
#' adjacency matrix) generated by a graph model and has two arguments: the graph
#' size and the model parameter, in this order.
#'
#' A list of arrays. Each elememt of the list is a three-dimensional array
#' containing the precomputed spectrum of each model. Let M be a graph model.
#' For each parameter p considered for M, the array of model M contains the
#' eigenvalues of graphs randomly generated by M with parameter p. The position
#' (i,j,k) of the array contains the j-th eigenvalue of the k-th graph
#' that generated by M with the i-th parameter. The attribute 'rownames' of
#' the array corresponds to the parameters converted to string.
#'
#' If the argument "models" is NULL, then the "ER", "WS", and "BA" models will
#' be considered for the model selection.
#'
#' @param parameters list of numeric vectors. Each vector contains the values
#' that will be considerated for the parameter estimation of the corresponding
#' model.
#' If the user does not provide the argument 'parameters', then default values
#' are used for the predefined models ("ER", "GRG", "KR", "WS", and "BA").
#' The default vector corresponds to a sequence from
#'
#' 0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in
#' which the parameter corresponds to the probability to connect a pair of
#' vertices;
#'
#' 0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in
#' which the parameter corresponds to the radius used to contruct the geometric
#' graph in a unit square;
#'
#' 0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in
#' which the parameter of the model corresponds to the degree 'k' of a regular
#' graph;
#'
#' 0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which
#' the parameter corresponds to the probability to reconnect a vertex;
#'
#' and 0 to 3 with step 'eps' for the "BA" model (Barabási-Albert model), in
#' which the parameter corresponds to the scaling exponent.
#'
#' @param eps precision of the grid (default is 0.01).
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed.
#' If no value is passed, then the eigenvalues of A will be computed by
#' 'graph.model.selection'.
#'
#' @return A list containing:
#' \item{model}{the indice(s) or name(s) of the selected model(s), i. e. the
#' model(s) that minimize(s) the Graph Information Criterion (GIC).}
#' \item{estimates}{a matrix in which each row corresponds to a model, the
#' column "p" corresponds to the parameter estimate, and the column "GIC"
#' corresponds to the Graph Information Criterion (GIC), i. e. the
#' Kullback-Leibler divergence between the observed graph and the model.}
#'
#' @keywords model_selection
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#'
#' @examples
#'
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(30, p=0.5)))
#' # Using strings to indicate the graph models
#' result1 <- graph.model.selection(A, models=c("ER", "WS"), eps=0.5)
#' result1
#' # Using functions to describe the graph models
#' # Erdos-Renyi graph
#' model1 <- function(n, p) {
#'    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' }
#' # Watts-Strougatz graph
#' model2 <- function(n, pr, K=8) {
#'     return(as.matrix(get.adjacency(watts.strogatz.game(1, n, K, pr))))
#' }
#' parameters <- list(seq(0, 1, 0.5), seq(0, 1, 0.5))
#' result2 <- graph.model.selection(A, list(model1, model2), parameters)
#' result2
#' @export
graph.model.selection <- function(A, models=NULL, parameters=NULL, eps=0.01,
                                  bandwidth="Silverman", eigenvalues=NULL) {
  n <- ncol(A)
  if (class(models) == "list" && is.null(parameters)) {
    stop("It is necessary to enter the parameters that will be evaluated.")
  }
  if (is.null(models)) {
    models <- c("ER", "WS", "BA")
  }
  results <- matrix(NA, length(models), 2)
  colnames(results) <- c("p", "GIC")
  if (class(models) == "character") {
    rownames(results) <- models
  }
  if (class(models) == "list") {
    if (!is.null(names(parameters)))
      rownames(results) <- names(parameters)
  }
  p <- NULL
  if (is.null(eigenvalues))
    eigenvalues <- (as.numeric(eigen(A, only.values=TRUE,
                                     symmetric=TRUE)$values)/sqrt(nrow(A)))
  for (i in 1:length(models)) {
    if (!is.null(parameters))
      p <- parameters[[i]]
    r <- graph.param.estimator(A, models[[i]], p, eps, bandwidth,
                               eigenvalues=eigenvalues)
    results[i, "p"] <- r$p
    results[i, "GIC"] <- r$GIC
  }
  m <- which(results[, "GIC"] == min(results[, "GIC"]))
  if (!is.null(rownames(results)))
    m <- rownames(results)[m]
  return(list("model"=m, "estimates"=results))
}

#' Test for the Jensen-Shannon divergence between graphs
#'
#' 'graph.test' tests whether two sets of graphs were generated by the same
#' random graph model.
#' This bootstrap test is based on the Jensen-Shannon (JS) divergence between
#' graphs.
#'
#' Given two lists of graphs, 'x' and 'y', 'graph.test' tests H0: "JS divergence
#' between 'x' and 'y' is 0" against H1: "JS divergence between 'x' and 'y' is
#' larger than 0".
#'
#' @param x a list of adjacency (symmetric) matrices. For unweighted graphs,
#' each matrix contains only 0s and 1s. For weighted graphs, each matrix
#' contains real values that correspond to the weights of the edges.
#'
#' @param y a list of adjacency (symmetric) matrices. For unweighted graphs,
#' each matrix contains only 0s and 1s. For weighted graphs, each matrix
#' contains real values that correspond to the weights of the edges.
#'
#' @param numBoot integer indicating the number of bootstrap resamplings.
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{JS}{the Jensen-Shannon divergence between 'x' and 'y'.}
#' \item{p.value}{the p-value of the test.}
#'
#' @keywords graph_comparison
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' library(igraph)
#' x <- y <- list()
#' for (i in 1:20)
#'    x[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.5)))
#' for (i in 1:20)
#'    y[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.51)))
#'
#' result <- graph.test(x, y, numBoot=100)
#' result
#'
#' @export
graph.test <- function(x, y, numBoot=1000, bandwidth="Silverman") {
  adjacencyMatrices <- append(x, y)
  labels <- c(rep(0, length(x)), rep(1, length(y)))
  f <- nSpectralDensities(adjacencyMatrices, bandwidth=bandwidth)
  densities <- f$densities
  x <- f$x
  y1 <- rowMeans(densities[, labels==0])
  y2 <- rowMeans(densities[, labels==1])
  n1 <- length(which(labels==0))
  n2 <- length(which(labels==1))
  results <- vector(length=numBoot)
  ngraphs <- length(adjacencyMatrices)
  result <- JS(list("x"=x, "y"=y1), list("x"=x, "y"=y2))
  for (i in 1:numBoot) {
    b1 <- sample(1:ngraphs, n1, replace=TRUE)
    b2 <- sample(1:ngraphs, n2, replace=TRUE)
    y1 <- rowMeans(densities[, b1])
    y2 <- rowMeans(densities[, b2])
    results[i] <- JS(list("x"=x, "y"=y1), list("x"=x, "y"=y2))
  }
  pvalue <- (sum(results >= result))/numBoot
  return(list("JS"=result, "p.value"=pvalue))
}

#' ANOGVA Analysis Of Graph Variability
#'
#' 'anogva' statistically tests whether two or more sets of graphs are generated
#' by the same random graph model. It is a generalization of the 'graph.test'
#' function.
#'
#' @param graphs a list of adjacency (symmetric) matrices of undirected graphs.
#' For unweighted graphs, each matrix contains only 0s and 1s. For weighted
#' graphs, each matrix may contain real values that correspond to the weights of
#' the edges.
#'
#' @param labels an array of integers indicating the labels of each graph.
#'
#' @param numBoot integer indicating the number of bootstrap resamplings.
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{statistic}{the statistic of the test.}
#' \item{p.value}{the p-value of the test.}
#'
#' @keywords analysis_of_graph_variability
#'
#' @examples
#'
#' require(igraph)
#' g1 <- g2 <- g3 <- list()
#' for (i in 1:20) {
#'    G1 <- erdos.renyi.game(50, 0.50)
#'    g1[[i]] <- get.adjacency(G1)
#'    G2 <- erdos.renyi.game(50, 0.50)
#'    g2[[i]] <- get.adjacency(G2)
#'    G3 <- erdos.renyi.game(50, 0.52)
#'    g3[[i]] <- get.adjacency(G3)
#' }
#' g <- c(g1, g2, g3)
#' label <- c(rep(1,20),rep(2,20),rep(3,20))
#' result <- anogva(g, label, numBoot=50)
#' result
#'
#' @references
#'
#' Fujita, A., Vidal, M. C. and Takahashi, D. Y. (2017) A Statistical Method to
#' Distinguish Functional Brain Networks. _Front. Neurosci._, *11*, 66.
#' doi:10.3389/fnins.2017.00066.
#'
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#' @export
anogva <- function(graphs, labels, numBoot=1000, bandwidth="Silverman") {
  f <- nSpectralDensities(graphs, bandwidth=bandwidth)
  densidade <- f$densities
  x_axis <- f$x
  band <- length(x_axis)

  media <- matrix(0, max(labels), band)
  mediaAll <- list()
  mediaAll$y <- array(0,band)
  for (j in 1:band) {
    for (i in 1:max(labels)) {
      media[i,j] <- mean(densidade[j, which(labels==i)])
    }
    mediaAll$y[j] <- mean(media[,j])
  }
  mediaAll$x <- x_axis

  distOrig <- 0
  meanGroup <- list()
  meanGroup$x <- x_axis

  for(i in 1:max(labels)) {
    meanGroup$y <- media[i,]
    distOrig <- distOrig + KL(meanGroup, mediaAll)
  }
  distOrig <- distOrig / max(labels)


  ## Permutation test
  distBoot <- array(0, numBoot)

  for (boot in 1:numBoot) {
    labelsB <- sample(labels, length(labels), replace=FALSE)

    mediaB <- matrix(0, max(labels), band)

    for (j in 1:band) {
      for (i in 1:max(labels)) {
        mediaB[i,j] <- mean(densidade[j, which(labelsB==i)])
      }
    }

    meanBoot <- list()
    meanBoot$x <- x_axis
    for(i in 1:max(labelsB)) {
      meanBoot$y <- mediaB[i,]
      distBoot[boot] <- distBoot[boot] + KL(meanBoot, mediaAll)
    }
    distBoot[boot] <- distBoot[boot] / max(labels)
  }

  pvalue <- length(which(distBoot >= distOrig)) / (numBoot+1)

  return(list("statistic"=distOrig, "p.value"=pvalue))
}

#' Semi-Parametric Analysis Of Graph Variability (ANOGVA)
#'
#' 'sp.anogva' statistically tests whether two or more graphs are
#' generated by the same model and set of parameters.
#'
#' @param graph a list of adjacency (symmetric) matrices of the undirected
#' graphs to be compared. For unweighted graphs, each matrix contains only 0s
#' and 1s. For weighted graphs, each matrix contains real values that correspond
#' to the weights of the edges.
#'
#' @param model A string that indicates one of the following models: "ER" (Erdös-Rényi random graph model),
#' "GRG" (geometric random graph model), "WS" (Watts-Strogatz random graph model),
#' and "BA" (Barabási-Albert random graph model).
#'
#' @param maxBoot integer indicating the number of bootstrap resamples (default is 500).
#'
#' @param spectra optional parameter containing the precomputed spectrum of the
#' model. It is a three-dimensional array in which the first dimension
#' corresponds to all parameters that will be explored in the parameter estimation,
#' the second dimension has the same size of the given graph, and the third one
#' corresponds to graphs randomly generated by the model. Thus, the position (i,j,k)
#' contains the j-th eigenvalue of the k-th graph generated with the i-th
#' parameter. The attribute 'rownames' of the array corresponds to the
#' parameters converted to string. If spectra is NULL (default), then 'model' is
#' used to generate random graphs and their spectra are computed automatically.
#'
#' @param eps (default is 0.01) precision of the grid when 'classic' = TRUE.
#'
#' @param classic logical. If FALSE (default) parameter is estimated using ternary search,
#' if TRUE parameter is estimated using grid search.
#'
#' @return A list containing:
#' \item{parameter}{an array containing the estimated parameters for each graph.}
#' \item{F.value}{the F statistic of the test.}
#' \item{p.value}{the p-value of the test.}
#'
#' @keywords semi_parametric_analysis_of_graph_variability
#'
#' @examples
#'
#' ## Please uncomment the following lines to run an example
#' # require(igraph)
#' # set.seed(42)
#' # model <- "ER"
#' # graph <- list()
#'
#' ## Under H0
#' # graph[[1]] <- get.adjacency(erdos.renyi.game(50, 0.5))
#' # graph[[2]] <- get.adjacency(erdos.renyi.game(50, 0.5))
#' # graph[[3]] <- get.adjacency(erdos.renyi.game(50, 0.5))
#' # result <- sp.anogva(graph, model, maxBoot = 300)
#' # result
#'
#' ## Under H1
#' # graph[[1]] <- get.adjacency(erdos.renyi.game(50, 0.5))
#' # graph[[2]] <- get.adjacency(erdos.renyi.game(50, 0.55))
#' # graph[[3]] <- get.adjacency(erdos.renyi.game(50, 0.5))
#' # result <- sp.anogva(graph, model, maxBoot = 300)
#' # result
#'
#' @export
sp.anogva <- function(graph, model, maxBoot=500, spectra = NULL, eps = 0.01, classic = FALSE) {
  g <- length(graph)
  p.hat <- list()
  for(l in 1:g) {
    n <- as.character(ncol(graph[[l]]))
    if(class(spectra) == "list") p.hat[[l]] <- graph.param.estimator(graph[[l]], model, eps=eps, spectra = spectra[[n]], classic = classic)$p
    else p.hat[[l]] <- graph.param.estimator(graph[[l]], model, eps=eps, spectra = spectra, classic = classic)$p
  }

  p.boot <- matrix(0, length(graph), maxBoot)
  for (boot in 1:maxBoot) {
    g.boot <- list()
    for (l in 1:g) {
      n <- as.character(ncol(graph[[l]]))
      if (model == "ER") {
        g.boot[[l]] <- ER(ncol(graph[[l]]), p.hat[[l]])
      }
      if (model == "GRG") {
        g.boot[[l]] <- GRG(ncol(graph[[l]]), p.hat[[l]])
      }
      if (model == "WS") {
        g.boot[[l]] <- WS(ncol(graph[[l]]),p.hat[[l]],2)
      }
      if (model == "BA") {
        g.boot[[l]] <- BA(ncol(graph[[l]]), p.hat[[l]])
      }

      if(class(spectra) == "list") p.boot[l,boot] <- graph.param.estimator(g.boot[[l]], model, eps=eps, spectra = spectra[[n]], classic = classic)$p
      else p.boot[l,boot] <- graph.param.estimator(g.boot[[l]], model, eps=eps, spectra = spectra, classic = classic)$p

    }
  }

  var.boot <- array(0, g)
  for(l in 1:g) {
    var.boot[l] <- var(p.boot[l,])
  }

  SSres <- 0
  SStr <- 0
  m <- mean(as.numeric(as.array(p.hat)))
  for(l in 1:g) {
    SSres <- SSres + (maxBoot-1) * var.boot[l]
    SStr <- SStr + (p.hat[[l]] - m)^2
  }

  F_ <- (SStr / (g-1)) / ( SSres / (g*maxBoot - g))

  p <- pf(F_, df1=(g-1), df2=(g*maxBoot - g), lower.tail=FALSE)

  res <- list()
  res$parameter <- as.array(p.hat)
  res$F.value <- F_
  res$p.value <- p
  return(res)
}

#' Test for Association / Correlation Between Paired Samples of Graphs
#'
#' 'graph.cor.test' tests for association between paired samples of graphs,
#' using Spearman's rho correlation coefficient.
#'
#' @param x a list of adjacency (symmetric) matrices of undirected graphs.
#' For unweighted graphs, each matrix contains only 0s and 1s. For weighted
#' graphs, each matrix contains real values that correspond to the weights of
#' the edges.
#'
#' @param y a list with the same length of 'x'. It contains adjacency
#' (symmetric) matrices of undirected graphs. For unweighted graphs, each matrix
#' contains only 0s and 1s. For weighted graphs, each matrix may contain
#' real values that correspond to the weights of the edges.
#'
#' @return
#' \item{statistic}{ the value of the test statistic.}
#' \item{p.value}{the p-value of the test.}
#' \item{estimate}{the estimated measure of association 'rho'.}
#'
#' @keywords correlation_coefficient
#'
#' @references
#' Fujita, A., Takahashi, D. Y., Balardin, J. B., Vidal, M. C. and Sato, J. R.
#' (2017) Correlation between graphs with an application to brain network
#' analysis. _Computational Statistics & Data Analysis_ *109*, 76–92.
#'
#' @examples
#' require(igraph)
#' x <- list()
#' y <- list()
#'
#' p <- MASS::mvrnorm(50, mu=c(0,0), Sigma=matrix(c(1, 0.5, 0.5, 1), 2, 2))
#'
#' ma <- max(p)
#' mi <- min(p)
#' p[,1] <- (p[,1] - mi)/(ma - mi)
#' p[,2] <- (p[,2] - mi)/(ma - mi)
#'
#' for (i in 1:50) {
#'     x[[i]] <- get.adjacency(erdos.renyi.game(50, p[i,1]))
#'     y[[i]] <- get.adjacency(erdos.renyi.game(50, p[i,2]))
#' }
#'
#' graph.cor.test(x, y)
#'
#' @import stats
#' @import MASS
#' @export
graph.cor.test <- function(x, y) {

  x.radius <- array(0, length(x))
  y.radius <- array(0, length(y))

  for (i in 1:length(x)) {
    x.radius[i] <- eigen(x[[i]], only.values=TRUE, symmetric=TRUE)$values[1]
    y.radius[i] <- eigen(y[[i]], only.values=TRUE, symmetric=TRUE)$values[1]
  }

  return(cor.test(x.radius, y.radius, method="spearman"))
}

#' Auto Correlation Function Estimation for Graphs
#'
#' The function 'graph.acf' computes estimates of the autocorrelation function
#' for graphs.
#'
#' @param x a list of adjacency (symmetric) matrices of undirected graphs. For
#' unweighted graphs, each matrix contains only 0s and 1s. For weighted graphs,
#' each matrix may contains real values that correspond to the weights of the
#' edges.
#'
#' @param plot logical. If TRUE (default) the graph.acf is plotted.
#'
#' @return An object of class acf.
#'
#' @keywords autocorrelation
#'
#' @references
#' Fujita, A., Takahashi, D. Y., Balardin, J. B., Vidal, M. C. and Sato, J. R.
#' (2017) Correlation between graphs with an application to brain network
#' analysis. _Computational Statistics & Data Analysis_ *109*, 76–92.
#'
#' @examples
#' require(igraph)
#' x <- list()
#' p <- array(0, 100)
#' p[1:3] <- rnorm(3)
#' for (t in 4:100) {
#'     p[t] <- 0.5*p[t-3] + rnorm(1)
#' }
#' ma <- max(p)
#' mi <- min(p)
#' p <- (p - mi)/(ma-mi)
#' for (t in 1:100) {
#'     x[[t]] <- get.adjacency(erdos.renyi.game(100, p[t]))
#' }
#' graph.acf(x, plot=TRUE)
#'
#' @import stats
#' @export
graph.acf <- function(x, plot=TRUE) {
  x.radius <- array(0, length(x))
  for (t in 1:length(x)) {
    x.radius[t] <- eigen(x[[t]], only.values=TRUE, symmetric=TRUE)$values[1]
  }
  res <- acf(x.radius, plot=plot)
  return(res)
}

#' Hierarchical cluster analysis on a list of graphs.
#'
#' Given a list of graphs, 'graph.cluster' builds a hierarchy of clusters
#' according to the Jensen-Shannon divergence between graphs.
#'
#' @param x a list of adjacency (symmetric) matrices of undirected graphs. For
#' unweighted graphs, each matrix contains only 0s and 1s. For weighted graphs,
#' each matrix may contains real values that correspond to the weights of the
#' edges.
#'
#' @param k the number of clusters.
#'
#' @param method the agglomeration method to be used. This should be (an
#' unambiguous abbreviation of) one of ‘"ward.D"’, ‘"ward.D2"’, ‘"single"’,
#' ‘"complete"’, ‘"average"’ (= UPGMA), ‘"mcquitty"’ (= WPGMA), ‘"median"’
#' (= WPGMC) or ‘"centroid"’ (= UPGMC).
#'
#' @param bandwidth string indicating which criterion will be used to choose
#' the bandwidth for the spectral density estimation. The available criteria are
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{hclust}{an object of class *hclust* which describes the tree produced
#' by the clustering process.}
#' \item{cluster}{the clustering labels for each graph.}
#'
#' @keywords clustering
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' g <- list()
#' for (i in 1:5) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         erdos.renyi.game(50, 0.5, type="gnp",
#'                                          directed = FALSE)))
#' }
#' for (i in 6:10) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         watts.strogatz.game(1, 50, 8, 0.2)))
#' }
#' for (i in 11:15) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         barabasi.game(50, power = 1,
#'                                       directed = FALSE)))
#' }
#' graph.cluster(g, 3)
#'
#' @import stats
#'
#' @export
graph.cluster <- function(x, k, method="complete", bandwidth="Silverman") {

  f <- nSpectralDensities(x, bandwidth=bandwidth)

  d <- matrix(0, length(x), length(x))
  for (i in 1:(length(x)-1)) {
    f1 <- list("x"=f$x, "y"=f$densities[,i])
    for (j in (i+1) : length(x)) {
      f2 <- list("x"=f$x, "y"=f$densities[,j])
      d[i,j] <- d[j,i] <- sqrt(JS(f1, f2))
    }
  }

  tmp <- hclust(as.dist(d), method=method)

  res <- list()
  res$hclust <- tmp
  res$cluster <- cutree(tmp, k)

  return(res)
}

#' Multidimensional scaling of graphs
#'
#' 'graph.mult.scaling' performs multidimensional scaling of graphs. It takes
#' the Jensen-Shannon divergence between graphs (JS) and uses the 'cmdscale'
#' function from the 'stats' package to obtain a set of points such that the
#' distances between the points are similar to JS.
#'
#' @param x a list of adjacency (symmetric) matrices of undirected graphs. For
#' unweighted graphs, each matrix contains only 0s and 1s. For weighted graphs,
#' each matrix may contains real values that correspond to the weights of the
#' edges.
#'
#' @param plot logical. If TRUE (default) the points chosen to represent the
#' Jensen-Shannon divergence between graphs are plotted.
#'
#' @param bandwidth character string indicating which criterion will be used
#' to choose the bandwidth for the spectral density estimation. The available
#' criteria are "Silverman" (default) and "Sturges".
#'
#' @param type what type of plot should be drawn. The defaut value is '"n"',
#' which indicates that the points will not be plotted (i. e. only the labels
#' of the graphs will be plotted).
#'
#' @param main title of the plot (default value is "").
#'
#' @param ... additional plotting parameters. See 'plot' function from the
#' 'graphics' package for the complete list.
#'
#' @return A matrix in which each column corresponds to a coordinate and each
#' row corresponds to a graph (point). Then, each row gives the coordinates of
#' the points chosen to represent the Jensen-Shannon divergence between graphs.
#'
#' @keywords multidimensional_scaling
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' g <- list()
#' for (i in 1:5) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         erdos.renyi.game(50, 0.5, type="gnp",
#'                                          directed = FALSE)))
#' }
#' for (i in 6:10) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         watts.strogatz.game(1, 50, 8, 0.2)))
#' }
#' for (i in 11:15) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         barabasi.game(50, power = 1,
#'                                       directed = FALSE)))
#' }
#' graph.mult.scaling(g)
#'
#' @import graphics
#' @export
graph.mult.scaling <- function(x, plot=TRUE, bandwidth="Silverman", type="n",
                               main="", ...) {

  f <- nSpectralDensities(x, bandwidth=bandwidth)

  d <- matrix(0, length(x), length(x))
  for (i in 1:(length(x)-1)) {
    f1 <- list("x"=f$x, "y"=f$densities[,i])
    for (j in (i+1) : length(x)) {
      f2 <- list("x"=f$x, "y"=f$densities[,j])
      d[i,j] <- d[j,i] <- sqrt(JS(f1, f2))
    }
  }

  if (is.null(names(x)))
    names <- as.character(1:length(x))
  else
    names <- names(x)
  colnames(d) <- rownames(d) <- names
  fit <- cmdscale(as.dist(d), k=2)

  x <- fit[,1]
  y <- fit[,2]
  names(x) <- names
  names(y) <- names
  if (plot) {
    plot(x, y, xlab="x", ylab="y", main=main, type=type, ...)
    text(x, y, labels=names, cex=1)
  }
  return(fit)
}

#' Tang hypothesis testing for random graphs.
#'
#' Given two independent finite-dimensional random dot product graphs, 'tang' tests if they have generating latent positions that are
#' drawn from the same distribution.
#'
#' @param G1 the first undirected graph to be compared. Must be an igraph object.
#'
#' @param G2 the second undirected graph to be compared. Must be an igraph object.
#'
#' @param dim dimension of the adjacency spectral embedding.
#'
#' @param sigma a real value indicating the kernel bandwidth. If NULL (default) the bandwidth is calculated by the method.
#'
#' @param alpha the significance level for the test (default is 0.05).
#'
#' @param bootstrap_sample integer indicating the number of bootstrap resamples (default is 200).
#'
#' @param printResult logical indicating if the test must print the result (default is FALSE).
#'
#' @return A list containing:
#' \item{X1}{the embedding of G1.}
#' \item{X2}{the embedding of G2.}
#' \item{test_stats}{the value of the test.}
#' \item{p_value}{the p-value of the test.}
#' \item{bootstrap_samples}{The test distrition on the bootstrap resamples.}
#'
#' @references
#' Tang, Minh, et al. "A nonparametric two-sample hypothesis testing problem for random graphs." Bernoulli 23.3 (2017): 1599-1630.
#'
#' Tang, Minh, et al. "A semiparametric two-sample hypothesis testing problem for random graphs." Journal of Computational and Graphical Statistics 26.2 (2017): 344-354.
#'
#' @examples
#' require(igraph)
#' set.seed(42)
#'
#' ## test under H0
#' lpvs <- matrix(rnorm(200), 20, 10)
#' lpvs <- apply(lpvs, 2, function(x) { return (abs(x)/sqrt(sum(x^2))) })
#' g1 <- sample_dot_product(lpvs)
#' g2 <- sample_dot_product(lpvs)
#' D <- tang(g1,g2, 5, printResult = TRUE)
#'
#' ## test under H1
#' lpvs2 <- matrix(pnorm(200), 20, 10)
#' lpvs2 <- apply(lpvs2, 2, function(x) { return (abs(x)/sqrt(sum(x^2))) })
#' g2 <- suppressWarnings(sample_dot_product(lpvs2))
#' D <- tang(g1,g2, 5, printResult = TRUE)
#'
#'
#' @export
tang <- function(G1, G2, dim, sigma = NULL, alpha = 0.05, bootstrap_sample = 200, printResult = FALSE){
  t.validateInput(G1, G2, dim, alpha, bootstrap_sample, printResult)
  Xhat1 = t.embed.graph(G1, dim)
  Xhat2 = t.embed.graph(G2, dim)
  if(is.null(sigma)){
    sigma = t.get.sigma(Xhat1, Xhat2)
  }
  test_stat = t.test.stat(Xhat1, Xhat2, sigma)
  test_distribution = t.sampling.distribution(G1, dim, bootstrap_sample)
  #test_distribution2 = sampling.distribution(G2, dim, bootstrap_sample)
  p_val = t.p_value(test_stat, test_distribution)
  num = round(bootstrap_sample * alpha)
  reject_threshold = test_distribution[num]
  reject = FALSE
  if (p_val <= alpha) {
    reject = TRUE
  }
  if (printResult) {
    if (reject) {
      print("Reject the nullhypothesis that two graphs are identically distributed.")
    } else {
      print("Fail to reject the null hypothesis that two graphs are identically distributed.")
    }
  }
  out = list(X1 = Xhat1, X2 = Xhat2, test_stats = test_stat,
             p_value = p_val, bootstrap_samples = test_distribution)
  return(out)
}

#' Ghoshdastidar hypothesis testing for large random graphs.
#'
#' Given two lists of graphs generated by the inhomogeneous random graph model, 'ghoshdastidar' tests if they were generated by the same parameters.
#'
#' @param x the first list of undirected graphs to be compared. Must be a list of matrices or igraph objects.
#'
#' @param y the second list of undirected graphs to be compared. Must be a list of matrices or igraph objects.
#'
#' @param maxPer integer indicating the number of bootstrap resamples (default is 300).
#'
#' @param alpha the significance level for the test (default is 0.05).
#'
#' @param two.sample logical. If TRUE the sets contain only one graph each. If FALSE the sets contain more than one graph each (default is FALSE).
#'
#' @param printResult logical indicating if the test must print the result (default is FALSE).
#'
#
#' @return A list containing:
#' \item{test_stats}{the value of the test.}
#' \item{p_value}{the p-value of the test (only returned when the parameter 'two.sample' is FALSE).}
#' \item{bootstrap_samples}{The test distrition on the bootstrap resamples (only returned when the parameter 'two.sample' is FALSE).}
#'
#' @references
#' Ghoshdastidar, Debarghya, et al. "Two-sample tests for large random graphs using network statistics". arXiv preprint arXiv:1705.06168 (2017).
#'
#' Ghoshdastidar, Debarghya, et al. "Two-sample hypothesis testing for inhomogeneous random graphs". arXiv preprint, arXiv:1707.00833 (2017).
#'
#' @examples
#' require(igraph)
#' set.seed(42)
#'
#' ## test for sets with more than one graph each under H0
#' x <- y <- list()
#' for(i in 1:10){
#'   x[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50,0.6)))
#'   y[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50,0.6)))
#' }
#' D <- ghoshdastidar(x, y, printResult = TRUE)
#'
#' ## test for sets with more than one graph each under H1
#' x <- y <- list()
#' for(i in 1:10){
#'   x[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50,0.6)))
#'   y[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50,0.7)))
#' }
#' D <- ghoshdastidar(x, y, printResult = TRUE)
#'
#' ## test for sets with only one graph each under H0
#' x <- y <- list()
#' x[[1]] <- erdos.renyi.game(300, 0.6)
#' y[[1]] <- erdos.renyi.game(300, 0.6)
#' D <- ghoshdastidar(x, y, two.sample= TRUE, printResult = TRUE)
#'
#' ## test for sets with only one graph each under H1
#' x <- y <- list()
#' x[[1]] <- erdos.renyi.game(300, 0.6)
#' y[[1]] <- erdos.renyi.game(300, 0.7)
#' D <- ghoshdastidar(x, y, two.sample= TRUE, printResult = TRUE)
#'
#'
#' @export
ghoshdastidar <- function(x, y, maxPer = 300, alpha = 0.05, two.sample = FALSE, printResult = FALSE)
{
  if(class(x)=='list' && class(x[[1]])=='igraph'){ x <- g.transform(x) }
  if(class(y)=='list' && class(y[[1]])=='igraph'){ y <- g.transform(y) }
  if(class(x)=='igraph' && class(y)=='igraph'){
    x <- g.transform(x)
    y <- g.transform(y)
  }

  if(class(x) != 'list' || class(x[[1]]) != 'matrix') stop("x must be a list of matrices or igraph objects.")
  if(class(y) != 'list' || class(y[[1]]) != 'matrix') stop("y must be a list of matrices or igraph objects.")

  D <- g.test(x, y)

  if( !two.sample ){
    test_distribution <- g.sampling.distribution(x,y,maxPer)
    p_val <- mean(test_distribution >= D)
    if (printResult) {
      if (p_val <= alpha) {
        print("Reject the nullhypothesis that two graphs are identically distributed.")
      } else {
        print("Fail to reject the null hypothesis that two graphs are identically distributed.")
      }
    }
    out = list(test_stats = D, p_value = p_val, bootstrap_samples = test_distribution)
  }
  else{
    if (printResult) {
      if (D > 1) {
        print("Reject the nullhypothesis that two graphs are identically distributed.")
      } else {
        print("Fail to reject the null hypothesis that two graphs are identically distributed.")
      }
    }
    out = list(test_stats = D)
  }
  return(out)
}

################################################################################
## Auxiliar functions
################################################################################

# Returns the Jensen-Shannon divergence between two densities
JS <- function(f1, f2) {
  fm <- f1
  fm$y <- (f1$y + f2$y)/2
  return((KL(f1, fm) + KL(f2, fm))/2)
}

# Returns the Kullback-Leibler divergence between two densities
KL <- function(f1, f2) {
  y <- f1$y
  n <- length(y)
  for (i in 1:n) {
    if (y[i] != 0 && f2$y[i] == 0){
      return (Inf)
    }
    if (y[i] != 0)
      y[i] <- y[i]*log(y[i]/f2$y[i])
  }
  return (trapezoidSum(f1$x, y))
}

# Given a partition x[1]...x[n] and y[i] = f(x[i]), returns the trapezoid sum
# approximation for int_{x[1]}^{x[n]}{f(x)dx}
trapezoidSum <- function (x, y) {
  n <- length(x)
  delta <- (x[2] - x[1])
  area <- sum(y[2:(n-1)])
  area <- (area + (y[1] + y[n])/2)*delta
  return(area)
}

# Returns the kernel bandwidth for a given sample x
kernelBandwidth <- function(x) {
  n <- length(x)
  # Sturges' criterion
  nbins <- ceiling(log2(n) + 1)
  return(abs(max(x) - min(x))/nbins)
}

# Returns the density function for a given sample x at n points in the interval
# [form, to]
gaussianDensity <- function(x, from=NULL, to=NULL, bandwidth="Silverman",
                            npoints=1024) {
  if (bandwidth == "Sturges")
    bw <- kernelBandwidth(x)
  else if (bandwidth == "Silverman")
    bw <- bw.nrd0(x)
  if (bw == 0)
    return(NA)
  if (is.null(from) || is.null(to))
    f <- density(x, bw=bw, n=npoints)
  #f <- density(x, n=npoints)

  else
    f <- density(x, bw=bw, from=from, to=to, n=npoints)
  #f <- density(x, from=from, to=to, n=npoints)
  f$y = f$y + 1e-12
  area <- trapezoidSum(f$x, f$y) # adding 1/npoints because we didn't want to obtain a zeroin the histogram
  return(list("x"=f$x, "y"=f$y/area))
}

# Returns the spectral density for a given adjacency matrix A
spectralDensity <- function(A, from=NULL, to=NULL, bandwidth="Silverman",
                            npoints=1024) {
  eigenvalues <- (as.numeric(eigen(A, only.values=TRUE,
                                   symmetric=TRUE)$values)/sqrt(nrow(A)))
  return(gaussianDensity(eigenvalues, from, to, bandwidth=bandwidth,
                         npoints=npoints))
}

nDensities <- function (spectra, from=NULL, to=NULL,
                        bandwidth="Silverman", npoints=1024) {
  ngraphs <- ncol(spectra)
  densities <- matrix(NA, npoints, ngraphs)
  minimum <- min(spectra)
  maximum <- max(spectra)
  if (!is.null(from) && !is.null(to)) {
    minimum <- min(minimum, from)
    maximum <- max(maximum, to)
  }
  for (i in 1:ngraphs) {
    f <- gaussianDensity(spectra[,i], bandwidth=bandwidth,
                         from=minimum, to=maximum,
                         npoints=npoints)
    if (sum(is.na(f)) > 0) {
      return(NA)
    }
    else {
      densities[,i] <- f$y
      x <- f$x
    }
  }
  return(list("x"=x, "densities"=densities))
}


# Returns the spectral densities for given adjacency matrices A1 and A2 at the
# same points
spectralDensities <- function(A1, A2, bandwidth="Silverman",
                              npoints=1024) {
  n1 <- nrow(A1)
  n2 <- nrow(A2)
  e1 <- (as.numeric(eigen(A1, only.values = TRUE,
                          symmetric=TRUE)$values)/sqrt(n1))
  e2 <- (as.numeric(eigen(A2, only.values = TRUE,
                          symmetric=TRUE)$values)/sqrt(n2))
  #b1 <- kernelBandwidth(e1)
  #b2 <- kernelBandwidth(e2)
  #from <- min(min(e1) - 3*b1, min(e2) - 3*b2)
  #to <- max(max(e1) + 3*b1, max(e2) + 3*b2)
  from <- min(e1, e2)
  to <- max(e1, e2)
  f1 <- gaussianDensity(e1, from=from, to=to, bandwidth=bandwidth,
                        npoints=npoints)
  f2 <- gaussianDensity(e2, from=from, to=to, bandwidth=bandwidth,
                        npoints=npoints)
  if (sum(is.na(f1)) > 0 || sum(is.na(f2)) > 0)
    return(NA)
  return(list("f1"=f1, "f2"=f2))
}

# Returns the spectral densities for a list of adjacency matrices at the
# same points
nSpectralDensities <- function (adjacencyMatrices, from=NULL, to=NULL,
                                bandwidth="Silverman") {
  npoints <- 1024
  ngraphs <- length(adjacencyMatrices)
  ns <- unlist(lapply(adjacencyMatrices, ncol))
  #n <- ncol(adjacencyMatrices[[1]])
  spectra <- matrix(NA, max(ns), ngraphs)
  for (i in 1:ngraphs) {
    A <- adjacencyMatrices[[i]]
    n <- ncol(A)
    eigenvalues <- (as.numeric(eigen(A, only.values = TRUE,
                                     symmetric=TRUE)$values)/sqrt(n))
    spectra[1:n,i] <- eigenvalues
  }
  densities <- matrix(NA, npoints, ngraphs)
  minimum <- min(spectra, na.rm=TRUE)
  maximum <- max(spectra, na.rm=TRUE)
  if (!is.null(from) && !is.null(to)) {
    minimum <- min(minimum, from)
    maximum <- max(maximum, to)
  }
  for (i in 1:ngraphs) {
    n <- ns[i]
    f <- gaussianDensity(spectra[1:n,i], bandwidth=bandwidth,
                         from=minimum, to=maximum,
                         npoints=npoints)

    densities[,i] <- f$y
    x <- f$x
  }
  return(list("x"=x, "densities"=densities))
}

# Estimates the spectral density of a graph model
modelSpectralDensity <- function(fun, n, p, ngraphs=100, from=NULL, to=NULL,
                                 bandwidth="Silverman", npoints=1024) {
  spectra <- matrix(NA, n, ngraphs)
  for (i in 1:ngraphs) {
    A <- fun(n, p)
    eigenvalues <- (as.numeric(eigen(A, only.values = TRUE,
                                     symmetric=TRUE)$values)/sqrt(nrow(A)))
    spectra[,i] <- eigenvalues
  }
  densities <- matrix(NA, npoints, ngraphs)
  minimum <- min(spectra)
  maximum <- max(spectra)
  if (!is.null(from) && !is.null(to)) {
    minimum <- min(minimum, from)
    maximum <- max(maximum, to)
  }
  for (i in 1:ngraphs) {
    f <- gaussianDensity(spectra[,i], from=minimum, to=maximum,
                         bandwidth=bandwidth, npoints=npoints)

    densities[,i] <- f$y
    x <- f$x
  }
  return(list("x" = x, "y" = rowMeans(densities)))
}

modelSpectra <- function(model, n, p, ngraphs=50) {
  fun <- model
  if (is.character(model)) {
    if (model == "WS")
      fun <- WSfun(2)
    else fun <- matchFunction(model)
  }
  spectra <- matrix(NA, n, ngraphs)
  for (i in 1:ngraphs) {
    A <- fun(n, p)
    eigenvalues <- (as.numeric(eigen(A, only.values = TRUE,
                                     symmetric=TRUE)$values)/sqrt(nrow(A)))
    spectra[,i] <- eigenvalues
  }
  return(spectra)
}

matchFunction <- function(name) {
  return(match.fun(name))
}

# Watts-Strougatz graph
WSfun <- function(K){
  f <- function(n, pr) {
    WS(n, pr, K=K)
  }
  return(f)
}

# Graph models  ----------------------------------------------------------------

# Erdos-Renyi graph
ER <- function(n, p) {
  return(as.matrix(igraph::get.adjacency(igraph::erdos.renyi.game(n, p,
                                                                  type="gnp",
                                                                  directed = FALSE))))
}

# Geometric graph
GRG <- function(n, r) {
  return(as.matrix(igraph::get.adjacency(igraph::grg.game(n, r))))
}

# Barabasi-Albert graph
BA <- function(n, ps) {
  return(as.matrix(igraph::get.adjacency(igraph::barabasi.game(n, power = ps,
                                                               directed = FALSE))))
}

# Watts-Strogatz graph
WS <- function(n, pr, K=8) {
  return(as.matrix(igraph::get.adjacency(igraph::watts.strogatz.game(1, n, K,
                                                                     pr))))
}

# K-regular game
KR <- function(n, k) {
  return(as.matrix(igraph::get.adjacency(igraph::k.regular.game(n, k))))
}

# new cost function method
# @eigenvalues -> the normalized eigenvalues of the graph of which we want its parameter
# @f           -> the p.d.f of the model
new_cost <- function(eigenvalues, f){
  sum = 0
  for(lambda in eigenvalues){
    found = -1
    dist = Inf
    for(k in 1:length(f$x)){
      if(abs(lambda - f$x[k]) < dist){
        dist = abs(lambda - f$x[k])
        found = k
      }
    }
    if(found != -1)
      sum = sum + log(f$y[found] + 1e-10)
    else
      sum = sum + log(1e-10)
  }
  return (sum)
}

# GIC with repetitions
k_GIC <- function(A, model, p=NULL, bandwidth="Silverman", eigenvalues=NULL,k = 10) {
  gic_estimator = c()
  for(i  in 1:10){
    gic_estimator = append(gic_estimator,GIC(A,model,p,bandwidth,eigenvalues))
  }
  return (mean(gic_estimator))
}

# L2 distance
distance <- function(f1,f2){
  y <- abs(f1$y - f2$y)
  return (trapezoidSum(f1$x,y))
}

## Auxiliary for Tang method.
t.test.stat <- function(X, Y, sigma) {
  n <- nrow(X)
  m <- nrow(Y)
  tmpXX <- sum(exp(-(as.matrix(stats::dist(X))^2)/(2*sigma^2)))
  tmpYY <- sum(exp(-(as.matrix(stats::dist(Y))^2)/(2*sigma^2)))
  tmpXY <- sum(exp(-(t.rect.dist(X,Y))/(2*sigma^2)))
  tmp <- tmpXX/(n*(n-1)) + tmpYY/(m*(m-1)) - 2*tmpXY/(m*n)
  return((m+n)*tmp)
}

## Auxiliary for Tang method.
t.embed.graph <- function(g, dim) {
  # Call ase to get latent position
  #lpv = embed.ase(g, dim)$X
  igraph.arpack.default$maxiter = .Machine$integer.max
  lpv = igraph::embed_adjacency_matrix(g,dim, options = igraph.arpack.default)$X

  # Fix signs of eigenvectors issue
  for (i in 1:dim) {
    if (sign(lpv[1, i]) != 1) {
      lpv[, i] = -lpv[, i]
    }
  }
  return(lpv)
}

## Auxiliary for Tang method.
t.rect.dist <- function(X,Y) {
  X <- as.matrix(X)
  Y <- as.matrix(Y)
  n <- nrow(X)
  m <- nrow(Y)
  tmp1 <- X%*%t(Y)
  tmp2 <- outer(rep(1, n), rowSums(Y^2))
  tmp3 <- outer(rowSums(X^2), rep(1,m))

  D <- tmp2 - 2*tmp1 + tmp3
  return(D)
}

## Auxiliary for Tang method.
t.get.sigma <- function(X1, X2) {
  v1 = as.vector(stats::dist(X1))
  v2 = as.vector(stats::dist(X2))
  v = base::append(v1, v2)
  sigma = stats::median(v)
  return(sigma)
}

## Auxiliary for Tang method.
t.sampling.distribution <- function(G1, dim, bootstrap_sample_size) {
  Xhat1 = t.embed.graph(G1,dim)
  P = t(Xhat1)
  test_distribution = c()
  i = 1
  while (i <= bootstrap_sample_size) {
    tryCatch({
      G_a = suppressWarnings(igraph::sample_dot_product(P))
      G_b = suppressWarnings(igraph::sample_dot_product(P))
      Xhat_a = suppressWarnings(t.embed.graph(G_a, dim))
      Xhat_b = suppressWarnings(t.embed.graph(G_b, dim))
      sigma = t.get.sigma(Xhat_a, Xhat_b)
      ts = t.test.stat(Xhat_a, Xhat_b, sigma)
      test_distribution[i] = ts
      i = i + 1
    }, error=function(e) {stop(print(e))})
  }
  test_distribution
}

## Auxiliary for Tang method.
t.p_value <- function(ts, test_distribution) {
  area = sum(test_distribution >= ts) / length(test_distribution)
  return(area)
}

## Auxiliary for Tang method.
t.validateInput <- function(G1, G2, dim, alpha, bootstrap_sample, printResult) {

  if (class(G1) == "dgCMatrix") { G1 = igraph::graph_from_adjacency_matrix(G1) }
  if (class(G1) == "matrix") { G1 = igraph::graph_from_adjacency_matrix(G1) }
  if (class(G1) != 'igraph') { stop("Input object 'G1' is not an igraph object.") }
  if (class(G2) == "dgCMatrix") { G2 = igraph::graph_from_adjacency_matrix(G2) }
  if (class(G2) == "matrix") { G2 = igraph::graph_from_adjacency_matrix(G2) }
  if (class(G2) != 'igraph') { stop("Input object 'G2' is not an igraph object.") }
  if (!is.null(dim)) {
    if (class(dim) != "numeric" && !is.integer(dim)) { stop("Input 'dim' is not a number.") }
    if (dim%%1 != 0) { stop("Input 'dim' must be an integer.") }
    if (length(dim) > 1) { stop("Input 'dim' has length > 1.") }
    if (dim < 1) { stop("Number of dimensions 'dim' is less than 1.") }
    if (dim >= igraph::gorder(G1) || dim >= igraph::gorder(G2)) { stop("Num. Embedded dimensions 'dim' is greater or equal than number of vertices.") }
  }

  if (class(alpha) != "numeric") {
    stop("Input object 'alpha' is not a numeric value.")
  } else if (length(alpha) != 1) {
    stop("Input object 'alpha' is not a numeric value.")
  } else {
    if (alpha >= 1 || alpha <= 0) {
      stop("Significance level alpha must be strictly between 0 and 1.")
    }
  }
  if (class(bootstrap_sample) != "numeric") {
    stop("Input object 'bootstrap_sample' is not a numeric value.")
  } else if (length(bootstrap_sample) != 1) {
    stop("Input object 'bootstrap_sample' is not a numeric value.")
  } else {
    if (bootstrap_sample <= 20) {
      stop("The size of bootstrap sample is too small. Pick a larger value.")
    }
  }
  if (!is.logical(printResult)) { stop("Error: Input 'printResult' must be a logical.")}
}

# Auxiliary for Ghoshdastidar method. Ghoshdastidar test itself according to article.
g.test <- function(x, y){
  m <- min(length(x),length(y))

  x <- lapply(x, function(s){
    s2 <- s
    s2[lower.tri(s2)]<-0
    eval.parent(substitute(s<-s2))
  })

  y <- lapply(y, function(s){
    s2 <- s
    s2[lower.tri(s2)]<-0
    eval.parent(substitute(s<-s2))
  })

  n <- dim(x[[1]])[1]
  Sm1 <- matrix(0,1,m)
  Sp1 <- matrix(0,1,m)
  Sm2 <- matrix(0,1,m)
  Sp2 <- matrix(0,1,m)

  for(i in 1:m){
    Sm1[i] <- sum(x[[i]]*(x[[i]]%*%x[[i]]))/(6*choose(n,3))
    Sp1[i] <- Sm1[i]*log(n)/choose(n,3)

    Sm2[i] <- sum(y[[i]]*(y[[i]]%*%y[[i]]))/(6*choose(n,3))
    Sp2[i] <- Sm2[i]*log(n)/choose(n,3)
  }

  num <- abs(sum(Sm1-Sm2));
  den <- 2*(sqrt(sum(Sp1))+sqrt(sum(Sp2)))

  if(den == 0) den <- 1
  stat <- abs(num/den)
  return(stat)
}


## Auxiliary for Ghoshdastidar method. Boostrap for the test.
g.sampling.distribution <- function(x, y, maxPer = 300)
{
  m1 <- length(x)
  m2 <- length(y)
  test_distribution = c()
  for (i_per in 1:maxPer){
    xe <- sample(append(x,y), m1, replace = TRUE)
    ye <- sample(append(x,y), m2, replace = TRUE)
    test_distribution[i_per] <- g.test(xe, ye)
  }
  return(sort(test_distribution))
}

## Auxiliary for Ghoshdastidar method. Padronize input.
g.transform <- function(g)
{
  if(class(g)=='igraph'){
    return(list(as.matrix(igraph::get.adjacency(g))))
  }
  else{
    result <- lapply(g, function(x){
      return(as.matrix(igraph::get.adjacency(x)))
    }
    )}
}
