% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cholUpdate.R
\name{cholUpdate}
\alias{cholUpdate}
\alias{cholUpdateRankOne}
\alias{cholUpdateDel}
\alias{cholUpdateDelBlock}
\title{Different Cholesky factor updates}
\usage{
cholUpdateRankOne(A, v, alpha, beta, lower = TRUE)

cholUpdateDel(A, del.index, lower = TRUE)

cholUpdateDelBlock(A, del.start, del.end, lower = TRUE)
}
\arguments{
\item{A}{an \eqn{n\times n} triangular matrix}

\item{v}{an \eqn{n\times 1} matrix/vector}

\item{alpha}{scalar; if not supplied, default is 1}

\item{beta}{scalar; if not supplied, default is 1}

\item{lower}{logical; if \code{A} is lower-triangular or not}

\item{del.index}{an integer from 1 to \eqn{n} indicating the row/column to be
deleted}

\item{del.start}{an integer from 1 to \eqn{n} indicating the first row/column
of a block to be deleted, must be at least 1 less than \code{del.end}}

\item{del.end}{an integer from 1 to \eqn{n} indicating the last row/column
of a block to be deleted, must be at least 1 more than \code{del.start}}
}
\value{
An \eqn{m \times m} lower-triangular \code{matrix} with \eqn{m = n} in
case of \code{cholUpdateRankOne()}, \eqn{m = n - 1} in case of \code{cholUpdateDel()},
and, \eqn{m = n - n_k} in case of \code{cholUpdateDelBlock()} where \eqn{n_k} is
the size of the block removed.
}
\description{
Provides functions that implements different types of updates of
a Cholesky factor that includes rank-one update, single row/column deletion
update and a block deletion update.
}
\details{
Suppose \eqn{B = AA^\top} is a \eqn{n \times n} matrix with \eqn{A}
being its lower-triangular Cholesky factor. Then rank-one update corresponds
to finding the Cholesky factor of the matrix
\eqn{C = \alpha B + \beta vv^\top} for some \eqn{\alpha,\beta\in\mathbb{R}}
given \eqn{A} (see, Krause and Igel 2015). Similarly, single row/column
deletion update corresponds to finding the Cholesky factor of the
\eqn{(n-1)\times(n-1)} matrix \eqn{B_i} which is obtained by removing the
\eqn{i}-th row and column of \eqn{B}, given \eqn{A} for some
\eqn{i - 1, \ldots, n}. Lastly, block deletion corresponds to finding the
Cholesky factor of the \eqn{(n-n_k)\times(n-n_k)} matrix \eqn{B_{I}} for a
subset \eqn{I} of \eqn{\{1, \ldots, n\}} containing \eqn{n_k} consecutive
indices, given the factor \eqn{A}.
}
\examples{
n <- 10
A <- matrix(rnorm(n^2), n, n)
A <- crossprod(A)
cholA <- chol(A)

## Rank-1 update
v <- 1:n
APlusvvT <- A + tcrossprod(v)
cholA1 <- t(chol(APlusvvT))
cholA2 <- cholUpdateRankOne(cholA, v, lower = FALSE)
print(all(abs(cholA1 - cholA2) < 1E-9))

## Single Row-deletion update
ind <- 2
A1 <- A[-ind, -ind]
cholA1 <- t(chol(A1))
cholA2 <- cholUpdateDel(cholA, del.index = ind, lower = FALSE)
print(all(abs(cholA1 - cholA2) < 1E-9))

## Block-deletion update
start_ind <- 2
end_ind <- 6
del_ind <- c(start_ind:end_ind)
A1 <- A[-del_ind, -del_ind]
cholA1 <- t(chol(A1))
cholA2 <- cholUpdateDelBlock(cholA, start_ind, end_ind, lower = FALSE)
print(all(abs(cholA1 - cholA2) < 1E-9))
}
\references{
Oswin Krause and Christian Igel. 2015. "A More Efficient Rank-one
Covariance Matrix Update for Evolution Strategies". In \emph{Proceedings of the
2015 ACM Conference on Foundations of Genetic Algorithms XIII} (FOGA '15).
Association for Computing Machinery, New York, NY, USA, 129-136.
\doi{10.1145/2725494.2725496}.
}
\author{
Soumyakanti Pan \href{mailto:span18@ucla.edu}{span18@ucla.edu},\cr
Sudipto Banerjee \href{mailto:sudipto@ucla.edu}{sudipto@ucla.edu}
}
