% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metrics-quantile.R
\name{quantile_score}
\alias{quantile_score}
\title{Quantile score}
\usage{
quantile_score(observed, predicted, quantile_level, weigh = TRUE)
}
\arguments{
\item{observed}{Numeric vector of size n with the observed values.}

\item{predicted}{Numeric nxN matrix of predictive
quantiles, n (number of rows) being the number of forecasts (corresponding
to the number of observed values) and N
(number of columns) the number of quantiles per forecast.
If \code{observed} is just a single number, then predicted can just be a
vector of size N.}

\item{quantile_level}{Vector of of size N with the quantile levels
for which predictions were made.}

\item{weigh}{Logical. If \code{TRUE} (the default), weigh the score by
\eqn{\alpha / 2}, so it can be averaged into an interval score that, in
the limit (for an increasing number of equally spaced quantiles/prediction
intervals), corresponds
to the CRPS. \eqn{\alpha} is the value that corresponds to the
(\eqn{\alpha/2}) or (\eqn{1 - \alpha/2}), i.e. it is the decimal
value that represents how much is outside a central prediction interval
(E.g. for a 90 percent central prediction interval, alpha is 0.1).}
}
\value{
Numeric vector of length n with the quantile score. The scores are
averaged across quantile levels if multiple quantile levels are provided
(the result of calling \code{rowMeans()} on the matrix of quantile scores that
is computed based on the observed and predicted values).
}
\description{
Proper Scoring Rule to score quantile predictions. Smaller values are better.
The quantile score is closely related to the interval score (see \code{\link[=wis]{wis()}}) and
is the quantile equivalent that works with single quantiles instead of
central prediction intervals.

The quantile score, also called pinball loss, for a single quantile
level \eqn{\tau} is defined as
\deqn{
  \text{QS}_\tau(F, y) = 2 \cdot \{ \mathbf{1}(y \leq q_\tau) - \tau\} \cdot (q_\tau - y) =
  \begin{cases}
2 \cdot (1 - \tau) * q_\tau - y,       & \text{if } y \leq q_\tau\\
2 \cdot \tau * |q_\tau - y|,           & \text{if } y > q_\tau,
\end{cases}
}
with \eqn{q_\tau} being the \eqn{\tau}-quantile of the predictive
distribution \eqn{F}, and \eqn{\mathbf{1}(\cdot)} the indicator function.

The weighted interval score for a single prediction interval can be obtained
as the average of the quantile scores for the lower and upper quantile of
that prediction interval:
\deqn{
  \text{WIS}_\alpha(F, y) = \frac{\text{QS}_{\alpha/2}(F, y)
  + \text{QS}_{1 - \alpha/2}(F, y)}{2}.
}
See the SI of Bracher et al. (2021) for more details.

\code{quantile_score()} returns the average quantile score across the quantile
levels provided. For a set of quantile levels that form pairwise central
prediction intervals, the quantile score is equivalent to the interval score.
}
\section{Input format}{
\if{html}{
  \out{<div style="text-align: left">}
  \figure{metrics-quantile.png}{options: style="width:750px;max-width:100\%;"}
  \out{</div><p>}
  Overview of required input format for quantile-based forecasts
}
\if{latex}{
  \figure{metrics-quantile.png}
}
}

\examples{
observed <- rnorm(10, mean = 1:10)
alpha <- 0.5

lower <- qnorm(alpha / 2, observed)
upper <- qnorm((1 - alpha / 2), observed)

qs_lower <- quantile_score(observed,
  predicted = matrix(lower),
  quantile_level = alpha / 2
)
qs_upper <- quantile_score(observed,
  predicted = matrix(upper),
  quantile_level = 1 - alpha / 2
)
interval_score <- (qs_lower + qs_upper) / 2
interval_score2 <- quantile_score(
  observed,
  predicted = cbind(lower, upper),
  quantile_level = c(alpha / 2, 1 - alpha / 2)
)

# this is the same as the following
wis(
  observed,
  predicted = cbind(lower, upper),
  quantile_level = c(alpha / 2, 1 - alpha / 2)
)
}
\references{
Strictly Proper Scoring Rules, Prediction,and Estimation,
Tilmann Gneiting and Adrian E. Raftery, 2007, Journal of the American
Statistical Association, Volume 102, 2007 - Issue 477

Evaluating epidemic forecasts in an interval format,
Johannes Bracher, Evan L. Ray, Tilmann Gneiting and Nicholas G. Reich, 2021,
\url{https://journals.plos.org/ploscompbiol/article?id=10.1371/journal.pcbi.1008618}
}
\keyword{metric}
