#' @title Evaluate forecasts
#'
#' @description The function \code{eval_forecasts} is an easy to use wrapper
#' of the lower level functions in the \code{scoringutils} package.
#' It can be used to score probabilistic or quantile forecasts of
#' continuous, integer-valued or binary variables.
#'
#' @details the following metrics are used where appropriate:
#' \itemize{
#'   \item {Interval Score} for quantile forecasts. Smaller is better. See
#'   \code{\link{interval_score}} for more information. By default, the
#'   weighted interval score is used.
#'   \item {Brier Score} for a probability forecast of a binary outcome.
#'   Smaller is better. See \code{\link{brier_score}} for more information.
#'   \item {aem} Absolute error of the median prediction
#'   \item {Bias} 0 is good, 1 and -1 are bad.
#'   See \code{\link{bias}} for more information.
#'   \item {Sharpness} Smaller is better. See \code{\link{sharpness}} for more
#'   information.
#'   \item {Calibration} represented through the p-value of the
#'   Anderson-Darling test for the uniformity of the Probability Integral
#'   Transformation (PIT). For integer valued forecasts, this p-value also
#'   has a standard deviation. Larger is better.
#'   See \code{\link{pit}} for more information.
#'   \item {DSS} Dawid-Sebastiani-Score. Smaller is better.
#'   See \code{\link{dss}} for more information.
#'   \item {CRPS} Continuous Ranked Probability Score. Smaller is better.
#'   See \code{\link{crps}} for more information.
#'   \item {Log Score} Smaller is better. Only for continuous forecasts.
#'   See \code{\link{logs}} for more information.
#' }
#'
#' @param data A data.frame or data.table with the predictions and observations.
#' Note: it is easiest to have a look at the example files provided in the
#' package and in the examples below.
#' The following columns need to be present:
#' \itemize{
#'   \item \code{true_value} - the true observed values
#'   \item \code{prediction} - predictions or predictive samples for one
#'   true value. (You only don't need to provide a prediction column if
#'   you want to score quantile forecasts in a wide range format.)}
#' For integer and continuous forecasts a \code{sample} column is needed:
#' \itemize{
#'   \item \code{sample} - an index to identify the predictive samples in the
#'   prediction column generated by one model for one true value. Only
#'   necessary for continuous and integer forecasts, not for
#'   binary predictions.}
#' For quantile forecasts the data can be provided in variety of formats. You
#' can either use a range-based format or a quantile-based format. (You can
#' convert between formats using \code{\link{quantile_to_range}},
#' \code{\link{range_to_quantile}},
#' \code{\link{sample_to_range}},
#' \code{\link{sample_to_quantile}})
#' For a quantile-format forecast you should provide:
#' \itemize{
#'   \item {prediction} - prediction to the corresponding quantile
#'   \item {qunaitle} - quantile to which the prediction corresponds}
#' For a range format (long) forecast you need
#' \itemize{
#'   \item \code{prediction} the quantile forecasts
#'   \item \code{boundary} values should be either "lower" or "upper", depending
#'   on whether the prediction is for the lower or upper bound of a given range
#'   \item {range} the range for which a forecast was made. For a 50\% interval
#'   the range should be 50. The forecast for the 25\% quantile should have
#'   the value in the \code{prediction} column, the value of \code{range}
#'   should be 50 and the value of \code{boundary} should be "lower".
#'   If you want to score the median (i.e. \code{range = 0}), you still
#'   need to include a lower and an upper estimate, so the median has to
#'   appear twice.}
#' Alternatively you can also provide the format in a wide range format.
#' This format needs
#' \itemize{
#'   \item pairs of columns called something like 'upper_90' and 'lower_90',
#'   or 'upper_50' and 'lower_50', where the number denotes the interval range.
#'   For the median, you need to proivde columns called 'upper_0' and 'lower_0'}
#' @param by character vector of columns to group scoring by. This should be the
#' lowest level of grouping possible, i.e. the unit of the individual
#' observation. This is important as many functions work on individual
#' observations. If you want a different level of aggregation, you should use
#' \code{summarise_by} to aggregate the individual scores.
#' Also not that the pit will be computed using \code{summarise_by}
#' instead of \code{by}
#' @param summarise_by character vector of columns to group the summary by. By
#' default, this is equal to `by` and no summary takes place.
#' But sometimes you may want to to summarise
#' over categories different from the scoring.
#' \code{summarise_by} is also the grouping level used to compute
#' (and possibly plot) the probability integral transform(pit).
# #' @param metrics the metrics you want to have in the output
#' @param quantiles numeric vector of quantiles to be returned when summarising.
#' Instead of just returning a mean, quantiles will be returned for the
#' groups specified through `summarise_by`. By default, no quantiles are
#' returned.
#' @param sd if TRUE (the default is FALSE) the standard deviation of all
#' metrics will be returned when summarising.
#' @param pit_plots if TRUE (not the default), pit plots will be returned. For
#' details see \code{\link{pit}}.
#' @param pit_arguments pass down additional arguments to the \code{\link{pit}}
#' function.
#' @param interval_score_arguments pass down additional arguments to the
#' \code{\link{interval_score}} function, e.g. \code{weigh = FAlSE}.
#' @param summarised Summarise arguments (i.e. take the mean per group
#' specified in group_by. Default is TRUE.
#' @param verbose print out additional helpful messages (default is TRUE)
#'
#' @return A data.table with appropriate scores. For binary predictions,
#' the Brier Score will be returned, for quantile predictions the interval
#' score, as well as adapted metrics for calibration, sharpness and bias.
#' For integer forecasts, Sharpness, Bias, DSS, CRPS, LogS, and
#' pit_p_val (as an indicator of calibration) are returned. For integer
#' forecasts, pit_sd is returned (to account for the randomised PIT),
#' but no Log Score is returned (the internal estimation relies on a
#' kernel density estimate which is difficult for integer-valued forecasts).
#' If \code{summarise_by} is specified differently from \code{by},
#' the average score per summary unit is returned.
#' If specified, quantiles and standard deviation of scores can also be returned
#' when summarising.
#'
#' @importFrom data.table ':=' setDT %like% rbindlist
#' @importFrom stats quantile
#'
#' @examples
#' ## Probability Forecast for Binary Target
#' binary_example <- data.table::setDT(scoringutils::binary_example_data)
#' eval <- scoringutils::eval_forecasts(binary_example,
#'                                      by = c("id", "model", "horizon"),
#'                                      summarise_by = c("model"),
#'                                      quantiles = c(0.5), sd = TRUE)
#' eval <- scoringutils::eval_forecasts(binary_example,
#'                                      by = c("id", "model", "horizon"))
#'
#' ## Quantile Forecasts
#' # wide format
#' quantile_example <- data.table::setDT(scoringutils::quantile_example_data_wide)
#' eval <- scoringutils::eval_forecasts(quantile_example,
#'                                      by = c("model", "horizon", "id"),
#'                                      summarise_by = "model",
#'                                      quantiles = c(0.05, 0.95),
#'                                      sd = TRUE)
#' eval <- scoringutils::eval_forecasts(quantile_example,
#'                                      by = c("model", "horizon", "id"))
#'
#' #long format
#'
#' eval <- scoringutils::eval_forecasts(scoringutils::quantile_example_data_long,
#'                                      by = c("model", "horizon", "id"),
#'                                      summarise_by = c("model", "range"))
#'
#' ## Integer Forecasts
#' integer_example <- data.table::setDT(scoringutils::integer_example_data)
#' eval <- scoringutils::eval_forecasts(integer_example,
#'                                      by = c("model", "id", "horizon"),
#'                                      summarise_by = c("model"),
#'                                      quantiles = c(0.1, 0.9),
#'                                      sd = TRUE,
#'                                      pit_plots = TRUE,
#'                                      pit_arguments = list(n_replicates = 30,
#'                                                           plot = TRUE))
#' eval <- scoringutils::eval_forecasts(integer_example)
#'
#' ## Continuous Forecasts
#' continuous_example <- data.table::setDT(scoringutils::continuous_example_data)
#' eval <- scoringutils::eval_forecasts(continuous_example,
#'                                      by = c("model", "id", "horizon"))
#' eval <- scoringutils::eval_forecasts(continuous_example,
#'                                      quantiles = c(0.5, 0.9),
#'                                      sd = TRUE,
#'                                      summarise_by = c("model"))
#'
#' @author Nikos Bosse \email{nikosbosse@gmail.com}
#' @references Funk S, Camacho A, Kucharski AJ, Lowe R, Eggo RM, Edmunds WJ
#' (2019) Assessing the performance of real-time epidemic forecasts: A
#' case study of Ebola in the Western Area region of Sierra Leone, 2014-15.
#' PLoS Comput Biol 15(2): e1006785.
#' \url{https://doi.org/10.1371/journal.pcbi.1006785}
#' @export

eval_forecasts <- function(data,
                           by = NULL,
                           summarise_by = by,
                           # metrics = NULL,
                           quantiles = c(),
                           sd = FALSE,
                           pit_plots = FALSE,
                           pit_arguments = list(plot = FALSE),
                           interval_score_arguments = list(weigh = TRUE),
                           summarised = TRUE,
                           verbose = TRUE) {

  # preparations ---------------------------------------------------------------
  # do a copy to avoid that the input may be altered in any way.
  data <- data.table::as.data.table(data)


  # error handling
  if (any(is.na(data$true_value))) {
    if(verbose) {
      warning("There are NA values in the true values provided")
    }
  }


  if (is.null(by)) {
    protected_columns <- c("prediction", "true_value", "sample", "quantile",
                           "range", "boundary")
    by <- setdiff(colnames(data), protected_columns)

    if (is.null(summarise_by)) {
      summarise_by <- by
    }
  }

  # helper function to add quantiles to summarised predictions
  add_quantiles <- function(dt, varnames, quantiles, by) {
    for (varname in varnames) {
      dt[, paste0(varname, "_", quantiles) := as.list(quantile(get(varname),
                                                               probs = quantiles,
                                                               na.rm = TRUE)),
         by = c(by)]
    }
    return(dt)
  }

  add_sd <- function(dt, varnames, by) {
    for (varname in varnames) {
      dt[, paste0(varname, "_sd") := sd(get(varname), na.rm = TRUE), by = by]
    }
    return(dt)
  }

  # if data is in quantile format, convert it to range format
  if ("quantile" %in% names(data) & !("range" %in% names(data))) {
    quantile_data <- data
    data <- scoringutils::quantile_to_range(data, keep_quantile_col = FALSE)
  }


  # check if predictions are integer, continuous, etc. -------------------------
  if (any(grepl("lower", names(data))) | "boundary" %in% names(data)) {
    prediction_type <- "quantile"
  } else if (all.equal(data$prediction, as.integer(data$prediction)) == TRUE) {
    prediction_type <- "integer"
  } else {
    prediction_type <- "continuous"
  }

  if (all.equal(data$true_value, as.integer(data$true_value)) == TRUE) {
    if (all(data$true_value %in% c(0,1))) {
      target_type = "binary"
    } else {
      target_type = "integer"
    }
  } else {
    target_type = "continuous"
  }

  # # only compute the metrics desired by the user -------------------------------
  # all_metrics <- list_of_avail_metrics()
  # if (is.null(metrics)) {
  #   metrics  <- all_metrics
  # } else {
  #   if (!all(metrics %in% all_metrics)) {
  #     msg <- paste(setdiff(metrics, all_metrics),
  #                  "is not an available metric and will not be computed")
  #   }
  # }



  # Score binary predictions ---------------------------------------------------
  if (target_type == "binary") {

    res <- data[, "brier_score" := scoringutils::brier_score(true_value, prediction),
         by = by]

    if (summarised) {
      # add quantiles
      if (!is.null(quantiles)) {
        res <- add_quantiles(res, "brier_score", quantiles, by = summarise_by)
      }

      # add standard deviation
      if (sd) {
        res[, "brier_score_sd" := sd(brier_score, na.rm = TRUE), by = c(summarise_by)]
      }

      # summarise by taking the mean over all relevant columns
      res <- data[, lapply(.SD, mean, na.rm = TRUE),
                 .SDcols = colnames(res) %like% "brier",
                 by = summarise_by]

    }
    return(res)
  }

  # Score quantile predictions -------------------------------------------------
  if (prediction_type == "quantile") {

    # check if long or wide format
    if ("boundary" %in% names(data)) {
      wide = FALSE
    } else {
      wide = TRUE
    }

    if (wide) {
      # convert into long format
      colnames <- colnames(data)
      ranges <- colnames[grepl("lower", colnames) | grepl("upper", colnames)]

      data <- data.table::melt(data,
                               measure.vars = ranges,
                               variable.name = "range",
                               value.name = "prediction")
      data[, boundary := gsub("_.*", "", range)]
      data[, range := as.numeric(gsub("^.*?_","", range))]
    }

    # save quantile version of the forecast for later
    quantile_data <- scoringutils::range_to_quantile(data,
                                                     keep_range_col = TRUE)

    data <- data.table::dcast(data, ... ~ boundary,
                              value.var = "prediction")

    # this is a bit weird --> find a more elegant solution. Idea is to add the
    # separate_results = TRUE argument to the function
    if (!("separate_results" %in% names(interval_score_arguments))) {
      interval_score_arguments <- c(interval_score_arguments,
                                    list(separate_results = TRUE))
    }

    res <- data[, c("interval_score",
                    "sharpness",
                    "underprediction",
                    "overprediction") := do.call(scoringutils::interval_score,
                                              c(list(true_value,
                                                     lower,
                                                     upper,
                                                     range),
                                                interval_score_arguments))]


    # compute coverage for every single observation
    res[, coverage := ifelse(true_value <= upper & true_value >= lower, 1, 0)]
    res[, coverage_deviation := coverage - range/100]


    # compute bias
    res[, bias := quantile_bias(range = range, lower = lower, upper = upper,
                                true_value = unique(true_value)),
        by = by]


    # compute absolute error based on quantile version
    abs_err <- function(true_value, prediction, quantile) {
      if (0.5 %in% quantile) {
        ae <- abs(unique(true_value - prediction[quantile == 0.5]))
      } else {
        ae <- NA
      }
      return(ae)
    }

    quantile_data[, aem := abs_err(true_value,
                                   prediction,
                                   quantile),
                  by = by]
    # tmp <- quantile_data[quantile == 0.5, .(ae = abs(true_value - prediction))]
    # quantile_data <- merge(quantile_data, tmp)


    # compute quantile coverage based on quantile version
    quantile_data[, quantile_coverage := (true_value <= prediction)]

    # delete unnecessary columns before merging back
    keep_cols <- unique(c(by, "quantile", "aem", "quantile_coverage",
                          "boundary", "range"))
    delete_cols <- names(quantile_data)[!(names(quantile_data) %in% keep_cols)]
    quantile_data[, eval(delete_cols) := NULL]

    #duplicate median column before merging
    median <- quantile_data[quantile == 0.5, ][, boundary := "upper"]
    quantile_data <- data.table::rbindlist(list(quantile_data, median))

    # merge back with other metrics
    merge_cols <- setdiff(keep_cols, c("aem", "quantile_coverage", "quantile",
                                       "boundary"))
    res <- merge(res, quantile_data, by = merge_cols)

    if (summarised) {

      if (!is.null(quantiles)) {
        # add quantiles for the scores
        res <- add_quantiles(res,
                             c("interval_score", "coverage",
                               "overprediction", "underprediction",
                               "coverage_deviation", "bias", "sharpness", "aem"),
                             quantiles,
                             by = c(summarise_by))
      }

      # add standard deviation
      if (sd) {
        res <- add_sd(res,
                      varnames = c("interval_score", "bias", "coverage",
                                   "overprediction", "underprediction",
                                   "coverage_deviation", "sharpness", "aem"),
                      by = c(summarise_by))
      }

      # summarise by taking the mean and omitting unnecessary columns
      res <- res[, lapply(.SD, mean, na.rm = TRUE),
                 by = c(summarise_by),
                 .SDcols = colnames(res) %like%
                   "coverage|bias|sharpness|coverage_deviation|interval_score|overprediction|underprediction|aem"]
    }
    return(res)
  }


  # Score integer or continuous predictions ------------------------------------
  # sharpness
  data[, sharpness := scoringutils::sharpness(t(prediction)), by = c(by)]

  # bias
  data[, bias := scoringutils::bias(unique(true_value),
                                     t(prediction)), by = c(by)]

  # DSS
  data[, dss := scoringutils::dss(unique(true_value),
                                    t(prediction)), by = c(by)]

  # CRPS
  data[, crps := scoringutils::crps(unique(true_value),
                                    t(prediction)), by = c(by)]

  # Log Score
  if (prediction_type == "continuous") {
    data[, log_score := scoringutils::logs(unique(true_value),
                                       t(prediction)), by = c(by)]
  }

  # coverage
  # reformat data.table to wide format
  dat <- data.table::dcast(data, ... ~ paste("sampl_", sample, sep = ""),
                           value.var = "prediction")



  # extract pit plots if specified
  if (pit_plots & summarised) {
    pit_arguments$plot <- TRUE

    split_dat <- split(dat, by = summarise_by)

    pits <- lapply(split_dat,
                   FUN = function(dat) {
                     samples <- as.matrix(dat[, grepl("sampl_", colnames(dat)),
                                              with = FALSE])

                     res <- do.call(pit, c(list(dat$true_value,
                                         samples),
                                    pit_arguments))

                     dat[, `:=` (pit_p_val = res$p_value,
                                 pit_sd = res$sd)]
                     plot <- res$hist_PIT
                     return(list(data = dat,
                                 hist_PIT = plot,
                                 samples = samples,
                                 true_value = dat$true_value))
                   })

    pit_histograms <- lapply(pits,
                             FUN = function(pit) {
                               return(pit$hist_PIT)
                             })

    pit_values <- lapply(pits,
                         FUN = function(pit) {
                           return(pit$data)
                         })

    overall_samples <- lapply(pits,
                              FUN = function(pit) {
                                return(pit$samples)
                              })

    overall_samples <- do.call(rbind, overall_samples)

    overall_true_value <- lapply(pits,
                                  FUN = function(pit) {
                                    return(pit$true_value)
                                  })

    overall_true_value <- do.call(c, overall_true_value)

    overall_pit_hist <- res <- do.call(pit, c(list(overall_true_value,
                                                   overall_samples),
                                              pit_arguments))

    pit_histograms[["overall_pit"]] <- overall_pit_hist$hist_PIT


    dat <- rbindlist(pit_values)
  } else {

    if (summarised) {

      if (identical(by, summarise_by)) {
        dat[, c("pit_p_val", "pit_sd") := NA]
        if (verbose) {
          message("In order to compute PIT values, 'summarise_by' must be different from 'by'")
        }
      } else {
        # compute pit p-values in a quicker way
        dat[, c("pit_p_val", "pit_sd") := do.call(pit, c(list(true_value,
                                                              as.matrix(.SD)),
                                                         pit_arguments)),
            .SDcols = names(dat)[grepl("sampl_", names(dat))], by = summarise_by]
      }
    }
  }

  # remove variables not necessary for merging
  dat[, names(dat)[grepl("sampl_", names(dat))] := NULL]
  dat[, c("sharpness", "bias", "dss", "crps") := NULL]

  # merge with previous data
  merge_cols = colnames(dat)[!colnames(dat) %in% c("pit_p_val", "pit_sd")]
  res <- merge(data, dat, by = merge_cols)

  # make scores unique to avoid redundancy.
  res <- res[, lapply(.SD, unique),
             .SDcols = colnames(res) %like% "pit_|bias|sharpness|dss|crps|log_score",
             by = c(by)]


  if (summarised) {
    # add quantiles
    if (!is.null(quantiles)) {

      if (prediction_type == "continuous") {
        quantile_vars <- c("crps", "dss", "log_score", "pit_p_val", "bias", "sharpness")
      } else {
        quantile_vars <- c("crps", "dss", "pit_p_val", "bias", "sharpness")
      }
      res <- add_quantiles(res, quantile_vars, quantiles, by = c(summarise_by))
    }

    if (sd) {
      # add standard deviations
      if (prediction_type == "continuous") {
        sd_vars <- c("crps", "dss", "log_score", "bias", "sharpness")
      } else {
        sd_vars <- c("crps", "dss", "bias", "sharpness")
      }

      res <- add_sd(res,
                    varnames = sd_vars,
                    by = c(summarise_by))
    }

    res <- res[, lapply(.SD, mean, na.rm = TRUE),
               .SDcols = colnames(res) %like% "pit_|bias|sharpness|dss|crps|log_score",
               by = summarise_by]
  }

  # if pit_plots is TRUE, add the plots as an output
  if (pit_plots) {
    res <- list(scores = res,
                pit_plots = pit_histograms)
  }

  # # return only those metrics the user wants to see
  # filter_cols <- unique(c(by, summarise_by, metrics))
  #
  # return (res[, filter_cols])
  return(res)
}







