#' @name exportInstruments
#'
#' @title Exports the REDCap Instruments
#' @description Returns a data frame of instruments, names, etc.
#'   
#' @section REDCap Version:
#' 6.5.0 + 
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @section REDCap API Documentation:
#' This function allows you to export a list of the data collection instruments 
#' for a project. This includes their unique instrument name as seen in the second 
#' column of the Data Dictionary, as well as each instrument's corresponding 
#' instrument label, which is seen on a project's left-hand menu when entering data. 
#' The instruments will be ordered according to their order in the project.
#' 
#' @section REDCap Version:
#' 5.8.2+
#' 
#' @section Known REDCap Limitations: 
#' None
#' 
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export

exportInstruments <- function(rcon, ...){
  UseMethod("exportInstruments")
}

#' @rdname exportInstruments
#' @export

exportInstruments.redcapApiConnection <- function(rcon, 
                                                  ...,
                                                  error_handling = getOption("redcap_error_handling"), 
                                                  config         = list(), 
                                                  api_param      = list()){
   ##################################################################
  # Argument Validation 
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Make Body List
  
  body <- list(token = rcon$token, 
               content = 'instrument',
               format = 'csv')
  
  body <- body[lengths(body) > 0]
  
   ##################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) return(redcap_error(response, error_handling))
  
  utils::read.csv(text = as.character(response), 
                  stringsAsFactors = FALSE, 
                  na.strings = "")
}
