% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimizer.R
\name{Optimizer}
\alias{Optimizer}
\title{Specify numerical optimizer as R6 object}
\description{
A \code{Optimizer} R6 object defines a numerical optimizer based on an
optimization function implemented in R.

The main advantage of working with an \code{Optimizer} object instead of
using the optimization function directly lies in the standardized inputs and
outputs.

Any R function that fulfills the following four constraints can be defined as
an \code{Optimizer} object:
\enumerate{
\item It must have an input for a \code{function}, the objective function to be
optimized.
\item It must have an input for a \code{numeric} vector, the initial values from
where the optimizer starts.
\item It must have a \code{...} argument for additional parameters passed on to
the objective function.
\item The output must be a named \code{list}, including the optimal function
value and the optimal parameter vector.
}
}
\examples{
### Task: compare minimization with 'stats::nlm' and 'pracma::nelder_mead'

# 1. define objective function and initial values
objective <- TestFunctions::TF_ackley
initial <- c(3, 3)

# 2. get overview of optimizers in dictionary
optimizer_dictionary$keys

# 3. define 'nlm' optimizer
nlm <- Optimizer$new(which = "stats::nlm")

# 4. define the 'pracma::nelder_mead' optimizer (not contained in the dictionary)
nelder_mead <- Optimizer$new(which = "custom")
nelder_mead$definition(
  algorithm = pracma::nelder_mead, # the optimization function
  arg_objective = "fn",            # the argument name for the objective function
  arg_initial = "x0",              # the argument name for the initial values
  out_value = "fmin",              # the element for the optimal function value in the output
  out_parameter = "xmin",          # the element for the optimal parameters in the output
  direction = "min"                # the optimizer minimizes
)

# 5. compare the minimization results
nlm$minimize(objective, initial)
nelder_mead$minimize(objective, initial)


## ------------------------------------------------
## Method `Optimizer$minimize`
## ------------------------------------------------

Optimizer$new("stats::nlm")$
  minimize(objective = function(x) x^4 + 3*x - 5, initial = 2)

## ------------------------------------------------
## Method `Optimizer$maximize`
## ------------------------------------------------

Optimizer$new("stats::nlm")$
  maximize(objective = function(x) -x^4 + 3*x - 5, initial = 2)
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{label}}{A \code{character}, the label for the optimizer.}

\item{\code{algorithm}}{A \code{function}, the optimization algorithm.}

\item{\code{arg_objective}}{A \code{character}, the argument name for the objective function in
\code{algorithm}.}

\item{\code{arg_initial}}{A \code{character}, the argument name for the initial values in
\code{algorithm}.}

\item{\code{out_value}}{A \code{character}, the element name for the optimal function value in
the output \code{list} of \code{algorithm}.}

\item{\code{out_parameter}}{A \code{character}, the element name for the optimal parameters in the
output \code{list} of \code{algorithm}.}

\item{\code{direction}}{Either \code{"min"} (if the optimizer minimizes) or \code{"max"}
(if the optimizer maximizes).}

\item{\code{arguments}}{A named \code{list} of custom arguments for \code{algorithm}. Defaults
are used for arguments that are not specified.}

\item{\code{seconds}}{A \code{numeric}, a time limit in seconds. Optimization is interrupted
prematurely if \code{seconds} is exceeded.

No time limit if \code{seconds = Inf} (the default).

Note the limitations documented in \code{\link[base]{setTimeLimit}}.}

\item{\code{hide_warnings}}{Either \code{TRUE} to hide warnings during optimization, or \code{FALSE}
(default) else.}

\item{\code{output_ignore}}{A \code{character} \code{vector} of elements to ignore in the
optimization output.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Optimizer-new}{\code{Optimizer$new()}}
\item \href{#method-Optimizer-definition}{\code{Optimizer$definition()}}
\item \href{#method-Optimizer-set_arguments}{\code{Optimizer$set_arguments()}}
\item \href{#method-Optimizer-validate}{\code{Optimizer$validate()}}
\item \href{#method-Optimizer-minimize}{\code{Optimizer$minimize()}}
\item \href{#method-Optimizer-maximize}{\code{Optimizer$maximize()}}
\item \href{#method-Optimizer-print}{\code{Optimizer$print()}}
\item \href{#method-Optimizer-clone}{\code{Optimizer$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-new"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-new}{}}}
\subsection{Method \code{new()}}{
Initializes a new \code{Optimizer} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$new(which, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{A \code{character}, either one of \code{optimizer_dictionary$keys} or
\code{"custom"} (in which case \code{$definition()} must be used to
define the optimizer details).}

\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{Optimizer} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-definition"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-definition}{}}}
\subsection{Method \code{definition()}}{
Defines an optimizer.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$definition(
  algorithm,
  arg_objective,
  arg_initial,
  out_value,
  out_parameter,
  direction
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{algorithm}}{A \code{function}, the optimization algorithm.}

\item{\code{arg_objective}}{A \code{character}, the argument name for the objective function in
\code{algorithm}.}

\item{\code{arg_initial}}{A \code{character}, the argument name for the initial values in
\code{algorithm}.}

\item{\code{out_value}}{A \code{character}, the element name for the optimal function value in
the output \code{list} of \code{algorithm}.}

\item{\code{out_parameter}}{A \code{character}, the element name for the optimal parameters in the
output \code{list} of \code{algorithm}.}

\item{\code{direction}}{Either \code{"min"} (if the optimizer minimizes) or \code{"max"}
(if the optimizer maximizes).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly the \code{Optimizer} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-set_arguments"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-set_arguments}{}}}
\subsection{Method \code{set_arguments()}}{
Sets optimizer arguments.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$set_arguments(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The \code{Optimizer} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-validate"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-validate}{}}}
\subsection{Method \code{validate()}}{
Validates the \code{Optimizer} object. A time limit in seconds for
the optimization can be set via the \code{$seconds} field.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$validate(
  objective = optimizeR::test_objective,
  initial = round(stats::rnorm(2)),
  ...,
  direction = "min"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{objective}}{A \code{function} to be optimized that
\enumerate{
\item has at least one argument that receives a \code{numeric} \code{vector}
\item and returns a single \code{numeric} value.
}

Alternatively, it can also be a \code{\link{Objective}} object for more
flexibility.}

\item{\code{initial}}{A \code{numeric} vector with starting parameter values for the optimization.}

\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}

\item{\code{direction}}{Either \code{"min"} for minimization or \code{"max"} for maximization.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The \code{Optimizer} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-minimize"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-minimize}{}}}
\subsection{Method \code{minimize()}}{
Performing minimization.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$minimize(objective, initial, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{objective}}{A \code{function} to be optimized that
\enumerate{
\item has at least one argument that receives a \code{numeric} \code{vector}
\item and returns a single \code{numeric} value.
}

Alternatively, it can also be a \code{\link{Objective}} object for more
flexibility.}

\item{\code{initial}}{A \code{numeric} vector with starting parameter values for the optimization.}

\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A named \code{list}, containing at least these five elements:
\describe{
\item{\code{value}}{A \code{numeric}, the minimum function value.}
\item{\code{parameter}}{A \code{numeric} vector, the parameter vector
where the minimum is obtained.}
\item{\code{seconds}}{A \code{numeric}, the optimization time in seconds.}
\item{\code{initial}}{A \code{numeric}, the initial parameter values.}
\item{\code{error}}{Either \code{TRUE} if an error occurred, or \code{FALSE}, else.}
}
Appended are additional output elements of the optimizer.

If an error occurred, then the error message is also appended as element
\code{error_message}.

If the time limit was exceeded, this also counts as an error. In addition,
the flag \code{time_out = TRUE} is appended.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{Optimizer$new("stats::nlm")$
  minimize(objective = function(x) x^4 + 3*x - 5, initial = 2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-maximize"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-maximize}{}}}
\subsection{Method \code{maximize()}}{
Performing maximization.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$maximize(objective, initial, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{objective}}{A \code{function} to be optimized that
\enumerate{
\item has at least one argument that receives a \code{numeric} \code{vector}
\item and returns a single \code{numeric} value.
}

Alternatively, it can also be a \code{\link{Objective}} object for more
flexibility.}

\item{\code{initial}}{A \code{numeric} vector with starting parameter values for the optimization.}

\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A named \code{list}, containing at least these five elements:
\describe{
\item{\code{value}}{A \code{numeric}, the maximum function value.}
\item{\code{parameter}}{A \code{numeric} vector, the parameter vector
where the maximum is obtained.}
\item{\code{seconds}}{A \code{numeric}, the optimization time in seconds.}
\item{\code{initial}}{A \code{numeric}, the initial parameter values.}
\item{\code{error}}{Either \code{TRUE} if an error occurred, or \code{FALSE}, else.}
}
Appended are additional output elements of the optimizer.

If an error occurred, then the error message is also appended as element
\code{error_message}.

If the time limit was exceeded, this also counts as an error. In addition,
the flag \code{time_out = TRUE} is appended.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{Optimizer$new("stats::nlm")$
  maximize(objective = function(x) -x^4 + 3*x - 5, initial = 2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-print"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-print}{}}}
\subsection{Method \code{print()}}{
Prints the optimizer label.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Optionally additional arguments to be passed to the optimizer algorithm.
Without specifications, default values are used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly the \code{Optimizer} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Optimizer-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Optimizer-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Optimizer$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
