#' @title 
#' Step 4: Replaying the experiment with optimal parameters
#'
#' @description
#' After completing Step 3 using `fit_p` to obtain the optimal parameters
#'  for each subject and saving the resulting CSV locally, this function
#'  allows you to load that result dataset. It then applies these optimal
#'  parameters back into the reinforcement learning model, effectively
#'  simulating how the "robot" (the model) would make its choices.
#'
#' Based on this generated dataset, you can then analyze the robot's data
#'  in the same manner as you would analyze human behavioral data. If a
#'  particular model's fitted data can successfully reproduce the
#'  experimental effects observed in human subjects, it strongly suggests
#'  that this model is a good and valid representation of the process.
#'
#' @param data [data.frame] 
#' This data should include the following mandatory columns: 
#'  \itemize{
#'    \item "sub"
#'    \item "time_line" (e.g., "Block", "Trial")
#'    \item "L_choice"
#'    \item "R_choice"
#'    \item "L_reward"
#'    \item "R_reward"
#'    \item "sub_choose"
#'  }
#'   
#' @param id [vector]
#' Specifies the subject ID(s) for whom optimal parameters are to be fitted.
#'  If you intend to fit all subjects within your dataset, it is strongly
#'  recommended to use \code{id = unique(data$Subject)}. This approach accounts
#'  for cases where subject IDs in the dataset may not be simple numeric
#'  sequences (e.g., "1", "2", "3", "4") or may contain string
#'  entries (e.g., "1", "2", "3", "004"). Using \code{id = 1:4} could lead to 
#'  errors if IDs are not sequentially numbered integers.
#'  
#'  \code{default: id = NULL}
#'  
#' @param result [data.frame]  
#' Output data generated by the `fit_p()` function. Each row represents 
#'  model fit results for a subject.
#'
#' @param model [function]  
#' A model function to be applied in evaluating the experimental effect.
#'
#' @param model_name [character]  
#' A character string specifying the name of the model to extract from 
#'  the result.
#'
#' @param param_prefix [character]  
#' A prefix string used to identify parameter columns in the  `result` data 
#' 
#' \code{default: param_prefix = "param_"}
#' 
#' @param n_trials [integer]
#' Represents the total number of trials a single subject experienced
#'  in the experiment. If this parameter is kept at its default value
#'  of `NULL`, the program will automatically detect how many trials
#'  a subject experienced from the provided data. This information
#'  is primarily used for calculating model fit statistics such as
#'  AIC (Akaike Information Criterion) and BIC (Bayesian Information
#'  Criterion).
#'  
#'  \code{default: n_trials = NULL}
#'
#' @return 
#' A list, where each element is a data.frame representing one subject's
#'  results. Each data.frame includes the value update history for each option,
#'  the learning rate (\code{eta}), utility function (\code{gamma}), and other 
#'  relevant information used in each update.
#'  
#' @examples
#' \dontrun{
#' list <- list()
#'
#' list[[1]] <- dplyr::bind_rows(
#'   binaryRL::rpl_e(
#'     data = binaryRL::Mason_2024_Exp2,
#'     result = read.csv("../OUTPUT/result_comparison.csv"),
#'     model = binaryRL::TD,
#'     model_name = "TD",
#'     param_prefix = "param_",
#'   )
#' )
#'
#' list[[2]] <- dplyr::bind_rows(
#'   binaryRL::rpl_e(
#'     data = binaryRL::Mason_2024_Exp2,
#'     result = read.csv("../OUTPUT/result_comparison.csv"),
#'     model = binaryRL::RSTD,
#'     model_name = "RSTD",
#'     param_prefix = "param_",
#'   )
#' )
#'
#' list[[3]] <- dplyr::bind_rows(
#'   binaryRL::rpl_e(
#'     data = binaryRL::Mason_2024_Exp2,
#'     result = read.csv("../OUTPUT/result_comparison.csv"),
#'     model = binaryRL::Utility,
#'     param_prefix = "param_",
#'     model_name = "Utility",
#'   )
#' )
#' }
#'  
rpl_e <- function(
  data, 
  id = NULL,
  result, 
  model,
  model_name,
  param_prefix,
  n_trials = NULL
) {
  # 事前准备. 探测信息
  info <- suppressWarnings(suppressMessages(detect_information(data = data)))
  
  # 自动获取被试id列的列名
  Subject <- info[["sub_col_name"]]
  
  # 如果没有指定重玩实验的是哪几个被试, 就会让所有被试重玩该实验.
  # <前提是有这个被试的最佳参数结果>
  if (is.null(id)) {id <- info[["all_ids"]]}
  
  # 如果没有告知实验有多少个试次, 就自动探测
  if (is.null(n_trials)) {n_trials <- info[["n_trials"]]}
  
  # 因为所有模型的结果都会在result中, 一次只导入result中某个模型的行
  fit_model <- "fit_model"
  result <- result[result[[fit_model]] == model_name, ]
  
  # 正式开始复现实验结果
  res <- list()
  
  for (i in 1:length(id)) {
    
    params <- stats::na.omit(unlist(result[i, grep(param_prefix, names(result))]))
    
    binaryRL.env <- new.env()
    binaryRL.env$data <- data
    binaryRL.env$id <- id[i]
    binaryRL.env$n_params <- length(params)
    binaryRL.env$n_trials <- n_trials
    binaryRL.env$mode <- "replay"
    
    
    obj_func <- model
    
    environment(obj_func) <- binaryRL.env
    res[[i]] <- obj_func(params = params)[[1]]
  }
  
  return(res)
}
